/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/internal/pch.h"

namespace ttv {
namespace chat {
class ChatRoomMessageHandler;
}
}  // namespace ttv

/**
 * Class that receives messages to parse out special commands such as `/ban` before being sent to
 * `IChatRoom::SendMessage`. Custom callback functions must inherit ICallbacks and passed in through the constructor
 * first. Messages being sent will be passed in through HandleMessage().
 */
class ttv::chat::ChatRoomMessageHandler {
 public:
  /**
   * CommandError - Represents which command had an error while sending.
   */
  enum class CommandError {
    Unknown,
    Ban,
    Unban,
    Timeout,
    Untimeout,
    Mod,
    Unmod,
    Ignore,
    Unignore,
    Color,
    Topic,
    Slow,
    Me
  };

  /**
   * Create your own callbacks class that inherits from this,
   * then pass in that class to the ChatMessageHandler constructor.
   */
  class ICallbacks {
   public:
    virtual ~ICallbacks() = default;

    /**
     * The message contains normal text or an unhandled command.
     * Callback typically should call ChatRoom::SendMessage().
     *
     * @param[in] message The message string being delivered through chat.
     * @return True if message was handled properly, else false.
     */
    virtual bool PassThrough(const std::string& message) = 0;

    /**
     * Called for valid `/ban` commands.
     * Typically should call ChatAPI::BanUser().
     *
     * @param[in] userName The name of the user to be banned.
     * @return True if the message was handled properly, else false.
     */
    virtual bool BanUser(const std::string& userName) = 0;

    /**
     * Called for valid `/unban` commands.
     * Typically should call ChatAPI::UnbanUser().
     *
     * @param[in] userName The name of the user being unbanned.
     * @return True if the message was handled properly, else false.
     */
    virtual bool UnbanUser(const std::string& userName) = 0;

    /**
     * Called for valid '/timeout' commands.
     * Typically should call ChatAPI::BanUser().
     *
     * @param[in] userName The name of the user to be timed out.
     * @param[in] duration How long the user is timed out for in seconds.
     * @return True if the message was handled properly, else false.
     */
    virtual bool TimeoutUser(const std::string& userName, uint32_t duration) = 0;

    /**
     * Called for valid '/untimeout' commands.
     * Typically should call ChatAPI::UnbanUser().
     *
     * @param[in] userName The name of the user being untimedout.
     * @return True if the message was handled properly, else false.
     */
    virtual bool UntimeoutUser(const std::string& userName) = 0;

    /**
     * Called for valid `/color` commands.
     * Typically should call ChatAPI::UpdateUserColor().
     *
     * @param[in] color The color being changed to.
     * @return True if the message was handled properly, else false.
     */
    virtual bool SetUserColor(const std::string& color) = 0;

    /**
     * Called for valid `/help` commands.
     * Typically should display information about the chat room commands.
     *
     * @return True if the message was handled properly, else false.
     */
    virtual bool Help() = 0;

    /**
     * Called for valid `/listrooms` commands.
     * Typically should call ChannelChatRoomManager::FetchChannelChatRoomsInfo().
     *
     * @return True if the message was handled properly, else false.
     */
    virtual bool ListRooms() = 0;

    /**
     * Called for valid `/mod` commands.
     * Typically should call ChatAPI::ModUser() after checking the sender of the message owns the current channel.
     *
     * @param[in] userName The name of the user being modded.
     * @return True if the message was handled properly, else false.
     */
    virtual bool ModUser(const std::string& userName) = 0;

    /**
     * Called for valid `/unmod` commands.
     * Typically should call ChatAPI::UnmodUser() after checking the sender of the message owns the current channel.
     *
     * @param[in] userName The name of the user being unmodded.
     * @return True if the message was handled properly, else false.
     */
    virtual bool UnmodUser(const std::string& userName) = 0;

    /**
     * Called for valid `/ignore` commands.
     * Typically should call ChatAPI::BlockUser().
     *
     * @param[in] userName The name of the user being blocked.
     * @return True if the message was handled properly, else false.
     */
    virtual bool BlockUser(const std::string& userName) = 0;

    /**
     * Called for valid `/unignore` commands.
     * Typically should call ChatAPI::UnblockUser().
     *
     * @param[in] userName The name of the user being unblocked.
     * @return True if the message was handled properly, else false.
     */
    virtual bool UnblockUser(const std::string& userName) = 0;

    /**
     * Called for valid `/mods` commands.
     * Typically should call ChatAPI::FetchChannelModerators().
     *
     * @return True if the message was handled properly, else false.
     */
    virtual bool ListModerators() = 0;

    /**
     * Called for valid `/topic` commands.
     * Typically should call ChatRoom::SetTopic().
     *
     * @param[in] topic The new topic of the room.
     * @return True if the message was handled properly, else false.
     */
    virtual bool SetTopic(const std::string& topic) = 0;

    /**
     * Called for valid '/slow' and '/slowoff' commands.
     * Typically should call ChatRoom::SetMode().
     *
     * @param[in] turnOn If true, we want to enable slow mode in the room. If false, we want to disable slow mode.
     * @param[in] durationSeconds The number of seconds a viewer has to wait between sending messages. Value is ignored
     * if we're turning slow mode off.
     * @return True if the message was handled properly, else false.
     */
    virtual bool SlowMode(bool turnOn, uint32_t durationSeconds) = 0;

    /**
     * Called for valid '/r9kbeta' and '/r9kbetaoff' commands.
     * Typically should call ChatRoom::SetMode().
     *
     * @param[in] turnOn If true, we want to enable r9k mode in the room. If false, we want to disable r9k mode.
     * @return True if the message was handled properly, else false.
     */
    virtual bool R9kMode(bool turnOn) = 0;

    /**
     * Called for valid '/emoteonly' and '/emoteonlyoff' commands.
     * Typically should call ChatRoom::SetMode().
     *
     * @param[in] turnOn If true, we want to enable emotes-only mode in the room. If false, we want to disable
     * emote-only mode.
     * @return True if the message was handled properly, else false.
     */
    virtual bool EmotesOnlyMode(bool turnOn) = 0;

    /**
     * Called for invalid commands.
     *
     * @param[in] command An enum representing which command failed.
     * @param[in] commandText The command used, AKA first word of the message
     * @return True if the error message was handled properly, else false.
     */
    virtual bool MalformedCommand(CommandError command, const std::string& commandText) = 0;
  };

  ChatRoomMessageHandler();

  /**
   * Takes in outgoing chat room messages and call the corresponding callback function based on the contents of the
   * message.
   *
   * @pre Callback functions must be set previously through SetCallbacks.
   * @param[in] message The message being sent to the chat room.
   * @return The result of the callback function.
   */
  bool HandleMessage(const std::string& message);

  /**
   * Sets the callbacks functions that are called when HandleMessage is called.
   *
   * @param[in] callbacks Pointer to the inherited ICallbacks class.
   */
  void SetCallbacks(const std::shared_ptr<ICallbacks>& callbacks) { mCallbacks = callbacks; }

 private:
  std::shared_ptr<ICallbacks> mCallbacks;
};
