/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chaterrortypes.h"
#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/optional.h"
#include "twitchsdk/core/types/coretypes.h"

#include <map>
#include <regex>

namespace ttv {
namespace chat {
using MessageId = uint32_t;

struct FeatureFlags {
  FeatureFlags();

  static FeatureFlags All();
  static FeatureFlags None();

  bool conversations : 1;  //!< Conversations support.
};

/**
 * TokenizationOptions - Controls the way chat messages are passed to the client.
 */
struct TokenizationOptions {
  TokenizationOptions();

  static TokenizationOptions All();
  static TokenizationOptions None();

  bool Any() const;

  bool emoticons : 1;  //!< Find emoticons when tokenizing.
  bool mentions : 1;   //!< Find @username when tokenizing.
  bool urls : 1;       //!< Find URLs when tokenizing.
  bool bits : 1;       //!< Find Bits(TM) tokens when tokenizing.
};

/**
 * ChatChannelState - The current state of the chat connection.
 */
enum class ChatChannelState { Disconnected, Connecting, Connected, Disconnecting };

/**
 * ChatMode - A mode a chat can be in.
 * For use internally by the SDK.
 */
enum class ChatMode {
  Slow,       //!< Slow mode.
  R9k,        //!< r9k mode.
  EmotesOnly  //!< Emotes-only mode.
};

/**
 * ChatModeInfo - Stores information about the modes that a chat is in
 */
struct ChatModeInfo {
  ChatModeInfo();

  uint32_t
    slowModeDurationSeconds;  //!< The number of seconds a user has to wait before sending another message. Value of 0 if not in slow mode.
  bool r9kMode;         //!< Whether the room is in r9k mode.
  bool emotesOnlyMode;  //!< Whether the room is in emotes-only mode.
};

struct UserMode {
  UserMode();

  bool operator==(const UserMode& rhs) const;
  bool operator!=(const UserMode& rhs) const;

  bool moderator : 1;        //!< A moderator.
  bool broadcaster : 1;      //!< The broadcaster.
  bool administrator : 1;    //!< An admin.
  bool staff : 1;            //!< A member of Twitch.
  bool system : 1;           //!< The Twitch system.
  bool globalModerator : 1;  //!< A global moderator.
  bool banned : 1;           //!< Banned from the channel.
  bool subscriber : 1;       //!< Subscribed to the channel.
  bool vip : 1;              //!< a VIP of the channel.
};

/**
 * ChatUserInfo - The user information relative to a channel's chat.
 */
struct ChatUserInfo {
  ChatUserInfo();

  bool operator==(const ChatUserInfo& rhs) const;
  bool operator!=(const ChatUserInfo& rhs) const;

  std::string userName;     //!< The user's login name.
  std::string displayName;  //!< The UTF-8 encoded displayed name.  Currently restricted to the ASCII subset.
  Color nameColorARGB;      //!< The current ARGB color of the user's name text.
  UserId userId;      //!< The ID of the user, if available. If the userId is not available, this will be set to 0.
  UserMode userMode;  //!< The mode which controls priviledges in a particular chat.
};

struct RestrictionReason {
  RestrictionReason();

  bool operator==(const RestrictionReason& rhs) const;
  bool operator!=(const RestrictionReason& rhs) const;

  bool anonymous : 1;        //!< Logged in anonymously so can't chat.
  bool subscribersOnly : 1;  //!< The channel is in subscribers only mode.
  bool
    slowMode : 1;  //!< The channel is in slow mode and messages are being paced.  Expect TTV_CHAT_RESTRICTION_REASON_NONE when the timeout is up.
  bool
    timeout : 1;  //!< The user has been temporarily banned for an amount of time.  Expect TTV_CHAT_RESTRICTION_REASON_NONE when the timeout is up.
  bool
    banned : 1;  //!< The user has been permanently banned.  Expect TTV_CHAT_RESTRICTION_REASON_NONE if a moderator ever unbans.
};

/**
 * ChatChannelInfo - Information about the state of a connected chat channel.
 */
struct ChatChannelInfo {
  std::string name;  //!< The UTF-8 encoded name of the channel.  Currently restricted to the ASCII subset.
  std::string
    broadcasterLanguage;  //!< The active broadcaster language code.  Empty if there is no language filter enabled.
  RestrictionReason localUserRestriction;  //!< The active restriction for the local user.
};

struct ChatChannelRestrictions {
  ChatChannelRestrictions();

  uint32_t
    followersDuration;  //!< If followersOnly is true, represents how long the user must have been following for before being able to chat.
  uint32_t
    slowModeDuration;  //!< 0 if slow mode is off, otherwise, represents how long the user must wait between sending messsages.
  Timestamp slowModeSetAt;  //!< Time at which slow mode was turned on.
  bool emoteOnly;           //!< Whether chat is in emote-only mode.
  bool verifiedOnly;        //!< Whether chat is in verified mode
  bool followersOnly;       //!< Whether chat is in followers-only mode.
  bool subscribersOnly;     //!< Whether chat is in subscribers-only mode.
  bool slowMode;            //!< Whether chat is in slow mode.
  bool r9k;                 //!< Whether chat is in r9k mode.
};

/**
 * AutoModFlags - stores the automod severity levels for part of a message. Levels range from 0-7, 7 being the most
 * severe, 0 being no violation.
 */
struct AutoModFlags {
  AutoModFlags();
  bool HasFlag();  //!< Returns true if any of the levels below are greater than 0, else returns false.

  uint32_t identityLevel;    //!< An attack on the user's identity.
  uint32_t sexualLevel;      //!< How sexual the message is in nature.
  uint32_t aggressiveLevel;  //!< How aggressive the message is in nature.
  uint32_t profanityLevel;   //!< How profane the message is in nature.
};

/**
 * MessageToken - Information about an message token.
 */
class MessageToken {
 public:
  enum class Type { Text, Emoticon, Mention, Url, Bits };

  virtual ~MessageToken() = default;

  MessageToken() = default;
  MessageToken(const MessageToken& src) = delete;
  MessageToken& operator=(const MessageToken& src) = delete;
  MessageToken(MessageToken&& src) = delete;
  MessageToken& operator=(MessageToken&& src) = delete;

  virtual std::unique_ptr<MessageToken> Clone() const = 0;
  virtual Type GetType() const = 0;
};

bool TokensEqual(const std::unique_ptr<MessageToken>& lhs, const std::unique_ptr<MessageToken>& rhs);

/**
 * TextToken - Information about a text token.
 */
struct TextToken : public Cloneable<MessageToken, TextToken> {
  TextToken() = default;
  TextToken(const std::string& text);
  TextToken(const std::string&, const AutoModFlags& flags);
  TextToken(const TextToken& src);
  TextToken& operator=(const TextToken& src);
  TextToken(TextToken&& src);
  TextToken& operator=(TextToken&& src);

  std::string text;
  AutoModFlags autoModFlags;

  virtual Type GetType() const override { return Type::Text; }
};

/**
 * EmoticonToken - Information about how to render an emoticon.  You can generate a URL for the emoticon image by
 * calling ChatAPI::GetEmoticonUrl().
 */
struct EmoticonToken : public Cloneable<MessageToken, EmoticonToken> {
  EmoticonToken() = default;
  EmoticonToken(const std::string& emoticonText, const std::string& emoticonId);
  EmoticonToken(const EmoticonToken& src);
  EmoticonToken& operator=(const EmoticonToken& src);
  EmoticonToken(EmoticonToken&& src);
  EmoticonToken& operator=(EmoticonToken&& src);

  std::string emoticonText;  //!< The raw text which was transformed into the image.
  std::string emoticonId;    //!< The emoticon id.

  virtual Type GetType() const override { return Type::Emoticon; }
};

/**
 * MentionToken - Information about a user mention typically done via \@user.
 */
struct MentionToken : public Cloneable<MessageToken, MentionToken> {
  MentionToken(const std::string& userName, const std::string& text, bool isLocalUser);
  MentionToken(const MentionToken& src);
  MentionToken& operator=(const MentionToken& src);
  MentionToken(MentionToken&& src);
  MentionToken& operator=(MentionToken&& src);

  std::string userName;  //!< The mentioned user.
  std::string text;      //!< The original mention text used in the message
  bool isLocalUser;      //!< True if mention is for the current local user

  virtual Type GetType() const override { return Type::Mention; }
};

/**
 * UrlToken - Information about a hyperlink.
 */
struct UrlToken : public Cloneable<MessageToken, UrlToken> {
  UrlToken();
  UrlToken(const std::string& url, bool hidden);
  UrlToken(const UrlToken& src);
  UrlToken& operator=(const UrlToken& src);
  UrlToken(UrlToken&& src);
  UrlToken& operator=(UrlToken&& src);

  std::string url;  //!< The url.
  bool hidden;      //!< Whether or not the URL token should be hidden by the client.

  virtual Type GetType() const override { return Type::Url; }
};

/**
 * BitsToken - Information about a bits transaction.
 */
struct BitsToken : public Cloneable<MessageToken, BitsToken> {
  BitsToken();
  BitsToken(const std::string& prefix, uint32_t numBits);
  BitsToken(const BitsToken& src);
  BitsToken& operator=(const BitsToken& src);
  BitsToken(BitsToken&& src);
  BitsToken& operator=(BitsToken&& src);

  std::string prefix;  //!< The prefix of the bits token (i.e. "cheer")
  uint32_t numBits;    //!< The number of bits for the token.

  virtual Type GetType() const override { return Type::Bits; }
};

/**
 * MessageBadge - A badge instance associated with a message.
 */
struct MessageBadge {
  std::string name;     //!< Corresponds to Badge::name.
  std::string version;  //!< Corresponds to BadgeVersion::name.
};

/**
 * Message struct that contains common state shared by all message types.
 * Used by LiveChatMessage, WhisperMessage, and ChatComment.
 */
struct MessageInfo {
  MessageInfo();
  MessageInfo(const MessageInfo& src);
  MessageInfo(MessageInfo&& src) = default;
  MessageInfo& operator=(const MessageInfo& src);
  MessageInfo& operator=(MessageInfo&& src) = default;

  struct Flags {
    Flags();

    bool operator==(const Flags& rhs) const;
    bool operator!=(const Flags& rhs) const;

    bool
      action : 1;  //!< Whether or not the message is an action.  If true, it should be displayed entirely in the name text color and of the form "<userName> <message>".
    bool notice : 1;        //!< Whether or not the message is a system notice.
    bool ignored : 1;       //!< Whether or not the message was sent by a blocked user.
    bool deleted : 1;       //!< Whether or not the message has been deleted.
    bool containsBits : 1;  //!< Whether or not the message indicates the user has sent bits.
  };

  std::string userName;                               //!< The UTF-8 encoded user name.
  std::string displayName;                            //!< The UTF-8 encoded displayed name.
  std::vector<std::unique_ptr<MessageToken>> tokens;  //!< The array of message tokens.
  std::vector<MessageBadge> badges;                   //!< The array of badges.
  UserMode userMode;                                  //!< The modes of the user who sent the message.
  Flags flags;                                        //!< Any additional details about the message.
  Color nameColorARGB;                                //!< The current ARGB color of the user's name text.
  Timestamp timestamp;                                //!< The time the message was originally received.
  UserId userId;                                      //!< The unique user id.
  uint32_t numBitsSent;                               //!< The total number of bits sent via this message.
  std::string messageType;                            //!< This is the |msg-id| associated with the message.
  std::map<std::string, std::string> messageTags;     //!< This is the message tags associated with the message.
};

/**
 * A message that is being sent through live chat.
 */
struct LiveChatMessage {
  LiveChatMessage() {}

  MessageInfo messageInfo;  //!< The common information about the message.
  std::string messageId;    //!< The message id associated with a live chat message.
};

/**
 * A message that is being sent as a whisper between users.
 */
struct WhisperMessage {
  WhisperMessage();

  MessageInfo messageInfo;  //!< The common information about the message.
  std::string threadId;     //!< The UTF-8 encoded id for the thread the whisper is received on.
  MessageId messageId;      //!< The id of the whisper message.
  std::string messageUuid;  //!< The message_id of the whisper message.
};

/**
 * ChatRoomMessage - A message from a single user in a chat room.
 */
struct ChatRoomMessage {
  MessageInfo messageInfo;    //!< The information about the message
  std::string roomId;         //!< The room id the message is being sent to.
  std::string roomMessageId;  //!< The message id for the room.
};

/**
 * ExtensionMessage - a message sent to viewers of a channel with the relevant extension enabled.
 */
struct ExtensionMessage {
  ExtensionMessage();

  std::vector<std::unique_ptr<MessageToken>> tokens;  //!< The message tokens
  std::vector<MessageBadge> badges;                   //!< The badges of the extension sender
  std::string messageId;                              //!< The id of the message
  std::string extensionClientId;                      //!< The client id of the extension
  std::string extensionVersion;                       //!< The version of the extension
  std::string extensionDisplayName;                   //!< The display name of the extension
  Color chatColor;                                    //!< The chat color of the extension
  Timestamp sentAt;                                   //!< The time the message was sent
};

/**
 * SubscriptionNotice - A message from when a user shares their sub or resub to a channel.
 */
struct SubscriptionNotice {
  SubscriptionNotice();
  SubscriptionNotice(const SubscriptionNotice& source);
  SubscriptionNotice(SubscriptionNotice&& source) = default;
  SubscriptionNotice& operator=(const SubscriptionNotice& source);
  SubscriptionNotice& operator=(SubscriptionNotice&& source) = default;

  enum class Type {
    // clang-format (expand)
    // This is exposed via the public API and each enum has and expected value
    Sub = 0,
    Resub = 1,
    SubGift = 2,
    SubMassGift = 3,
    Charity = 4,
    ExtendSub = 5,
    Unknown = 6
  };

  /**
   * Plan - A type that indicates the plan/tier of the sub that is being announced.
   */
  enum class Plan {
    Prime,    //!< Sub acquired through Twitch Prime
    Sub1000,  //!< Currently $4.99 sub
    Sub2000,  //!< Currently $9.99 sub
    Sub3000,  //!< Currently $24.99 sub
    Unknown   //!< Some other plan type we did not recognize
  };

  /**
   * Recipient - Info about the recipient of the subscription (only for subgift usernotices).
   */
  struct Recipient {
    Recipient();

    std::string userName;
    std::string displayName;
    UserId userId;
  };

  std::unique_ptr<MessageInfo> userMessage;  //!< Message the user provided when sharing their subscription.
  std::string systemMessage;                 //!< Message generated by the system itself accompanying the user message.
  std::string planDisplayName;               //!< The human readable name of the sub plan.
  std::string messageId;                     //!< The id of the subscription notice message (only for usernotices).
  Recipient recipient;                       //!< The recipient of a subscription (only for SubGift usernotices).
  uint32_t subStreakMonthCount;              //!< How many consecutive months the user has been subbed.
  uint32_t subCumulativeMonthCount;          //!< How many months the user has subbed in total.
  uint32_t
    senderCount;  //!< How many gift subscriptions the sender has gifted in the current channel (only for SubGift/SubMassGift usernotices).
  uint32_t massGiftCount;  //!< How many gifts in the current mass gifting purchase (only for SubMassGift usernotices).
  uint32_t benefitEndMonth;  //!< The month the sub benifit ends: (0 - 11) => Jan - Dec
  Type type;                 //!< Whether this is a sub, resub, subgift, submassgift, or charity.
  Plan plan;                 //!< The plan/tier of the user's subscription.
  bool shouldShowSubStreak;  //!< Whether the sub streak tenure should be displayed or not.
};

/**
 * FirstTimeChatterNotice - A message indicating that a user has completed the first time chatter ritual.
 */
struct FirstTimeChatterNotice {
  MessageInfo userMessage;    //!< Message the user provided when they completed their first chatter ritual.
  std::string systemMessage;  //!< Message generated by the system itself accompanying the user message
  std::string messageId;      //!< The id of the first time chatter notice.
};

/**
 * RaidNotice - A message indicating a channel is raiding the currently joined channel.
 */
struct RaidNotice {
  RaidNotice();

  std::string systemMessage;    //!< Message generated by the system accompanying the raid.
  std::string profileImageUrl;  //!< Profile image url for the raiding channel.
  UserInfo raidingUserInfo;     //!< The user info for the raiding channel.
  uint32_t viewerCount;         //!< How many users are part of the raid.
};

/**
 * UnraidNotice - A message indicating that the current channel has stopped raiding another channel.
 */
struct UnraidNotice {
  std::string systemMessage;  //!< Message generated by the system accompanying the unraid.
};

/**
 * GenericMessageNotice - A message contains unhandled messages received on the chat channel.
 */
struct GenericMessageNotice {
  GenericMessageNotice() = default;

  //!< Message that was received on the chat channel
  MessageInfo messageInfo;

  //!< This is the |id| associated with the message info
  std::string messageId;
};

/**
 * BadgeEntitlement - Info about badge entitlement status with a bits cheer in a channel
 */
struct BadgeEntitlement {
  BadgeEntitlement();

  uint32_t newLevel;       //!< User's new badge level
  uint32_t previousLevel;  //!< User's previous badge level
  bool
    isNewBadgeLevel;  //!< Marked true if user has reached a new badge level. The fields above are only relevant if this is true.
};

/**
 * BitsReceivedEvent - A message indicating that the given channel received a bits cheer.
 */
struct BitsReceivedEvent {
  // displayName, userMode not filled out in message
  MessageInfo message;  //!< Chat message sent with the cheer.

  std::string channelName;  //!< Name of the channel on which bits were used.
  std::string context;      //!< Event type associated with this use of bits.

  ChannelId channelId;     //!< User ID of the channel on which bits were used.
  uint32_t bitsUsed;       //!< Number of bits used.
  uint32_t totalBitsUsed;  //!< All-time total number of bits used on this channel by the specified user.

  BadgeEntitlement
    badge;  //!< Information about the user's new badge level, if the user reached a new badge level with this cheer.
};

/**
 * BitsSentEvent - A message indicating that the given channel sent a bits cheer to another channel.
 */
struct BitsSentEvent {
  BitsSentEvent();

  ChannelId channelId;        //!< Channel ID of the channel recieving bits from the user.
  uint32_t userBitsBalance;   //!< Bits balance of the user cheering.
  uint32_t channelBitsTotal;  //!< Total amount of bits cheered by the user in the channel.
};

/**
 * FollowerAddedEvent - Info received about a new follower to a channel
 */
struct FollowerAddedEvent {
  std::string displayName;  //!< Display name of the person who followed the channel.
  std::string userName;     //!< Username of the person who followed the channel.
  UserId userId;            //!< User Id of the person who followed the channel.
};

/**
 * SubscriberAddedEvent - Info received about a new subscriber to a channel
 */
struct SubscriberAddedEvent {
  // userMode not filled out in subNotice.userMessage
  // systemMessage not filled out in subNotice
  SubscriptionNotice subNotice;  //!< Details of the subscription

  std::string userName;     //!< Login name of the person who subscribed to the channel.
  std::string displayName;  //!< Display name of the person who subscribed to the channel.
  std::string channelName;  //!< Name of the channel subscribed to.
  UserId userId;            //!< User ID of the person who subscribed to the channel.
  ChannelId channelId;      //!< User ID of the channel subscribed to.
  Timestamp timestamp;      //!< Time when subscribed event occured.
};

/**
 * ChatCommentSource - The origin of a comment - either from live chat or from VOD.
 */
enum class ChatCommentSource {
  Unknown,  //!< Comment originated from an unknown source.
  Comment,  //!< Comment originated from VOD.
  Chat      //!< Comment originated from the live stream chat.
};

/**
 * ChatCommentPublishedState - Whether a comment is published, pending review, or unpublished.
 */
enum class ChatCommentPublishedState {
  Unknown,            //!< Comment is in an unknown state.
  Published,          //!< Comment is published.
  Unpublished,        //!< Comment was manually unpublished by a mod or channel owner.
  PendingReview,      //!< Comment was automatically flagged for review (AutoMod).
  PendingReviewSpam,  //!< Comment is possibly spam and needs moderator review.
  Deleted             //!< Comment has been deleted.
};

/**
 * ChatComment - A comment that is posted on a VOD.
 */
struct ChatComment {
  ChatComment();

  MessageInfo messageInfo;  //!< The common information about the message.
  std::vector<ChatComment>
    replies;                    //!< Contains up to two replies to the original comment. Only applies for root comments.
  std::string commentId;        //!< The id of the chat comment.
  std::string parentCommentId;  //!< The id of the parent comment if this comment if a reply. Otherwise, empty string.
  std::string contentId;        //!< The ID of the VOD.
  ChannelId channelId;          //!< The channel ID of the VOD.
  uint64_t
    timestampMilliseconds;  //!< How far into the VOD the comment is in milliseconds. Only applies for root comments.
  Timestamp updatedAt;      //!< When the comment was last updated.
  ChatCommentSource commentSource;           //!< Whether the comment came from live chat or a VOD comment.
  ChatCommentPublishedState publishedState;  //!< The state of the message, whether it's published, pending review, etc.
  bool
    moreReplies;  //!< True if there are more replies that are not stored in the replies vector. Only applies for root comments.
};

/**
 * BadgeImage - An image to be used to render a badge at a specific scale.
 */
struct BadgeImage {
  BadgeImage();

  std::string url;  //!< The image url.
  float scale;      //!< The scale factor of the image.
};

/**
 * BadgeVersion - Information about a specific version of a badge.
 */
struct BadgeVersion {
  BadgeVersion();

  enum class Action {
    // clang-format expand
    None = 0,
    Subscribe = 1,
    VisitUrl = 2,
    GetTurbo = 3,
    GetBits = 4
  };

  std::string name;
  std::string title;
  std::string description;
  std::string clickUrl;  //!< The url to visit if clickAction is "visit_url".
  std::vector<BadgeImage> images;
  Action clickAction;  //!< The action to take when a user invokes the badge via a click.

  /**
   * Finds the closest matching image for the given scale.
   * Returns:
   *   TTV_EC_SUCCESS if found and copies into result.
   *   TTV_EC_NOT_AVAILABLE if no images available.
   */
  TTV_ErrorCode FindImage(float scale, BadgeImage& result) const;
};

/**
 * Badge - Information about a badge set which may have several versions.
 */
struct Badge {
  std::string name;                              //!< The unique name of the badge.
  std::map<std::string, BadgeVersion> versions;  //!< The versions available for rendering, keyed by version name.
};

/**
 * BadgeSet - A collection of badges that apply to a context.
 */
struct BadgeSet {
  std::string language;                 //!< The language of the badge details.
  std::map<std::string, Badge> badges;  //!< The badges in the badge set, keyed by badge name.

  /**
   * Finds the corresponding BadgeVersion for the corresponding badge.
   * Returns:
   *   TTV_EC_SUCCESS if found and copies into result.
   *   TTV_EC_NOT_AVAILABLE if not found in this badge set.
   */
  TTV_ErrorCode FindBadge(const MessageBadge& badge, BadgeVersion& result) const;

  /**
   * Finds the closest matching badge image for the given scale, if the badge exists in the given badge set.
   * Returns:
   *   TTV_EC_SUCCESS if found and copies into result.
   *   TTV_EC_NOT_AVAILABLE if not found in this badge set.
   */
  TTV_ErrorCode FindBadgeImage(const MessageBadge& badge, float scale, BadgeImage& result) const;
};

/**
 * EmoticonModifier - Describes a modifier that can be applied to an emoticon.  Valid modifiers are provided
 * for individual emoticons and are used in both the token parsing and the id to download icon assets.
 **/
struct EmoticonModifier {
  EmoticonModifier() = default;
  bool operator==(const EmoticonModifier& rhs) const { return code == rhs.code; }

  std::string code;
};

/**
 * StringEmoticon - Describes an emoticon with a simple string match.
 */
struct Emoticon {
  Emoticon();
  bool operator==(const Emoticon& rhs) const;

  std::regex regex;
  std::string match;
  std::string emoticonId;
  std::vector<EmoticonModifier> modifiers;
  bool isRegex;
};

/**
 * EmoticonSet - Describes a single emoticon set.
 */
struct EmoticonSet {
  EmoticonSet() = default;
  bool operator==(const EmoticonSet& rhs) const;

  std::vector<Emoticon> emoticons;
  std::string emoticonSetId;
  /// The display name of the user who owns this emoticon set if available.
  std::string ownerDisplayName;
};

/**
 * BitsConfiguration - Describes the various bits actions and config info for them
 */
class BitsConfiguration {
 public:
  struct CheermoteImage {
    CheermoteImage();

    enum class Theme { Unknown, Dark, Light };

    std::string url;
    Theme theme;
    float dpiScale;
    bool isAnimated;
  };

  struct CheermoteTier {
    CheermoteTier();

    std::vector<CheermoteImage> images;
    std::string tierID;
    uint32_t bits;
    Color color;
    bool canCheer;
    bool canShowInBitsCard;
  };

  struct Cheermote {
    Cheermote();

    enum class Type { Unknown, Custom, Sponsored, FirstParty, ThirdParty, DisplayOnly };

    std::string prefix;
    std::vector<CheermoteTier> tiers;
    Type type;
  };

 public:
  BitsConfiguration();
  BitsConfiguration(const std::vector<Cheermote>& cheermotes, UserId userId, ChannelId channelId);
  BitsConfiguration(std::vector<Cheermote>&& cheermotes, UserId userId, ChannelId channelId);

  /**
   * Returns the bits image URL and the color given various config arguments.
   * GetHighestDpiBitsImageUrl will return the URL that has the highest DPI less than or equal to dpiScaleLimit.
   * Returns:
   *   TTV_EC_SUCCESS if found and copies into the output variables.
   *   TTV_EC_NOT_AVAILABLE if we can't find a URL that match the given config arguments. Url will be the empty string.
   */
  TTV_ErrorCode GetBitsImageUrl(const std::string& prefix, uint32_t numBits,
    BitsConfiguration::CheermoteImage::Theme theme, float dpiScale, bool isAnimated, std::string& url,
    Color& color) const;
  TTV_ErrorCode GetHighestDpiBitsImageUrl(const std::string& prefix, uint32_t numBits,
    BitsConfiguration::CheermoteImage::Theme theme, float dpiScaleLimit, bool isAnimated, std::string& url,
    Color& color) const;

  const std::vector<Cheermote>& GetCheermotes() const;  //!< Getter for mCheermotes
  UserId GetUserId() const;                             //!< Getter for mUserId
  ChannelId GetChannelId() const;                       //!< Getter for mChannelId

 private:
  void SortCheermoteTiers();  //!< Sanity sort cheermotes by increasing tier

  std::vector<Cheermote> mCheermotes;  //!< Holds the cheermotes for the channel
  UserId mUserId;                      //!< A UserId of 0 implies the anonymous user
  ChannelId mChannelId;                //!< A ChannelId of 0 implies a global bits configuration
};

struct UnreadThreadCounts {
  UnreadThreadCounts();

  bool operator==(const UnreadThreadCounts&) const;
  bool operator!=(const UnreadThreadCounts&) const;

  uint32_t unreadThreadCount;
  uint32_t unreadMessageCount;
  bool exhaustive;
};

struct UserList {
  UserList();

  std::vector<std::string> moderators;
  std::vector<std::string> globalModerators;
  std::vector<std::string> staff;
  std::vector<std::string> admins;
  std::vector<std::string> vips;
  std::vector<std::string> viewers;

  uint32_t totalUserCount;
};

/**
 * RoomRole - Describes the level a user must be at to perform certain actions in a room, listed from lowest to highest
 * in hierarchy.
 */
enum class RoomRole { Unknown, Everyone, Subscriber, Moderator, Broadcaster };

/**
 * RoomRolePermissions - Describes the role needed to perform actions in a room.
 */
struct RoomRolePermissions {
  RoomRolePermissions();
  RoomRolePermissions(RoomRole read, RoomRole send);

  RoomRole read;  //!< The role a user must be at to read messages in the room.
  RoomRole send;  //!< The role a user must be at to send messages in the room.
};

/**
 * ChatRoomPermissions - what a user can do in a chat room.
 */
struct ChatRoomPermissions {
  ChatRoomPermissions();

  bool readMessages;  //!< The user can read messages in the chat room
  bool sendMessages;  //!< The user can send messages in the chat room
  bool moderate;      //!< The user can moderate the chat room
};

/**
 * ChatRoomView - user-specific information about what they see for a chat room.
 */
struct ChatRoomView {
  ChatRoomView();

  Timestamp lastReadAt;             //!< The last time the user read a message in the room
  uint32_t unreadMentionCount;      //!< The number of unread mentions in the room
  ChatRoomPermissions permissions;  //!< What actions the user is allowed to do in the room
  bool isMuted;                     //!< If the user has muted the room
  bool isArchived;                  //!< If the user has the room archived
  bool isUnread;                    //!< If the user has unread messages for the room
};

/**
 * ChatRoomInfo - The room information.
 */
struct ChatRoomInfo {
  UserInfo owner;                       //!< Owner of the chatroom
  ChatRoomView view;                    //!< User-specific information for the room.
  ChatModeInfo modes;                   //!< Information on what chat modes the room is in.
  std::string id;                       //!< Unique id of the chatroom
  std::string name;                     //!< Name of the chatroom
  std::string topic;                    //!< Topic of the chatroom
  RoomRolePermissions rolePermissions;  //!< Describes the roles necessary to perform certain actions in the room
};

/**
 * RoomMentionInfo - Information about a mention in a chat room
 */
struct RoomMentionInfo {
  RoomMentionInfo();

  std::string roomOwnerName;
  std::string roomOwnerLogin;
  std::string senderName;
  std::string roomId;
  std::string roomName;
  std::string messageId;
  ChannelId roomOwnerId;
  UserId senderId;
  Timestamp sentAt;
};

/**
 * RaidStatus - Information about an ongoing raid
 */
struct RaidStatus {
  RaidStatus();

  bool operator==(const RaidStatus& other) const;
  bool operator!=(const RaidStatus& other) const;

  std::string raidId;                     //!< The unique id for the raid.
  std::string targetUserLogin;            //!< The target user's login name.
  std::string targetUserDisplayName;      //!< The target user's display name.
  std::string targetUserProfileImageUrl;  //!< The target user's profile image url.
  UserId
    creatorUserId;  //!< The user's id who created the raid - can be the broadcaster, moderator of the source channel, or staff.
  ChannelId sourceChannelId;         //!< The channel which users would transition from.
  ChannelId targetChannelId;         //!< The channel users would transition to.
  uint32_t numUsersInRaid;           //!< The number of users who have opted into the raid.
  uint32_t transitionJitterSeconds;  //!< The jitter used before transitioning to the new channel for larger raids.
  uint32_t forceRaidNowSeconds;      //!< How many seconds until the client should force a raid now.
  bool joined;                       //!< Whether or not the local user has joined the raid.
};

/**
 * CommentPublishingMode - The mode for posting comments on a channel's VODs.
 */
enum class CommentPublishingMode {
  Unknown,  //!< Unknown publishing mode.
  Open,     //!< Allow all comments except those caught by AutoMod.
  Review,   //!< Hold all comments on video for review.
  Disabled  //!< Disable all comments on videos.
};

/**
 * ChannelVodCommentSettings - Channel-specific settings relating to commenting on that channel's VODs.
 */
struct ChannelVodCommentSettings {
  ChannelVodCommentSettings();

  ChannelId channelId;                    //!< ID of the channel.
  Timestamp createdAt;                    //!< When these settings were created.
  Timestamp updatedAt;                    //!< When these settings were last updated.
  uint32_t followersOnlyDurationSeconds;  //!< The duration someone is required to follow in order to post comments.
  CommentPublishingMode publishingMode;   //!< The comment publishing mode for the channel's VODs.
};

/**
 * ModerationAction - Info on the moderator initiating the action and the targetted user
 */
struct ModerationActionInfo {
  std::string moderatorName;
  std::string targetName;
  UserId moderatorId;
  UserId targetId;
};

/**
 * MultiviewContentAttribute - holds information about properties of chanlets
 */
struct MultiviewContentAttribute {
  MultiviewContentAttribute();

  std::string attributeId;
  std::string key;
  std::string name;
  std::string parentId;
  std::string parentKey;
  std::string value;
  std::string valueShortName;
  std::string imageUrl;
  ChannelId ownerChannelId;
  Timestamp createdAt;
  Timestamp updatedAt;
};

/**
 * Chanlet - a channel part of a multi-view stream.
 */
struct Chanlet {
  Chanlet();

  std::vector<MultiviewContentAttribute> attributes;
  ChannelId chanletId;
};
}  // namespace chat
}  // namespace ttv
