/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"

namespace ttv {
namespace chat {
class IChannelChatRoomManager;
}
}  // namespace ttv

/**
 * The interface to manage a channel's chat rooms.
 */
class ttv::chat::IChannelChatRoomManager {
 public:
  /**
   * Used with JoinChatRooms().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The rooms was successfully joined.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to join the rooms.
   */
  using JoinCallback = std::function<void(TTV_ErrorCode ec)>;

  /**
   * Used with LeaveChatRooms().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The rooms was successfully left.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to leave the rooms.
   */
  using LeaveCallback = std::function<void(TTV_ErrorCode ec)>;

  /**
   * Used with AddNewChatRoom().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The channel has created a new chat room.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to create a new chat room.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   * @param[in] error Error object describing the reason why creating the room failed.
   * @param[in] info The information of the newly created room.
   */
  using AddRoomCallback = std::function<void(TTV_ErrorCode ec, CreateRoomError&& error, ChatRoomInfo&& info)>;

  /**
   * Used with FetchRoomsInfo().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The channel has created a new chat room.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to create a new chat room.
   * @param[in] infos A list of the information for chat rooms on the channel.
   */
  using FetchRoomsInfoCallback = std::function<void(TTV_ErrorCode ec, std::vector<ChatRoomInfo>&& infos)>;

  virtual ~IChannelChatRoomManager() = default;

  /**
   * This should be called when the application is done with the instance.
   *
   * @return
   *   - TTV_EC_SUCCESS: Instance has been disposed.
   */
  virtual TTV_ErrorCode Dispose() = 0;

  /**
   * Adds a new chat room for the channel.
   * Currently to create a new room, the user must own the channel (userId and channelId must be the same).
   *
   * @param[in] roomName The name of the newly created room.
   * @param[in] topic The topic of the newly created room.
   * @param[in] permissions The minimum roles required to take certain actions in the room.
   * @param[in] callback Callback when API call returns, receives an error code and the newly created room's
   * information.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to add a new chat room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   *   - TTV_EC_INVALID_CHANNEL_ID: ChannelChatRoomManager's user id does not match channel id (currently the user must
   * own the channel to add a new room)
   *   - TTV_EC_INVALID_ARG: Permissions need to have a valid role.
   */
  virtual TTV_ErrorCode AddNewChatRoom(const std::string& roomName, const std::string& topic,
    RoomRolePermissions permissions, const AddRoomCallback& callback) = 0;

  /**
   * Join the chat rooms for a channel.
   *
   * @param[in] callback Callback when the API call returns, receives an error code.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to join the channel.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode JoinChatRooms(const JoinCallback& callback) = 0;

  /**
   * Leave the chat rooms for a channel.
   *
   * @param[in] callback Callback when the API call returns, receives an error code.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to leave the channel.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode LeaveChatRooms(const LeaveCallback& callback) = 0;

  /**
   * Return all of the information for chat rooms on this channel.
   *
   * @param[in] callback Callback when the API call returns, receives an error code and a list of information for the
   * chat rooms of the channel.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to fetch the chat rooms information.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode FetchChatRoomsInfo(const FetchRoomsInfoCallback& callback) = 0;
};
