/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"

namespace ttv {
namespace chat {
class IChannelChatRoomManagerListener;
}
}  // namespace ttv

/**
 * Listener interface for unsolicited events from IChannelChatRoomManager.
 */
class ttv::chat::IChannelChatRoomManagerListener {
 public:
  virtual ~IChannelChatRoomManagerListener() = default;

  /**
   * Called when a user's messages need to be purged from chat rooms after a ban/timeout.
   *
   * @param[in] userId The id of the user whose messages need to be purged.
   * @param[in] channelId The channel that needs to be purged of the messages.
   * @param[in] purgeAfter The timestamp to start purging messages from that user.
   */
  virtual void PurgeMessages(UserId userId, ChannelId channelId, Timestamp purgeAfter) = 0;

  /**
   * Called when a new room is created in the channel.
   *
   * @param[in] ownerId The id of the channel that owns the room.
   * @param[in] roomInfo The information of the newly created room. The view and owner fields are invalid.
   */
  virtual void RoomCreated(ChannelId ownerId, ChatRoomInfo&& roomInfo) = 0;

  /**
   * Called when a new room is deleted in the channel.
   *
   * * @param[in] ownerId The id of the channel that owns the room.
   * @param[in] roomInfo The information of the deleted room. The view and owner fields are invalid.
   */
  virtual void RoomDeleted(ChannelId ownerId, ChatRoomInfo&& roomInfo) = 0;
};
