/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"

namespace ttv {
namespace chat {
class IChatChannelPropertyListener;
class ChatChannelPropertyListenerProxy;
}  // namespace chat
}  // namespace ttv

/**
 * Listener interface for unsolicited events for chat channel property updates.
 */
class ttv::chat::IChatChannelPropertyListener {
 public:
  virtual ~IChatChannelPropertyListener() = default;

  /**
   * Called when rituals is enabled or disabled on a channel by the owner.
   *
   * @param[in] ritualsEnabled Whether or not rituals is enabled.
   */
  virtual void RitualsEnabled(bool ritualsEnabled) = 0;

  /**
   * Called when the current channel changes the channel that they're hosting.
   *
   * @param[in] channelId The current channel, the one hosting.
   * @param[in] previousTarget The previously hosted channel. Is 0 if channel wasn't hosting anyone before.
   * @param[in] currentTarget The channel now being hosted.
   * @param[in] currentTargetName The name of the channel now being hosted.
   * @param[in] numViewers How many viewers are viewing through the host.
   */
  virtual void OutgoingHostChanged(ChannelId channelId, ChannelId previousTarget, ChannelId currentTarget,
    std::string&& currentTargetName, uint32_t numViewers) = 0;

  /**
   * Called when the channel starts being hosted by another channel.
   *
   * @param[in] channelId The current channel, the one being hosted.
   * @param[in] hostChannelId The channel ID of the host.
   * @param[in] hostChannelName The channel name of the host.
   * @param[in] numViewers How many viewers are viewing through the host.
   */
  virtual void IncomingHostStarted(
    ChannelId channelId, ChannelId hostChannelId, std::string&& hostChannelName, uint32_t numViewers) = 0;

  /**
   * Called when the channel stops being hosted by another channel.
   *
   * @param[in] channelId The current channel, the one that is no longer being hosted.
   * @param[in] hostChannelId The channel ID of the ex-host.
   * @param[in] hostChannelName The channel name of the ex-host.
   */
  virtual void IncomingHostEnded(ChannelId channelId, ChannelId hostChannelId, std::string&& hostChannelName) = 0;

  /**
   * Called when an extension a user is using sends a message to the channel
   *
   * @param[in] message The message the extension sent.
   */
  virtual void ExtensionMessageReceived(ExtensionMessage&& message) = 0;

  /**
   * Called when the chat channel restrictions are changed.
   *
   * @param[in] chatChannelRestrictions The restrictions modes for the channel.
   */
  virtual void ChatChannelRestrictionsReceived(ChatChannelRestrictions&& chatChannelRestrictions) = 0;
};

class ttv::chat::ChatChannelPropertyListenerProxy : public IChatChannelPropertyListener {
 public:
  using RitualsEnabledCallback = std::function<void(bool ritualsEnabled)>;
  using OutgoingHostChangedCallback = std::function<void(ChannelId channelId, ChannelId previousTarget,
    ChannelId currentTarget, std::string&& currentTargetName, uint32_t numViewers)>;
  using IncomingHostStartedCallback = std::function<void(
    ChannelId channelId, ChannelId hostChannelId, std::string&& hostChannelName, uint32_t numViewers)>;
  using IncomingHostEndedCallback =
    std::function<void(ChannelId channelId, ChannelId hostChannelId, std::string&& hostChannelName)>;
  using ExtensionMessageReceivedCallback = std::function<void(ExtensionMessage&& message)>;
  using ChatChannelRestrictionsReceivedCallback =
    std::function<void(ChatChannelRestrictions&& chatChannelRestrictions)>;

  // IChatChannelPropertyListener implementation
  virtual void RitualsEnabled(bool ritualsEnabled) override {
    if (ritualsEnabledCallback != nullptr) {
      ritualsEnabledCallback(ritualsEnabled);
    }
  }

  virtual void OutgoingHostChanged(ChannelId channelId, ChannelId previousTarget, ChannelId currentTarget,
    std::string&& currentTargetName, uint32_t numViewers) override {
    if (outgoingHostChangedCallback != nullptr) {
      outgoingHostChangedCallback(channelId, previousTarget, currentTarget, std::move(currentTargetName), numViewers);
    }
  }

  virtual void IncomingHostStarted(
    ChannelId channelId, ChannelId hostChannelId, std::string&& hostChannelName, uint32_t numViewers) override {
    if (incomingHostStartedCallback != nullptr) {
      incomingHostStartedCallback(channelId, hostChannelId, std::move(hostChannelName), numViewers);
    }
  }

  virtual void IncomingHostEnded(ChannelId channelId, ChannelId hostChannelId, std::string&& hostChannelName) override {
    if (incomingHostEndedCallback != nullptr) {
      incomingHostEndedCallback(channelId, hostChannelId, std::move(hostChannelName));
    }
  }

  virtual void ExtensionMessageReceived(ExtensionMessage&& message) override {
    if (extensionMessageReceivedCallback != nullptr) {
      extensionMessageReceivedCallback(std::move(message));
    }
  }

  virtual void ChatChannelRestrictionsReceived(ChatChannelRestrictions&& chatChannelRestrictions) override {
    if (chatChannelRestrictionsReceivedCallback != nullptr) {
      chatChannelRestrictionsReceivedCallback(std::move(chatChannelRestrictions));
    }
  }

  RitualsEnabledCallback ritualsEnabledCallback;
  OutgoingHostChangedCallback outgoingHostChangedCallback;
  IncomingHostStartedCallback incomingHostStartedCallback;
  IncomingHostEndedCallback incomingHostEndedCallback;
  ExtensionMessageReceivedCallback extensionMessageReceivedCallback;
  ChatChannelRestrictionsReceivedCallback chatChannelRestrictionsReceivedCallback;
};
