/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/core/result.h"

#include <unordered_set>

#include <map>
#include <memory>
#include <string>
#include <vector>

namespace ttv {
namespace chat {
class IChatCommentManager;
}
}  // namespace ttv

/**
 * The interface for VOD chat comments functionality.
 *
 * There are two ways to retrieve comments. The first is through using the Playhead, which can be controlled with the
 * functions Pause(), Play(), Seek(). The Playhead is initally set to Paused at the beginning of the VOD. Once playing,
 * comments are fired through the listener once their timestamp has been reached.
 *
 * The second way to retrieve comments is an explicit fetch for a range of messages using FetchComments(), which will
 * fetch comments up to the passed in limit, starting from either the specified timestamp or from the cursor. Comments
 * are then sent through the callback.
 */
class ttv::chat::IChatCommentManager {
 public:
  /**
   * PlayingState - Whether the manager is currently playing comments or not.
   */
  enum class PlayingState {
    Paused,     //!< The playhead is paused.
    Playing,    //!< The playhead is moving.
    Buffering,  //!< We are blocked on fetching more comments, playhead is stopped.
    Finished,   //!< The playhead has reached the end of the comments.
  };

  using FetchCommentsCallback =
    std::function<void(TTV_ErrorCode ec, std::vector<ChatComment>&& result, std::string&& nextCursor)>;
  using FetchCommentCallback = std::function<void(TTV_ErrorCode ec, ChatComment&& result)>;
  using DeleteCommentCallback = std::function<void(TTV_ErrorCode ec)>;
  using PostCommentCallback = std::function<void(TTV_ErrorCode ec, ChatComment&& result, std::string&& errorMessage)>;
  using ReportCommentCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchCommentRepliesCallback = std::function<void(TTV_ErrorCode ec, std::vector<ChatComment>&& result)>;
  using PostCommentReplyCallback = std::function<void(TTV_ErrorCode ec, ChatComment&& result)>;

 public:
  virtual ~IChatCommentManager();

  /**
   * This should be called when the application is done with the instance.
   */
  virtual TTV_ErrorCode Dispose() = 0;

  /**
   * Initiates playback of the comments and causes the playhead to advance. Should be called when the video player
   * starts playing, or resumes from buffering. Has no effect when current
   * ChatCommentManagerState is BUFFERING, because playback will automatically resume when done buffering.
   *
   * @return
   *   - TTV_EC_SUCCESS: Successfully set to a Playing state.
   */
  virtual TTV_ErrorCode Play() = 0;

  /**
   * Pauses the playhead. Should be called when the video player's pause is called, or when the video player starts
   * buffering.
   *
   * @return
   *   - TTV_EC_SUCCESS: Successfully set to a Paused state.
   */
  virtual TTV_ErrorCode Pause() = 0;

  /**
   * Sets the playhead's time to the given timestamp. This function must be called whenever the playhead is updated to
   * keep the ChatCommentManager in sync with the video.
   *
   * @param[in] timestampMilliseconds The number of milliseconds into the VOD that we're setting the playhead to.
   * @return
   *   - TTV_EC_SUCCESS: Successfully set the playhead to timestampMilliseconds
   */
  virtual TTV_ErrorCode UpdatePlayhead(uint64_t timestampMilliseconds) = 0;

  /**
   * Returns the time in milliseconds for the playhead, starting from the start of the VOD.
   *
   * @return Result containing an error code and timestamp of the playhead in milliseconds.
   *   - TTV_EC_SUCCESS: Successfully returned the current playhead timestamp.
   */
  virtual Result<uint64_t> GetPlayheadTime() const = 0;

  /**
   * Returns the channel that owns the VOD.
   *
   * @return Result containing an error code and channel ID.
   *   - TTV_EC_SUCCESS: Successfully returned the channel ID.
   */
  virtual Result<ChannelId> GetChannelId() const = 0;

  /**
   * Returns the current ChatCommentManager state.
   *
   * @return Result containing an error code and the playing state.
   *   - TTV_EC_SUCCESS: Successfully returned the current player state of the manager.
   */
  virtual Result<PlayingState> GetPlayingState() const = 0;

  /**
   * Fetches a set of messages at offsetSeconds and passes back to the callback.
   * This function fetches messages independently of the playhead.
   *
   * @param[in] timestampMilliseconds How many milliseconds into the VOD we want to start fetching comments at.
   * @param[in] limit The max number of comments to return. The maximum allowed by the back-end is 60.
   * @param[in] callback The callback function that will receive the retrieved comments.
   * @return
   *   - TTV_EC_SUCCESS: Successfully sent a request to fetch the next batch of comments.
   *   - TTV_EC_SHUT_DOWN: The component is shutting down.
   *   - TTV_EC_INVALID_ARG: Limit cannot be 0.
   */
  virtual TTV_ErrorCode FetchComments(
    uint64_t timestampMilliseconds, uint32_t limit, const FetchCommentsCallback& callback) = 0;

  /**
   * Fetches a set of messages at cursor and passes back to the callback.
   * This function fetches messages independently of the playhead.
   *
   * @param[in] cursor A url for the next batch of comments to fetch.
   * @param[in] limit The max number of comments to return. The maximum allowed by the back-end is 60.
   * @param[in] callback The callback function that will receive the retrieved comments.
   * @return
   *   - TTV_EC_SUCCESS: Successfully sent a request to fetch the next batch of comments.
   *   - TTV_EC_SHUT_DOWN: The component is shutting down.
   *   - TTV_EC_INVALID_ARG: Cursor cannot be empty string. Limit cannot be 0.
   */
  virtual TTV_ErrorCode FetchComments(
    const std::string& cursor, uint32_t limit, const FetchCommentsCallback& callback) = 0;

  /**
   * Fetches the given comment from the VOD. Does not include the replies to the comment.
   *
   * @param[in] commentId The ID of the comment we want to retrieve.
   * @param[in] callback Function to be called after we have retrieved the chat comment.
   * @return
   *   - TTV_EC_SUCCESS: Successfuly sent a request to fetch the comment.
   *   - TTV_EC_INVALID_ARG: Comment id is empty.
   *   - TTV_EC_SHUT_DOWN: The component is shutting down.
   */
  virtual TTV_ErrorCode FetchComment(const std::string& commentId, const FetchCommentCallback& callback) = 0;

  /**
   * Deletes the given comment from the VOD so that it cannot be viewed by other users.
   *
   * @param[in] commentId The ID of the comment we want to delete.
   * @param[in] callback Function to be called after we have deleted the chat comment.
   * @return
   *   - TTV_EC_SUCCESS: Successfuly sent a request to delete the comment.
   *   - TTV_EC_INVALID_ARG: Comment id is empty.
   *   - TTV_EC_AUTHENTICATION: User is not properly logged in.
   *   - TTV_EC_SHUT_DOWN: The component is shutting down.
   */
  virtual TTV_ErrorCode DeleteComment(const std::string& commentId, const DeleteCommentCallback& callback) = 0;

  /**
   * Posts a new comment with the specified message at the specified time for the ChatCommentManager's VOD.
   *
   * @param[in] message The content of the comment we want to post.
   * @param[in] timestampMilliseconds The time of the VOD we want to post the comment at.
   * @param[in] callback Function to be called after we have posted the chat comment.
   * @return
   *   - TTV_EC_SUCCESS: Successfuly sent a request to post a new comment.
   *   - TTV_EC_INVALID_ARG: Message is empty.
   *   - TTV_EC_AUTHENTICATION: User is not properly logged in.
   *   - TTV_EC_SHUT_DOWN: The component is shutting down.
   */
  virtual TTV_ErrorCode PostComment(
    const std::string& message, uint64_t timestampMilliseconds, const PostCommentCallback& callback) = 0;

  /**
   * Reports the given comment.
   *
   * @param[in] commentId The id of the comment we want to report.
   * @param[in] reason The reason the comment is being reported. This field is required.
   * @param[in] description Further description of the report comment. If not needed, can pass in the empty string.
   * @param[in] callback Function to be called after we have reported the chat comment.
   * @return
   *   - TTV_EC_SUCCESS: Successfuly sent a request to report the comment.
   *   - TTV_EC_INVALID_ARG: Comment id or reason is empty.
   *   - TTV_EC_AUTHENTICATION: User is not properly logged in.
   *   - TTV_EC_SHUT_DOWN: The component is shutting down.
   */
  virtual TTV_ErrorCode ReportComment(const std::string& commentId, const std::string& reason,
    const std::string& description, const ReportCommentCallback& callback) = 0;

  /**
   * Fetches the replies to the given comment.
   *
   * @param[in] commentId The id of the comment we want to get the replies for.
   * @param[in] callback Function to be called after we have received the replies.
   * @return
   *   - TTV_EC_SUCCESS: Successfuly sent a request to fetch the replies to the comment.
   *   - TTV_EC_INVALID_ARG: Comment id is empty.
   *   - TTV_EC_SHUT_DOWN: The component is shutting down.
   */
  virtual TTV_ErrorCode FetchCommentReplies(
    const std::string& commentId, const FetchCommentRepliesCallback& callback) = 0;

  /**
   * Posts a reply to the given comment with the specified message.
   *
   * @param[in] commentId The id of the comment we want to post a reply for.
   * @param[in] message The message we want to post in the reply.
   * @param[in] callback Function to be called after we have posted the reply.
   * @return
   *   - TTV_EC_SUCCESS: Successfuly sent a request to post a reply to the comment.
   *   - TTV_EC_INVALID_ARG: Comment id or message is empty.
   *   - TTV_EC_AUTHENTICATION: User is not properly logged in.
   *   - TTV_EC_SHUT_DOWN: The component is shutting down.
   */
  virtual TTV_ErrorCode PostCommentReply(
    const std::string& parentCommentId, const std::string& message, const PostCommentReplyCallback& callback) = 0;
};
