/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"

namespace ttv {
namespace chat {
class IChatRaid;
}
}  // namespace ttv

/**
 * The interface for chat raid functionality.
 *
 * A chat raid is a mass transition of users from one channel to another.
 */
class ttv::chat::IChatRaid {
 public:
  using JoinCallback = std::function<void(TTV_ErrorCode ec)>;
  using LeaveCallback = std::function<void(TTV_ErrorCode ec)>;
  using StartCallback = std::function<void(TTV_ErrorCode ec)>;
  using CancelCallback = std::function<void(TTV_ErrorCode ec)>;
  using RaidNowCallback = std::function<void(TTV_ErrorCode ec)>;

 public:
  virtual ~IChatRaid() = default;

  /**
   * This should be called when the application is done with the instance.
   */
  virtual TTV_ErrorCode Dispose() = 0;

  /**
   * As a viewer, joins the raid.
   * @param[in] raidId The id of the raid that is being joined.
   * @param[in] callback Callback function called after joining the raid. Can be null.
   * @return
   *  - TTV_EC_SUCCESS: The raid was joined successfully.
   *  - TTV_EC_SHUT_DOWN: The chat module is not initialized.
   *  - TTV_EC_NEED_TO_LOGIN: User needs to be logged in to participate in raid.
   *  - TTV_EC_AUTHENTICATION: User has an invalid OAuth token.
   */
  virtual TTV_ErrorCode Join(const std::string& raidId, const JoinCallback& callback) = 0;

  /**
   * As a viewer, leaves the raid.  You can only leave a raid if you have previously joined.
   * @param[in] raidId The id of the raid that is being left.
   * @param[in] callback Callback function called after leaving the raid. Can be null.
   * @return
   *  - TTV_EC_SUCCESS: The raid was left successfully.
   *  - TTV_EC_SHUT_DOWN: The chat module is not initialized.
   *  - TTV_EC_NEED_TO_LOGIN: User needs to be logged in to participate in raid.
   *  - TTV_EC_AUTHENTICATION: User has an invalid OAuth token.
   */
  virtual TTV_ErrorCode Leave(const std::string& raidId, const LeaveCallback& callback) = 0;

  /**
   * As a channel editor or broadcaster, this starts a raid on the channel targeting the given user.
   * @param[in] targetUserId The id of the user that is being targetted by the raid.
   * @param[in] callback Callback function called after starting the raid. Can be null.
   * @return
   *  - TTV_EC_SUCCESS: The raid was started successfully.
   *  - TTV_EC_SHUT_DOWN: The chat module is not initialized.
   *  - TTV_EC_NEED_TO_LOGIN: User needs to be logged in to start a raid.
   *  - TTV_EC_AUTHENTICATION: User has an invalid OAuth token.
   */
  virtual TTV_ErrorCode Start(UserId targetUserId, const StartCallback& callback) = 0;

  /**
   * As a channel editor or broadcaster, this causes the raid to fire.
   * A raid must have been already started on the channel for anything to happen.
   * @return
   *  - TTV_EC_SUCCESS: The raid now request was sent.
   *  - TTV_EC_SHUT_DOWN: The chat module is not initialized.
   *  - TTV_EC_NEED_TO_LOGIN: User needs to be logged in to request a raid now.
   *  - TTV_EC_AUTHENTICATION: User has an invalid OAuth token.
   */
  virtual TTV_ErrorCode RaidNow(const RaidNowCallback& callback) = 0;

  /**
   * As a channel moderator or broadcaster, this cancels a raid. This only works if a raid has been started
   * and is currently underway.
   * @param[in] targetUserId The id of the user that is being targetted by the raid.
   * @param[in] callback Callback function called after cancelling the raid. Can be null.
   * @return
   *  - TTV_EC_SUCCESS: The raid was cancelled successfully.
   *  - TTV_EC_SHUT_DOWN: The chat module is not initialized.
   *  - TTV_EC_NEED_TO_LOGIN: User needs to be logged in to leave the raid.
   *  - TTV_EC_AUTHENTICATION: User has an invalid OAuth token.
   */
  virtual TTV_ErrorCode Cancel(const CancelCallback& callback) = 0;
};
