/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"

namespace ttv {
namespace chat {
class IChatRoom;
}
}  // namespace ttv

/**
 * The interface for a chat room.
 */
class ttv::chat::IChatRoom {
 public:
  /**
   * Used with DeleteRoom().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The room was successfully deleted.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to delete the room.
   */
  using DeleteRoomCallback = std::function<void(TTV_ErrorCode ec)>;

  /**
   * Used with SendMessage().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The message was successfully sent.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Received an invalid response from the back-end.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   * @param[in] error Error object received from GraphQL describing reason why sending message failed.
   * @param[in] message The message received back from the server.
   */
  using SendMessageCallback =
    std::function<void(TTV_ErrorCode ec, SendRoomMessageError&& error, ChatRoomMessage&& message)>;

  /**
   * Used with EditMessage().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The message was successfully edited.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to edit the message.
   * @param[in] message The message received back from the server.
   */
  using EditMessageCallback = std::function<void(TTV_ErrorCode ec, ChatRoomMessage&& message)>;

  /**
   * Used with DeleteMessage().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The message was successfully deleted.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to delete the message.
   * @param[in] message The message received back from the server.
   */
  using DeleteMessageCallback = std::function<void(TTV_ErrorCode ec)>;

  /**
   * Used with FetchMessages functions.
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The messages were retrieved.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to fetch the messages.
   * @param[in] messages The list of messages received back from the server. If fetching before a time/cursor, listed
   * from newest to oldest. Else if fetching after a time/cursor, listed from oldest to newest.
   * @param[in] nextCursor The cursor to use next time to fetch messages from where we left off, in the same pagination
   * direction.
   * @param[in] moreMessages True if there are more messages to be fetched at the cursor, else false.
   */
  using FetchMessagesCallback = std::function<void(
    TTV_ErrorCode ec, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages)>;

  /**
   * Used when updating the information of a room.
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The room info was changed.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to change the room info.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   * @param[in] error Error object received from GraphQL describing the reason why updating room info failed.
   * @param[in] info The updated information of the room.
   */
  using UpdateRoomInfoCallback = std::function<void(TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& info)>;

  /**
   * Used when updating a room's chat mode.
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The room's chat mode was changed.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to change the room's chat mode.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   * @param[in] error Error object received from GraphQL describing the reason why updating room mode failed.
   * @param[in] info The updated information of the room.
   */
  using UpdateRoomModesCallback =
    std::function<void(TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info)>;

  /**
   * Used when updating a room's view.
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The room's view was changed.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to change the room's view.
   * @param[in] info The updated information of the room.
   */
  using UpdateRoomViewCallback = std::function<void(TTV_ErrorCode ec, ChatRoomInfo&& info)>;

  /**
   * Used with FetchRoomInfo().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The room info was retrieved.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Something went wrong when trying to retrieve the room info.
   * @param[in] info The updated information of the room.
   */
  using FetchRoomInfoCallback = std::function<void(TTV_ErrorCode ec, ChatRoomInfo&& info)>;

  virtual ~IChatRoom() = default;

  /**
   * This should be called when the application is done with the instance.
   *
   * @return
   *   - TTV_EC_SUCCESS: Instance has been disposed.
   */
  virtual TTV_ErrorCode Dispose() = 0;

  /**
   * Deletes the chat room from the channel. User must be the owner of the chat room.
   *
   * @param[in] callback Callback when API call returns, receives an error code.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to delete the room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode DeleteRoom(const DeleteRoomCallback& callback) = 0;

  /**
   * Send a message to the chat room. User must be joined to the chat room.
   *
   * @param[in] message Content of the message we want to send to the chat room.
   * @param[out] placeholderMessage A ChatRoomMessage created client-side to be displayed until we receive the actual
   * message result from the backend.
   * @param[in] callback Callback when API call returns, receives an error code and the sent message.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to send the message.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode SendMessage(
    const std::string& message, ChatRoomMessage& placeholderMessage, const SendMessageCallback& callback) = 0;

  /**
   * Edits a message's content in the chat room. User must have written the message.
   *
   * @param[in] messageId The id of the message we want to edit.
   * @param[in] message The content that we want to set the message to.
   * @param[out] placeholderMessage A ChatRoomMessage created client-side to be displayed until we receive the actual
   * message result from the backend.
   * @param[in] callback Callback when API call returns, receives an error code and the edited message.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to edit the message.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode EditMessage(const std::string& messageId, const std::string& message,
    ChatRoomMessage& placeholderMessage, const EditMessageCallback& callback) = 0;

  /**
   * Deletes a message in the chat room.
   *
   * @param[in] messageId The id of the message we want to delete.
   * @param[in] callback Callback when API call returns, receives an error code.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to delete the message.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode DeleteMessage(const std::string& messageId, const DeleteMessageCallback& callback) = 0;

  /**
   * Fetches older messages from the chat room starting at the cursor, from newest to oldest.
   *
   * @param[in] cursor The cursor that we want to start fetching messages from. If the empty string is passed in, we
   * fetch the most recent messages in the room.
   * @param[in] limit The max number of messages to fetch. Can be a number from 1-100.
   * @param[in] callback Callback when API call returns, receives an error code and the list of messages.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to fetch the messages.
   *   - TTV_EC_INVALID_ARG: One of the parameters passed in is not a valid value.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode FetchMessagesBeforeCursor(
    const std::string& cursor, uint32_t limit, const FetchMessagesCallback& callback) = 0;

  /**
   * Fetches newer messages from the chat room starting at the cursor, from oldest to newest.
   *
   * @param[in] cursor The cursor that we want to start fetching messages from. If the empty string is passed in, we
   * fetch the oldest messages in the room.
   * @param[in] limit The max number of messages to fetch. Can be a number from 1-100.
   * @param[in] callback Callback when API call returns, receives an error code and the list of messages.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to fetch the messages.
   *   - TTV_EC_INVALID_ARG: One of the parameters passed in is not a valid value.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode FetchMessagesAfterCursor(
    const std::string& cursor, uint32_t limit, const FetchMessagesCallback& callback) = 0;

  /**
   * Fetches messages from the chat room before the timestamp, from newest to oldest.
   *
   * @param[in] timestamp The timestamp that we want to start fetching messages from.
   * @param[in] limit The max number of messages to fetch. Can be a number from 1-100.
   * @param[in] callback Callback when API call returns, receives an error code and the list of messages.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to fetch the messages.
   *   - TTV_EC_INVALID_ARG: One of the parameters passed in is not a valid value.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode FetchMessagesBeforeTimestamp(
    Timestamp timestamp, uint32_t limit, const FetchMessagesCallback& callback) = 0;

  /**
   * Fetches messages from the chat room and after the timestamp (inclusive), from oldest to newest.
   *
   * @param[in] timestamp The timestamp that we want to start fetching messages from.
   * @param[in] limit The max number of messages to fetch. Can be a number from 1-100.
   * @param[in] callback Callback when API call returns, receives an error code and the list of messages.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to fetch the messages.
   *   - TTV_EC_INVALID_ARG: One of the parameters passed in is not a valid value.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode FetchMessagesAfterTimestamp(
    Timestamp timestamp, uint32_t limit, const FetchMessagesCallback& callback) = 0;

  /**
   * Fetches the information of the chat room and returns it in the callback.
   *
   * @param[in] callback Callback when API call returns, receives an error code and the room info.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to fetch the room info.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode FetchRoomInfo(const FetchRoomInfoCallback& callback) = 0;

  /**
   * Update the name of the chat room.
   *
   * @param[in] name The new name of the room.
   * @param[in] callback Callback when API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to set the room name.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode SetRoomName(const std::string& name, const UpdateRoomInfoCallback& callback) = 0;

  /**
   * Update the topic of the chat room.
   *
   * @param[in] name The new topic of the room.
   * @param[in] callback Callback when API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to set the room topic.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode SetTopic(const std::string& topic, const UpdateRoomInfoCallback& callback) = 0;

  /**
   * Update the minimum roles required to perform certain actions in the chat room.
   *
   * @param[in] permissions The new minimum required roles to perform actions in the room.
   * @param[in] callback Callback when API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to set the room's minimum role permissions.
   *   - TTV_EC_INVALID_ARG: The room roles passed in cannot be Unknown.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode SetRoomRolePermissions(
    RoomRolePermissions permissions, const UpdateRoomInfoCallback& callback) = 0;

  /**
   * Turns on slow mode for the chat room.
   *
   * @param[in] durationSeconds The number of seconds a user must wait between sending messages. Cannot be 0.
   * @param[in] callback Callback when the API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've send the request to turn on slow mode for the chat room.
   *   - TTV_EC_INVALID_ARG: durationSeconds cannot be 0.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode EnableSlowMode(uint32_t durationSeconds, const UpdateRoomModesCallback& callback) = 0;

  /**
   * Turns off slow mode for the chat room.
   *
   * @param[in] callback Callback when the API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've send the request to turn off slow mode for the chat room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode DisableSlowMode(const UpdateRoomModesCallback& callback) = 0;

  /**
   * Turns on r9k mode for the chat room.
   *
   * @param[in] callback Callback when the API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've send the request to turn on r9k mode for the chat room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode EnableR9kMode(const UpdateRoomModesCallback& callback) = 0;

  /**
   * Turns off r9k mode for the chat room.
   *
   * @param[in] callback Callback when the API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've send the request to turn off r9k mode for the chat room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode DisableR9kMode(const UpdateRoomModesCallback& callback) = 0;

  /**
   * Turns on emotes-only mode for the chat room.
   *
   * @param[in] callback Callback when the API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've send the request to turn on emotes-only mode for the chat room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode EnableEmotesOnlyMode(const UpdateRoomModesCallback& callback) = 0;

  /**
   * Turns off emotes-only mode for the chat room.
   *
   * @param[in] callback Callback when the API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've send the request to turn off emotes-only mode for the chat room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode DisableEmotesOnlyMode(const UpdateRoomModesCallback& callback) = 0;

  /**
   * Update when the user last read a message in the chat room.
   *
   * @param[in] lastReadAt The time the user last read a message.
   * @param[in] callback Callback when API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request set the last read at time.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode SetLastReadAt(Timestamp lastReadAt, const UpdateRoomViewCallback& callback) = 0;

  /**
   * Update if the user has muted the chat room.
   *
   * @param[in] isMuted Whether the user has muted the room.
   * @param[in] callback Callback when API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to mute the room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode SetMuted(bool isMuted, const UpdateRoomViewCallback& callback) = 0;

  /**
   * Update if the user has archived the chat room.
   *
   * @param[in] isArchived Whether the user has archived the room.
   * @param[in] callback Callback when API call returns, receives an error code and the new room info.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to archive the room.
   *   - TTV_EC_SHUT_DOWN: The chat room object is not initialized.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not properly logged in.
   */
  virtual TTV_ErrorCode SetArchived(bool isArchived, const UpdateRoomViewCallback& callback) = 0;
};
