/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"

namespace ttv {
namespace chat {
class IChatRoomNotificationsListener;
}
}  // namespace ttv

/**
 * Listener interface for unsolicited events for chat rooms on a user level.
 */
class ttv::chat::IChatRoomNotificationsListener {
 public:
  virtual ~IChatRoomNotificationsListener() = default;

  /**
   * Called when the user is timed out from a channel.
   *
   * @param[in] userId The id of the user that was timed out.
   * @param[in] channelId The channel that the user was timed out of.
   * @param[in] expiresAt The timestamp when the timeout is over.
   */
  virtual void UserTimedOut(UserId userId, ChannelId channelId, Timestamp expiresAt) = 0;

  /**
   * Called when the user is banned from a channel.
   *
   * @param[in] userId The id of the user that was banned.
   * @param[in] channelId The channel that the user was banned from.
   */
  virtual void UserBanned(UserId userId, ChannelId channelId) = 0;

  /**
   * Called when the user is unbanned from a channel.
   *
   * @param[in] userId The id of the user that was unbanned
   * @param[in] channelId The channel that the user was unbanned from.
   */
  virtual void UserUnbanned(UserId userId, ChannelId channelId) = 0;

  /**
   * Called when the user's view of the chat room is changed.
   *
   * @param[in] userId The id of the user whose room view has been updated.
   * @param[in] roomOwnerId The id of the channel that owns the room.
   * @param[in] roomId The id of the room that has an updated view.
   * @param[in] roomViewInfo The new view information for the user and room.
   */
  virtual void RoomViewUpdated(
    UserId userId, ChannelId roomOwnerId, const std::string& roomId, ChatRoomView&& roomViewInfo) = 0;

  /**
   * Called when the user is mentioned in a chat room.
   *
   * @param[in] userId The id of the user who is being mentioned.
   * @param[in] mentionInfo Information about the mention message.
   */
  virtual void RoomMentionReceived(UserId userId, RoomMentionInfo&& mentionInfo) = 0;
};
