/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

namespace ttv {
namespace chat {
class IChatTransportReader;
class IChatTransport;
}  // namespace chat
}  // namespace ttv

/**
 * Interface implemented by components which read from a transport.
 */
class ttv::chat::IChatTransportReader {
 public:
  /**
   * Destructor.
   */
  virtual ~IChatTransportReader();
  /**
   * Inject information into the read when a connection occurs.
   */
  virtual void OnConnect(TTV_ErrorCode hr, const char* error) = 0;
  /**
   * Inject information into the read when an error occurs.
   */
  virtual void OnError(TTV_ErrorCode hr) = 0;
  /**
   * Inject information into the read when a disconnection occurs.
   */
  virtual void OnClose() = 0;
  /**
   * Read raw data from a transport.
   */
  virtual void OnRead(const char* pData, size_t nSize) = 0;
};

/**
 * Interface implemented by components which send and receive data.
 */
class ttv::chat::IChatTransport {
 public:
  static const uint32_t kMaxIrcTextLength = 2048;  //!< The max number of bytes allowed in the text area of the message.
  static const uint32_t kMaxIrcTagsLength =
    4096;  //!< The max number of bytes allowed in the IRCv3 tags area of the message.
  static const uint32_t kMaxBufferSize = kMaxIrcTextLength + kMaxIrcTagsLength;

 public:
  /**
   * Destructor.
   */
  virtual ~IChatTransport();
  virtual TTV_ErrorCode Connect(const std::string& uri) = 0;
  virtual TTV_ErrorCode Close() = 0;
  virtual bool IsOpen() = 0;

  /**
   * Sets the reader which will receive data from reads.
   */
  virtual void SetReader(std::shared_ptr<IChatTransportReader> reader) = 0;
  /**
   * Writes the given data to the transport.
   */
  virtual TTV_ErrorCode Write(const char* pData, size_t nSize) = 0;
  /**
   * Reads a single incoming event. Does not block.  This will make calls into the reader.
   *
   * @param handled Whether or not there was an event to handle.
   */
  virtual TTV_ErrorCode ProcessIncomingEvent(bool& handled) = 0;
};
