/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/component.h"
#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/lrucache.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/user/userrepository.h"

#include <map>
#include <memory>
#include <queue>
#include <vector>

namespace std {
// So we can cache on std::pair<UserId, ChannelId>
template <>
struct hash<std::pair<uint32_t, uint32_t>> {
  size_t operator()(const std::pair<uint32_t, uint32_t>& pair) const {
    return (hash<uint32_t>()(pair.first) ^ (hash<uint32_t>()(pair.second) << 1));
  }
};
}  // namespace std

namespace ttv {
class TaskRunner;

namespace chat {
class BitsConfigRepository;
class ChatGetBitsConfigTask;
class BitsConfiguration;
}  // namespace chat
}  // namespace ttv

class ttv::chat::BitsConfigRepository : public Component {
 public:
  using LookupCallback =
    std::function<void(TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& bitsConfiguration)>;
  using LookupId = uint64_t;

 public:
  BitsConfigRepository();

  /**
   * Returns global cached bits config or fetches if needed.
   */
  TTV_ErrorCode FetchGlobalBitsConfiguration(LookupCallback callback);
  /**
   * Returns channel cached bits config or fetches if needed.
   */
  TTV_ErrorCode FetchChannelBitsConfiguration(UserId userId, ChannelId channelId, LookupCallback callback);
  /**
   * Fetch the bits configuration with a cancellation token.
   *
   * @param[out] token LookupId of the bitsConfiguration fetch to (possibly) cancel later with CancelFetch.
   */
  TTV_ErrorCode FetchGlobalBitsConfiguration(LookupCallback callback, LookupId& token);
  TTV_ErrorCode FetchChannelBitsConfiguration(
    UserId userId, ChannelId channelId, LookupCallback callback, LookupId& token);
  /**
   * Cancel the fetching of the bits if the task to fetch the bitsconfioguration hasn't been added to the taskrunner
   * yet.
   *
   * @param[in] token LookupId of the bitsConfiguration fetch to cancel.
   */
  TTV_ErrorCode CancelFetch(LookupId token);

  // Component overrides
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override;

  void SetUserRepository(const std::shared_ptr<UserRepository>& repository) { mUserRepository = repository; }

 private:
  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  TTV_ErrorCode ScheduleLookup();
  TTV_ErrorCode CacheBitsConfig(const std::shared_ptr<BitsConfiguration> bitsConfiguration);

  LruCache<std::pair<UserId, ChannelId>, std::shared_ptr<BitsConfiguration>> mBitsConfigCache;
  std::map<std::pair<UserId, ChannelId>, std::shared_ptr<CallbackQueue<LookupCallback>>> mPendingBitsConfigLookups;
  std::map<LookupId, std::pair<UserId, ChannelId>> mChannelLookups;
  CallbackQueue<LookupCallback> mCancelledBitsConfigLookups;
  std::shared_ptr<UserRepository> mUserRepository;
  LookupId mCurrentLookupId;
  bool mLookupInProgress;
};
