/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
class UserRepository;

namespace chat {
class ChatAPITaskHost;
}
}  // namespace ttv

/**
 * This class is used to call API endpoints for ChatAPI functions.
 * A separate component is necessary to ensure we have properly finished running tasks when ChatAPI shuts down.
 */
class ttv::chat::ChatAPITaskHost : public Component {
 public:
  using BanUserCallback = std::function<void(TTV_ErrorCode ec, BanUserError&& error)>;
  using UnbanUserCallback = std::function<void(TTV_ErrorCode ec, UnbanUserError&& error)>;
  using ModUserCallback = std::function<void(TTV_ErrorCode ec, ModUserError&& error)>;
  using UnmodUserCallback = std::function<void(TTV_ErrorCode ec, UnmodUserError&& error)>;
  using GrantVIPCallback = std::function<void(TTV_ErrorCode ec, GrantVIPErrorCode error)>;
  using RevokeVIPCallback = std::function<void(TTV_ErrorCode ec, RevokeVIPErrorCode error)>;
  using FetchChannelVIPsCallback = std::function<void(TTV_ErrorCode ec, std::vector<std::string>&& vipNames)>;
  using UpdateUserColorCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchChannelModeratorsCallback =
    std::function<void(TTV_ErrorCode ec, std::vector<std::string>&& modNames, std::string&& nextCursor)>;
  using FetchBadgesCallback = std::function<void(TTV_ErrorCode ec, BadgeSet&& result)>;
  using FetchVodCommentSettingsCallback = std::function<void(TTV_ErrorCode ec, ChannelVodCommentSettings&& result)>;
  using SetVodCommentSettingsCallback = std::function<void(TTV_ErrorCode ec)>;

 public:
  ChatAPITaskHost();

  TTV_ErrorCode BanUser(UserId userId, ChannelId channelId, const std::string& bannedUserName, uint32_t duration,
    BanUserCallback&& callback);
  TTV_ErrorCode UnbanUser(
    UserId userId, ChannelId channelId, const std::string& unbannedUserName, UnbanUserCallback&& callback);
  TTV_ErrorCode ModUser(UserId userId, ChannelId channelId, const std::string& modUserName, ModUserCallback&& callback);
  TTV_ErrorCode UnmodUser(
    UserId userId, ChannelId channelId, const std::string& unmodUserName, UnmodUserCallback&& callback);
  TTV_ErrorCode GrantVIP(
    UserId userId, ChannelId channelId, const std::string& vipUserName, GrantVIPCallback&& callback);
  TTV_ErrorCode RevokeVIP(
    UserId userId, ChannelId channelId, const std::string& unvipUserName, RevokeVIPCallback&& callback);
  TTV_ErrorCode UpdateUserColor(UserId userId, const std::string& color, UpdateUserColorCallback&& callback);
  TTV_ErrorCode FetchChannelModerators(
    ChannelId channelId, const std::string& cursor, FetchChannelModeratorsCallback&& callback);
  TTV_ErrorCode FetchChannelVIPs(ChannelId channelId, FetchChannelVIPsCallback&& callback);

  TTV_ErrorCode FetchGlobalBadges(FetchBadgesCallback&& callback);
  TTV_ErrorCode FetchChannelBadges(ChannelId channelId, FetchBadgesCallback&& callback);

  TTV_ErrorCode FetchVodCommentSettings(UserId userId, ChannelId channelId, FetchVodCommentSettingsCallback&& callback);
  TTV_ErrorCode SetVodCommentFollowersOnlyDurationSeconds(
    UserId userId, ChannelId channelId, uint32_t durationSeconds, SetVodCommentSettingsCallback&& callback);
  TTV_ErrorCode SetVodCommentPublishingMode(
    UserId userId, ChannelId channelId, CommentPublishingMode mode, SetVodCommentSettingsCallback&& callback);

  void SetUserRepository(const std::shared_ptr<UserRepository>& repository) { mUserRepository = repository; }

  // Component overrides
  virtual std::string GetLoggerName() const override { return "ChatAPITaskHost"; }

 private:
  std::shared_ptr<UserRepository> mUserRepository;
};
