/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/
#pragma once

#include "twitchsdk/chat/chatlistener.h"
#include "twitchsdk/chat/internal/bitsconfigrepository.h"
#include "twitchsdk/chat/internal/chatconnection.h"
#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/task/chatpropertiestask.h"
#include "twitchsdk/core/channel/channelrepository.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/user/user.h"

namespace ttv {
class IThread;
class User;
class SettingRepository;

namespace chat {
class ChatAPI;
class ChatChannel;
class ChatSession;
class IChatTransport;
class ChatUserBlockList;
class BitsConfigRepository;
}  // namespace chat
}  // namespace ttv

/**
 * Provides the functionality for accessing a chat channel.
 */
class ttv::chat::ChatChannel : public ChatConnection::Listener {
 private:
  struct State {
    enum Enum {
      Initialized,
      Connecting,
      Connected,
      ShuttingDown,
      ShutDown,
    };
  };

 public:
  using FetchUserListCallback = std::function<void(TTV_ErrorCode ec, UserList&& userList)>;
  using LookupId = uint64_t;

  ChatChannel(const std::shared_ptr<User>& user, ChannelId channelId,
    std::shared_ptr<IChatChannelListener> chatCallbacks, std::shared_ptr<TaskRunner> taskRunner);
  virtual ~ChatChannel() override;

  void SetTokenizationOptions(const TokenizationOptions& options) { mTokenizationOptions = options; }
  void SetChannelRepository(const std::shared_ptr<ChannelRepository>& channelRepository) {
    mChannelRepository = channelRepository;
  }
  void SetBitsConfigRepository(const std::shared_ptr<BitsConfigRepository>& repository) {
    mBitsConfigRepository = repository;
  }
  /**
   * Sets the factory for creating custom implementation of chat components.
   */
  void SetChatObjectFactory(std::shared_ptr<IChatObjectFactory> factory);
  void SetSettingRepository(std::shared_ptr<SettingRepository> settings);

  /**
   * Connect to the channel.
   */
  TTV_ErrorCode Connect();
  /**
   * Disconnect from the channel.
   */
  TTV_ErrorCode Disconnect();
  /**
   * Flushes events to the main thread for the game to process.  This should only be called on the main thread.
   */
  TTV_ErrorCode FlushClientEvents();
  /**
   * Determines whether the channel is disconnected and ready to be destroyed.
   */
  bool IsShutdown() const { return mChatThread == nullptr; }
  /**
   * Sends a message from the current user.
   */
  TTV_ErrorCode SendChatMessage(const std::string& message);

  TTV_ErrorCode FetchUserList(const FetchUserListCallback& callback);

  /**
   * Exposes the remaining time till a user can send a message in slow mode for chat notice localization.
   */
  uint64_t GetRemainingSlowModeTime();

  /**
   * Retrieves the channel ID.
   */
  inline UserId GetChannelId() const { return mChannelId; }

  void SetConnectTrackingStartTime(uint64_t startMilliseconds) { mConnectTrackingStartTime = startMilliseconds; }

  void SetMessageFlushInterval(uint64_t milliseconds) { mUserMessageFlushInterval = milliseconds; }
  uint64_t GetMessageFlushInterval() const { return mUserMessageFlushInterval; }

  ChatChannelState GetState() const { return mReportedClientState; }

  std::shared_ptr<IChatChannelListener> GetChannelListener() { return mChatCallbacks; }
  std::shared_ptr<User> GetUser() { return mUser.lock(); }

  // ChatConnection::Listener implementation
  virtual void OnConnected(ChatConnection* source) override;
  virtual void OnConnectionFailed(ChatConnection* source, TTV_ErrorCode ec) override;
  virtual void OnConnectionLost(ChatConnection* source, TTV_ErrorCode ec) override;
  virtual void OnCleared(ChatConnection* source, const std::string& username,
    const std::map<std::string, std::string>& messageTags) override;
  virtual void OnPrivateMessageReceived(ChatConnection* source, const std::string& username, const std::string& message,
    const std::map<std::string, std::string>& messageTags, bool action) override;
  virtual void OnUserNoticeReceived(
    ChatConnection* source, const std::string& message, const std::map<std::string, std::string>& messageTags) override;
  virtual void OnUserStateChanged(
    ChatConnection* source, const std::map<std::string, std::string>& messageTags) override;
  virtual void OnChatRestrictionsChanged(ChatConnection* source, const ChatChannelRestrictions& restrictions) override;
  virtual void OnPermanentBanChanged(ChatConnection* source, bool banned) override;
  virtual void OnTemporaryBanChanged(ChatConnection* source, bool temporarilyBanned, uint32_t timeout) override;
  virtual void OnBadgesChanged(
    ChatConnection* source, const std::string& username, const std::string& badgesMessageTag) override;
  virtual void OnHostTargetChanged(
    ChatConnection* source, const std::string& targetChannel, uint32_t numViewers) override;
  virtual void OnNoticeReceived(
    ChatConnection* source, const std::string& id, const std::map<std::string, std::string>& params) override;
  virtual void OnIgnoreChanged(ChatConnection* source, const std::string& blockUserName, bool ignore) override;
  virtual void OnMessageDeleted(ChatConnection* source, std::string&& deletedMessageId, std::string&& senderLoginName,
    std::string&& deletedMessageContent) override;

 private:
  class PubSubTopicListener : public PubSubClient::ITopicListener {
   public:
    PubSubTopicListener(ChatChannel* owner);

    // PubSubClient::ITopicListener implementation
    virtual void OnTopicSubscribeStateChanged(PubSubClient* source, const std::string& topic,
      PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) override;
    virtual void OnTopicMessageReceived(
      PubSubClient* source, const std::string& topic, const json::Value& msg) override;
    virtual void OnTopicListenerRemoved(PubSubClient* source, const std::string& topic, TTV_ErrorCode ec) override;

   private:
    ChatChannel* mOwner;
  };

  TTV_ErrorCode SubscribeTopics();

  // PubSubClient::ITopicListener implementation
  void OnTopicSubscribeStateChanged(
    const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec);
  void OnTopicMessageReceived(const std::string& topic, const json::Value& msg);

  void FlushUserMessages(bool force);

  /**
   * This waits for the chat thread to close and blocks until it is safe to free the Channel instance.
   */
  void ForceShutdown();
  /**
   * Handle requests from the main thread in the chat thread.  This will create internal events and may
   * initiate messages to the chat server.  This should only be called on the chat thread.
   */
  TTV_ErrorCode ProcessClientRequestQueue();
  TTV_ErrorCode ProcessClientChatMessage(const std::string& msg);
  TTV_ErrorCode FetchChannelInfo();
  TTV_ErrorCode FetchChatProperties();
  TTV_ErrorCode FetchBitsConfiguration();
  TTV_ErrorCode ProcessDisconnectRequest();
  TTV_ErrorCode ProcessChannelInfoFetchResult(const ChannelInfo& channelInfo);
  TTV_ErrorCode ProcessChatPropertyFetchResult(TTV_ErrorCode ec, std::shared_ptr<ChatPropertiesTask::Result> result);
  TTV_ErrorCode ProcessBitsConfigFetchResult(TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config);
  void SetClientChannelInfo(const ChatChannelInfo& channelInfo);
  void SetClientChatRestrictions(const ChatChannelRestrictions& restrictions);

  void HandleMessageReceived(const std::string& username, const std::string& message,
    const std::map<std::string, std::string>& messageTags, const MessageInfo::Flags& flags);
  void HandleUserNotice(const std::string& message, const std::map<std::string, std::string>& messageTags);
  void HandleSubscriptionNotice(
    SubscriptionNotice::Type type, const std::string& message, const std::map<std::string, std::string>& messageTags);
  void HandleFirstTimeChatterNotice(const std::string& message, const std::map<std::string, std::string>& messageTags);
  void HandleRaidNotice(const std::string& message, const std::map<std::string, std::string>& messageTags);
  void HandleUnraidNotice(const std::string& message, const std::map<std::string, std::string>& messageTags);
  void HandleGenericNotice(const std::string& message, const std::map<std::string, std::string>& messageTags);
  void GenerateUserInfo(
    const std::string& username, const std::map<std::string, std::string>& messageTags, ChatUserInfo& userInfo);
  void GenerateMessage(MessageInfo& msg, const TokenizationOptions& tokenizationOptions, const ChatUserInfo& userInfo,
    const std::string& message, const std::map<std::string, std::string>& messageTags, const MessageInfo::Flags& flags);
  void GenerateLiveMessage(LiveChatMessage& msg, const TokenizationOptions& tokenizationOptions,
    const ChatUserInfo& userInfo, const std::string& message, const std::map<std::string, std::string>& messageTags,
    const MessageInfo::Flags& flags);
  void UpdateLocalUserInfo(const ChatUserInfo& newUserInfo);

  /**
   * The thread which listens for incoming messages from the chat server.
   */
  void ThreadProc();
  /**
   * Free the resources used by the chat thread.
   */
  void CompleteShutdown();
  bool UpdateRestriction();
  TTV_ErrorCode AttemptConnection();
  void CloseConnection();

  void SetState(State::Enum state);
  void HandleConnectionIssue(bool recoverableError);

  std::string mBroadcasterName;

  std::shared_ptr<IChatChannelListener>
    mChatCallbacks;  //!< Client-configured callbacks for chat notifications for this channel.
  std::shared_ptr<ChatPropertiesTask::Result> mChatProperties;  //!< The chat properties for the channel.
  std::vector<std::string> mHosts;                              //!< The hosts to use to connect to the channel

  std::shared_ptr<IChatObjectFactory> mChatObjectFactory;  //!< The internal chat object factory.
  std::shared_ptr<IThread> mChatThread;                    //!< The thread managing the socket.
  std::shared_ptr<ChatConnection> mChatConnection;         //!< The active connection.
  std::shared_ptr<ChannelRepository> mChannelRepository;
  std::shared_ptr<SettingRepository> mSettingRepository;
  std::shared_ptr<BitsConfigRepository> mBitsConfigRepository;
  std::shared_ptr<TaskRunner> mTaskRunner;
  std::shared_ptr<TaskRunner> mBackgroundTaskRunner;

  std::shared_ptr<PubSubClient> mPubSub;
  std::shared_ptr<PubSubTopicListener> mPubSubTopicListener;
  std::shared_ptr<PubSubTopicListenerHelper> mPubSubTopicListenerHelper;

  WaitForExpiry mFetchChatPropertiesTimer;  //!< The timer to use for fetching chat properties.
  WaitForExpiry mFetchBitsConfigTimer;      //!< The timer to use for fetching the channel's bit configuration
  RetryTimer mConnectionRetryTimer;         //!< The master timer to use for connection retries.
  uint32_t mNextChatHostIndex;              //!< The next chat host to connect to.

  using ThreadEvent = std::function<void()>;
  ConcurrentQueue<ThreadEvent> mToMainQ;  //!< The queue for messages from the chat thread to main thread.
  ConcurrentQueue<ThreadEvent> mToChatQ;  //!< The queue for messages from the main thread to chat thread.

  std::vector<LiveChatMessage>
    mUserMessageBatch;  //!< A batch of messages to be sent together to the client instead of one at a time (for performance).
  std::vector<LiveChatMessage>
    mDelayedMessageBatch;  //!< A batch of messages to be sent together, but are currently held due to a chat delay.
  std::unordered_set<UserId>
    mTimedOutUsersInDelayWindow;  //!< Used to remove messages from the messages to send when using a chat delay.

  uint64_t mConnectTrackingStartTime;  //!< For chat connect tracking purposes - the time ChatAPI::Connect() was called
  uint64_t mUserMessageFlushInterval;  //!< The max amount of time between user message flushes to the main thread.
  WaitForExpiry
    mNextUserMessageFlush;  //!< The next time in which to automatically flush all user messages to the main thread.
  ChatMessagePacer
    mMessagePacer;  //!< Prevents the local user from sending too many messages too close together in time and getting banned.
  WaitForExpiry
    mSlowModeTimer;  //!< When the channel is in slow mode this will mark the time when the next message is allowed.
  WaitForExpiry
    mTimeoutTimer;  //!< When the user has been banned for a temporary amount of time this will mark the time when the next message is allowed.

  TTV_ErrorCode mConnectionError;

  State::Enum mChannelState;              //!< The current state, accessed only through the chat thread.
  ChatChannelState mReportedClientState;  //!< The last reported state.

  ChatUserInfo mSystemUserInfo;        //!< The display information for messages from JTV or TwitchNotify.
  ChatUserInfo mClientLocalUserInfo;   //!< The info for the local user on the client thread.
  ChatUserInfo mServerLocalUserInfo;   //!< The info for the local user on the chat thread.
  ChatChannelInfo mClientChannelInfo;  //!< The current channel information as the client sees it.
  ChatChannelInfo mServerChannelInfo;  //!< The current channel information on the chat thread.
  ChatChannelRestrictions
    mClientChatRestrictions;  //!< The chat restrictions that are active for the channel as the client sees it.
  ChatChannelRestrictions
    mServerChatRestrictions;  //!< The chat restrictions that are active for the channel on the chat thread.

  std::weak_ptr<User> mUser;                     //!> The local user connecting to the channel.
  std::vector<std::string> mQueuedChatMessages;  // The list of messages that were sent from the client to the chat
                                                 // thread while a reconnect was pending.
  std::shared_ptr<BitsConfiguration> mBitsConfiguration;

  std::string mModeratorActionsPubSubTopic;  //!< The pubsub topic to listen to chat moderation actions.
  std::string mChatroomUserV1PubSubTopic;    //!< The pubsub topic to listen to user chat mod updates.

  TokenizationOptions mTokenizationOptions;  //!< Marks what to tokenize in a message
  LookupId mBitsConfigFetchToken;            //!< The cancellation token for fetching the BitsConfiguration.
  ChannelId mChannelId;                      //!< The ID of the channel.

  std::atomic<uint32_t> mNumOutstandingTasks;

  bool mApplySlowMode;
  bool mAnonymous;  //!< Whether or not connecting anonymously to the channel.
  bool
    mDisconnectionRequested;  //!< Whether or not the request to disconnect has been satisfied (callback fired to client).
};
