/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chatlistener.h"
#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/ichatchannel.h"
#include "twitchsdk/core/component.h"

#include <unordered_set>

#include <atomic>
#include <map>
#include <memory>
#include <string>
#include <vector>

namespace ttv {
class TaskRunner;
class User;
class SettingRepository;
class ChannelRepository;

namespace chat {
class ChatChannel;
class IChatObjectFactory;
class ChatChannelSet;
class ChatChannelWrapper;
class BitsConfigRepository;
}  // namespace chat
}  // namespace ttv

class ttv::chat::ChatChannelSet : public ttv::UserComponent {
 public:
  using FetchUserListCallback = std::function<void(TTV_ErrorCode ec, UserList&& userList)>;

  ChatChannelSet(const std::shared_ptr<User>& user);
  virtual ~ChatChannelSet() override;

  void SetTokenizationOptions(const TokenizationOptions& tokenizationOptions) {
    mTokenizationOptions = tokenizationOptions;
  }
  void SetChatObjectFactory(std::shared_ptr<IChatObjectFactory> factory) { mChatObjectFactory = factory; }
  void SetSettingRepository(std::shared_ptr<SettingRepository> settings) { mSettingRepository = settings; }
  void SetChannelRepository(const std::shared_ptr<ChannelRepository>& channelRepository) {
    mChannelRepository = channelRepository;
  }
  void SetBitsConfigRepository(std::shared_ptr<BitsConfigRepository> repository) { mBitsConfigRepository = repository; }

  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;

  TTV_ErrorCode Connect(ChannelId channelId, const std::shared_ptr<IChatChannelListener>& listener);
  TTV_ErrorCode Disconnect(ChannelId channelId);
  TTV_ErrorCode SendChatMessage(ChannelId channelId, const std::string& message);
  TTV_ErrorCode FetchUserList(ChannelId channelId, const FetchUserListCallback& callback);
  uint64_t GetRemainingSlowModeTime(ChannelId channelId);

  TTV_ErrorCode SetConnectTrackingStartTime(ChannelId channelId, uint64_t startMilliseconds);

  void SetMessageFlushInterval(uint64_t milliseconds);
  uint64_t GetMessageFlushInterval() const { return mUserMessageFlushInterval; }

  virtual std::string GetLoggerName() const override;
  static std::string GetComponentName() { return "ttv::chat::ChatChannelSet"; }

 protected:
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

 private:
  struct ChannelEntry {
    std::shared_ptr<ChatChannel> channel;
    std::shared_ptr<IChatChannelListener> channelListener;
  };

  class InternalChannelListener : public IChatChannelListener {
   public:
    InternalChannelListener(ChatChannelSet* owner);

    // IChatChannelListener implementation
    virtual void ChatChannelStateChanged(
      UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec) override;
    virtual void ChatChannelInfoChanged(
      UserId userId, ChannelId channelId, const ChatChannelInfo& channelInfo) override;
    virtual void ChatChannelRestrictionsChanged(
      UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions) override;
    virtual void ChatChannelLocalUserChanged(UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) override;
    virtual void ChatChannelMessagesReceived(
      UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) override;
    virtual void ChatChannelSubscriptionNoticeReceived(
      UserId userId, ChannelId channelId, const SubscriptionNotice& notice) override;
    virtual void ChatChannelFirstTimeChatterNoticeReceived(
      UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) override;
    virtual void ChatChannelRaidNoticeReceived(UserId userId, ChannelId channelId, const RaidNotice& notice) override;
    virtual void ChatChannelUnraidNoticeReceived(
      UserId userId, ChannelId channelId, const UnraidNotice& notice) override;
    virtual void ChatChannelGenericNoticeReceived(
      UserId userId, ChannelId channelId, const GenericMessageNotice& notice) override;
    virtual void ChatChannelMessagesCleared(UserId userId, ChannelId channelId) override;
    virtual void ChatChannelUserMessagesCleared(UserId userId, ChannelId channelId, UserId clearUserId) override;
    virtual void ChatChannelMessageDeleted(UserId userId, ChannelId channelId, std::string&& messageId,
      std::string&& senderLoginName, std::string&& messageContent) override;
    virtual void ChatChannelHostTargetChanged(
      UserId userId, ChannelId channelId, const std::string& targetChannelName, uint32_t numViewers) override;
    virtual void ChatChannelNoticeReceived(UserId userId, ChannelId channelId, const std::string& id,
      const std::map<std::string, std::string>& params) override;
    virtual void AutoModCaughtSentMessage(UserId userId, ChannelId channelId) override;
    virtual void AutoModDeniedSentMessage(UserId userId, ChannelId channelId) override;
    virtual void AutoModApprovedSentMessage(UserId userId, ChannelId channelId) override;
    virtual void AutoModCaughtMessageForMods(UserId userId, ChannelId channelId, std::string&& messageId,
      std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) override;
    virtual void AutoModMessageApprovedByMod(UserId userId, ChannelId channelId, std::string&& messageId,
      UserId moderatorId, std::string&& moderatorName) override;
    virtual void AutoModMessageDeniedByMod(UserId userId, ChannelId channelId, std::string&& messageId,
      UserId moderatorId, std::string&& moderatorName) override;
    virtual void AutoModDeniedSentCheer(UserId userId, ChannelId channelId) override;
    virtual void AutoModTimedOutSentCheer(UserId userId, ChannelId channelId) override;
    virtual void AutoModCaughtCheerForMods(UserId userId, ChannelId channelId, std::string&& messageId,
      std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) override;
    virtual void ChatChannelModNoticeUserTimedOut(UserId userId, ChannelId channelId,
      ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) override;
    virtual void ChatChannelModNoticeUserBanned(
      UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason) override;
    virtual void ChatChannelModNoticeUserUntimedOut(
      UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) override;
    virtual void ChatChannelModNoticeUserUnbanned(
      UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) override;
    virtual void ChatChannelModNoticeMessageDeleted(UserId userId, ChannelId channelId,
      ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) override;
    virtual void ChatChannelModNoticeClearChat(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;
    virtual void ChatChannelModNoticeEmoteOnly(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;
    virtual void ChatChannelModNoticeEmoteOnlyOff(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;
    virtual void ChatChannelModNoticeFollowersOnly(UserId userId, ChannelId channelId, UserId modId,
      std::string&& modName, uint32_t minimumFollowingDurationMinutes) override;
    virtual void ChatChannelModNoticeFollowersOnlyOff(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;
    virtual void ChatChannelModNoticeR9K(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;
    virtual void ChatChannelModNoticeR9KOff(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;
    virtual void ChatChannelModNoticeSlow(UserId userId, ChannelId channelId, UserId modId, std::string&& modName,
      uint32_t slowModeDurationSeconds) override;
    virtual void ChatChannelModNoticeSlowOff(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;
    virtual void ChatChannelModNoticeSubsOnly(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;
    virtual void ChatChannelModNoticeSubsOnlyOff(
      UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override;

   private:
    ChatChannelSet* mOwner;
  };

  // Private IChatChannelListener implementation
  void ChatChannelStateChanged(UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec);
  void ChatChannelInfoChanged(UserId userId, ChannelId channelId, const ChatChannelInfo& channelInfo);
  void ChatChannelRestrictionsChanged(UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions);
  void ChatChannelLocalUserChanged(UserId userId, ChannelId channelId, const ChatUserInfo& userInfo);
  void ChatChannelMessagesReceived(UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList);
  void ChatChannelSubscriptionNoticeReceived(UserId userId, ChannelId channelId, const SubscriptionNotice& notice);
  void ChatChannelFirstTimeChatterNoticeReceived(
    UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice);
  void ChatChannelRaidNoticeReceived(UserId userId, ChannelId channelId, const RaidNotice& notice);
  void ChatChannelUnraidNoticeReceived(UserId userId, ChannelId channelId, const UnraidNotice& notice);
  void ChatChannelGenericNoticeReceived(UserId userId, ChannelId channelId, const GenericMessageNotice& notice);
  void ChatChannelMessagesCleared(UserId userId, ChannelId channelId);
  void ChatChannelUserMessagesCleared(UserId userId, ChannelId channelId, UserId clearUserId);
  void ChatChannelMessageDeleted(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& senderLoginName, std::string&& messageContent);
  void ChatChannelHostTargetChanged(
    UserId userId, ChannelId channelId, const std::string& targetChannelName, uint32_t numViewers);
  void ChatChannelNoticeReceived(
    UserId userId, ChannelId channelId, const std::string& id, const std::map<std::string, std::string>& params);
  void AutoModCaughtSentMessage(UserId userId, ChannelId channelId);
  void AutoModDeniedSentMessage(UserId userId, ChannelId channelId);
  void AutoModApprovedSentMessage(UserId userId, ChannelId channelId);
  void AutoModCaughtMessageForMods(UserId userId, ChannelId channelId, std::string&& messageId, std::string&& message,
    UserId senderId, std::string&& senderName, std::string&& reason);
  void AutoModMessageApprovedByMod(
    UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName);
  void AutoModMessageDeniedByMod(
    UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName);
  void AutoModDeniedSentCheer(UserId userId, ChannelId channelId);
  void AutoModTimedOutSentCheer(UserId userId, ChannelId channelId);
  void AutoModCaughtCheerForMods(UserId userId, ChannelId channelId, std::string&& messageId, std::string&& message,
    UserId senderId, std::string&& senderName, std::string&& reason);
  void ChatChannelModNoticeUserTimedOut(UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo,
    uint32_t timeoutDurationSeconds, std::string&& reason);
  void ChatChannelModNoticeUserBanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason);
  void ChatChannelModNoticeUserUntimedOut(UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo);
  void ChatChannelModNoticeMessageDeleted(UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo,
    std::string&& messageId, std::string&& message);
  void ChatChannelModNoticeUserUnbanned(UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo);
  void ChatChannelModNoticeClearChat(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);
  void ChatChannelModNoticeEmoteOnly(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);
  void ChatChannelModNoticeEmoteOnlyOff(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);
  void ChatChannelModNoticeFollowersOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t minimumFollowingDurationMinutes);
  void ChatChannelModNoticeFollowersOnlyOff(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);
  void ChatChannelModNoticeR9K(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);
  void ChatChannelModNoticeR9KOff(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);
  void ChatChannelModNoticeSlow(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t slowModeDurationSeconds);
  void ChatChannelModNoticeSlowOff(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);
  void ChatChannelModNoticeSubsOnly(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);
  void ChatChannelModNoticeSubsOnlyOff(UserId userId, ChannelId channelId, UserId modId, std::string&& modName);

  TTV_ErrorCode LookupChannel(ChannelId channelId, std::shared_ptr<ChatChannel>& channel);
  TTV_ErrorCode FlushChannelEvents();
  std::shared_ptr<ChannelEntry> CreateChannel(ChannelId channelId);

  std::shared_ptr<IChatObjectFactory> mChatObjectFactory;          //!> The internal chat object factory.
  std::shared_ptr<IChatChannelListener> mInternalChannelListener;  //!> The listener for channel events.
  std::shared_ptr<SettingRepository> mSettingRepository;
  std::shared_ptr<ChannelRepository> mChannelRepository;
  std::shared_ptr<BitsConfigRepository> mBitsConfigRepository;
  std::map<ChannelId, std::shared_ptr<ChannelEntry>> mChannels;  //!< The mapping of lower case channel name to channel.

  TokenizationOptions mTokenizationOptions;

  uint64_t mUserMessageFlushInterval;  //!< The max amount of time between user message flushes to the main thread.
};

class ttv::chat::ChatChannelWrapper : public IChatChannel {
 public:
  using DisposerFunc = std::function<void()>;

 public:
  ChatChannelWrapper(
    const std::shared_ptr<User>& user, ChannelId channelId, const std::shared_ptr<IChatChannelListener>& listener);

  void SetDisposer(DisposerFunc&& func) { mDisposerFunc = func; }
  std::shared_ptr<ChatChannelSet> GetChatChannelSet() const { return mChatChannelSet; }

  // IChatChannel implementation
  virtual void Dispose() override;
  virtual TTV_ErrorCode Connect() override;
  virtual TTV_ErrorCode Disconnect() override;
  virtual TTV_ErrorCode SendMessage(const std::string& message) override;
  virtual TTV_ErrorCode FetchUserList(const FetchUserListCallback& callback) override;
  virtual uint64_t GetRemainingSlowModeTime() override;

 private:
  DisposerFunc mDisposerFunc;
  std::shared_ptr<ChatChannelSet> mChatChannelSet;
  std::shared_ptr<IChatChannelListener> mChatChannelListener;
  ChannelId mChannelId;
};
