/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chatlistener.h"
#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/ichatcommentmanager.h"
#include "twitchsdk/chat/internal/bitsconfigrepository.h"
#include "twitchsdk/chat/internal/task/chatgetvodcommentstask.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/timer.h"

#include <atomic>
#include <memory>

namespace ttv {
class ChannelRepository;
class TaskRunner;
class UserRepository;
class SettingRepository;
class PagedRequestFetcher;

namespace chat {
class ChatCommentManager;
class BitsConfigRepository;
}  // namespace chat
}  // namespace ttv

class ttv::chat::ChatCommentManager : public Component, public IChatCommentManager {
 public:
  using DisposerFunc = std::function<void()>;

 public:
  ChatCommentManager(const std::shared_ptr<User>& user, const std::string& vodId);

  void SetDisposer(DisposerFunc&& func) { mDisposerFunc = func; }

  // Call these setters + SetTaskRunner before Initialize()
  void SetTokenizationOptions(const TokenizationOptions& tokenizationOptions) {
    mTokenizationOptions = tokenizationOptions;
  }
  void SetListener(const std::shared_ptr<IChatCommentListener>& listener) { mListener = listener; }
  void SetChannelRepository(const std::shared_ptr<ChannelRepository>& repository) { mChannelRepository = repository; }
  void SetBitsConfigRepository(const std::shared_ptr<BitsConfigRepository>& repository) {
    mBitsConfigRepository = repository;
  }

  std::string GetVodId() const { return mVodId; }

  // Component Overrides
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual bool CheckShutdown() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override;

  // IChatCommentManager Implementation
  virtual TTV_ErrorCode Dispose() override;
  virtual TTV_ErrorCode Play() override;
  virtual TTV_ErrorCode Pause() override;
  virtual TTV_ErrorCode UpdatePlayhead(uint64_t timestampMilliseconds) override;
  virtual Result<uint64_t> GetPlayheadTime() const override;
  virtual Result<ChannelId> GetChannelId() const override;
  virtual Result<PlayingState> GetPlayingState() const override;

  virtual TTV_ErrorCode FetchComments(
    uint64_t timestampMilliseconds, uint32_t limit, const FetchCommentsCallback& callback) override;
  virtual TTV_ErrorCode FetchComments(
    const std::string& cursor, uint32_t limit, const FetchCommentsCallback& callback) override;
  virtual TTV_ErrorCode FetchComment(const std::string& commentId, const FetchCommentCallback& callback) override;
  virtual TTV_ErrorCode DeleteComment(const std::string& commentId, const DeleteCommentCallback& callback) override;
  virtual TTV_ErrorCode PostComment(
    const std::string& message, uint64_t timestampMilliseconds, const PostCommentCallback& callback) override;
  virtual TTV_ErrorCode ReportComment(const std::string& parentCommentId, const std::string& reason,
    const std::string& description, const ReportCommentCallback& callback) override;

  virtual TTV_ErrorCode FetchCommentReplies(
    const std::string& commentId, const FetchCommentRepliesCallback& callback) override;
  virtual TTV_ErrorCode PostCommentReply(
    const std::string& commentId, const std::string& message, const PostCommentReplyCallback& callback) override;

 private:
  enum class CommentsState {
    Idle,           //!< Startup state
    FillingBuffer,  //!< The chat comment buffer is being populated
    MoreToFetch,    //!< There are still comments later in the VOD to fetch.
    DoneFetching,   //!< We've fetched all the comments that there are for the VOD.
    Finished,       //!< We've fetched all comments and have sent them all to the listener, nothing left to do.
  };

  static std::string CommentsStateString(CommentsState state) {
    switch (state) {
      case CommentsState::Finished:
        return "Finished";
      case CommentsState::DoneFetching:
        return "DoneFetching";
      case CommentsState::MoreToFetch:
        return "MoreToFetch";
      case CommentsState::Idle:
        return "Idle";
      case CommentsState::FillingBuffer:
        return "FillingBuffer";
      default:
        TTV_ASSERT(false);
        return "Unknown";
    }
  }

  TTV_ErrorCode FetchVod();
  TTV_ErrorCode FetchBitsConfig();
  TTV_ErrorCode FetchPlayheadComments();

  /**
   * Returns true if the given position has buffered comments available
   **/
  bool IsPositionInsideCommentBuffer(uint64_t playheadPos);

  /**
   * How many seconds worth of comments we have fetched
   **/
  int64_t CommentBufferSizeMs();

  void Advance();
  void SetPlayingState(PlayingState state);
  void SetCommentsState(CommentsState state);

  std::shared_ptr<IChatCommentListener> mListener;
  std::vector<ChatGetVodCommentsTask::Result::CommentsBatch> mFetchedLists;
  std::shared_ptr<User> mUser;
  std::shared_ptr<ChannelRepository> mChannelRepository;
  std::shared_ptr<BitsConfigRepository> mBitsConfigRepository;
  std::shared_ptr<BitsConfiguration> mBitsConfiguration;

  std::string mVodId;
  std::string mNextCursorUrl;

  RetryTimer mFetchVodRetryTimer;
  RetryTimer mFetchBitsConfigRetryTimer;
  WaitForExpiry mFetchMessagesRetryTimer;

  DisposerFunc mDisposerFunc;

  TokenizationOptions mTokenizationOptions;

  uint64_t mPlayheadTimeMilliseconds;  //!< Current play timestamp in the VOD.
  uint64_t mStartOfCommentBuffer;      //!< Play timestamp that marks the start of the comment buffer in memory
  BitsConfigRepository::LookupId mBitsConfigFetchToken;  //!< The cancellation token for fetching the BitsConfiguration.

  uint32_t mClearCount;
  UserId mUserId;
  ChannelId mChannelId;
  PlayingState mPlayingState;
  CommentsState mCommentsState;

  bool mVodFetchInFlight;
  bool mCommentsFetchInFlight;
  bool mHasFetchedVod;
  bool mHasFetchedBitsConfig;
};
