/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/chatnetworkevent.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
class User;

namespace chat {
class ChatConnection;
class ChatConnectionListenerProxy;

class ChatAPI;
class ChatChannel;
class ChatSession;
class ChatReader;
class ChatWriter;
class IChatTransport;
class IChatObjectFactory;
}  // namespace chat
}  // namespace ttv

/**
 * Wraps an instance of an IRC connection.
 */
class ttv::chat::ChatConnection : public IChatReceiveNetworkEvent {
 public:
  class Listener {
   public:
    virtual ~Listener() = default;
    virtual void OnConnected(ChatConnection* source) = 0;
    virtual void OnConnectionFailed(ChatConnection* source, TTV_ErrorCode ec) = 0;
    virtual void OnConnectionLost(ChatConnection* source, TTV_ErrorCode ec) = 0;
    virtual void OnCleared(
      ChatConnection* source, const std::string& username, const std::map<std::string, std::string>& messageTags) = 0;
    virtual void OnPrivateMessageReceived(ChatConnection* source, const std::string& username,
      const std::string& message, const std::map<std::string, std::string>& messageTags, bool action) = 0;
    virtual void OnUserNoticeReceived(
      ChatConnection* source, const std::string& message, const std::map<std::string, std::string>& messageTags) = 0;
    virtual void OnUserStateChanged(ChatConnection* source, const std::map<std::string, std::string>& messageTags) = 0;
    virtual void OnChatRestrictionsChanged(ChatConnection* source, const ChatChannelRestrictions& restrictions) = 0;
    virtual void OnPermanentBanChanged(ChatConnection* source, bool banned) = 0;
    virtual void OnTemporaryBanChanged(ChatConnection* source, bool temporarilyBanned, uint32_t timeout) = 0;
    virtual void OnBadgesChanged(
      ChatConnection* source, const std::string& username, const std::string& badgesMessageTag) = 0;
    virtual void OnHostTargetChanged(ChatConnection* source, const std::string& targetChannel, uint32_t numViewers) = 0;
    virtual void OnNoticeReceived(
      ChatConnection* source, const std::string& id, const std::map<std::string, std::string>& params) = 0;
    virtual void OnIgnoreChanged(ChatConnection* source, const std::string& blockUserName, bool ignore) = 0;
    virtual void OnMessageDeleted(ChatConnection* source, std::string&& deletedMessageId, std::string&& senderLoginName,
      std::string&& deletedMessageContent) = 0;
  };

 private:
  enum ConnectionState {
    CONNECTIONSTATE_DISCONNECTED,
    CONNECTIONSTATE_CONNECTING,
    CONNECTIONSTATE_WELCOMING,
    CONNECTIONSTATE_WELCOMED,
    CONNECTIONSTATE_JOINING,
    CONNECTIONSTATE_CONNECTED,
    CONNECTIONSTATE_CONNECTION_FAILED,
    CONNECTIONSTATE_CONNECTION_LOST,
  };

 public:
  ChatConnection(ChannelId channelId, const std::shared_ptr<User>& user);
  virtual ~ChatConnection() override;

  void SetListener(Listener* listener) { mListener = listener; }
  void SetChatObjectFactory(std::shared_ptr<IChatObjectFactory> factory);
  void SetChannelName(const std::string& channelName);

  /**
   * Connect to the channel.
   */
  TTV_ErrorCode Connect(const std::string& uri);
  /**
   * Disconnect from the channel.
   */
  TTV_ErrorCode Disconnect();
  /**
   * Sends a message from the current user.
   */
  TTV_ErrorCode SendChatMessage(const std::string& message, const ChatUserInfo& localUserInfo);

  TTV_ErrorCode Update();

  std::string GetLocalUserName() const { return mUserName; }

  // IChatReceiveNetworkEvent implementation
  virtual void ReceiveEvent(const ChatNetworkEvent& evt) override;

  static TTV_ErrorCode CheckFactoryAvailability(const std::string& uri);

 private:
  void SetState(ConnectionState state);

  void Join();

  void HandleCapMessage(const ChatNetworkEvent& evt);
  /**
   * Handles PRIVMSG.
   */
  void HandlePrivateMessage(const ChatNetworkEvent& evt);
  void HandleUserState(const ChatNetworkEvent& evt);
  void HandleNotice(const ChatNetworkEvent& evt);
  void HandleClearChatMessage(const ChatNetworkEvent& evt);
  void HandleHostTargetMessage(const ChatNetworkEvent& evt);
  void HandleUserNotice(const ChatNetworkEvent& evt);
  void HandleDeleteChatMessage(const ChatNetworkEvent& evt);

  void HandleMessageTags(const ChatNetworkEvent& evt);
  void HandleRoomState(const ChatNetworkEvent& evt);

  void CreateObjects();
  void ReleaseObjects();

  std::string CreateAnonymousUserName();

  std::shared_ptr<ChatReader> mReader;                     //!< The socket reader.
  std::shared_ptr<ChatWriter> mWriter;                     //!< The socket writer.
  std::shared_ptr<ChatSession> mSession;                   //!< The message interpreter.
  std::shared_ptr<IChatObjectFactory> mChatObjectFactory;  //!< The internal chat object factory

  std::shared_ptr<IChatTransport> mTransport;  //!< The transport for sending and receiving chat messages.

  WaitForEventWithTimeout mWelcomeMessageTimeout;
  WaitForEventWithTimeout mJoinAttemptTimeout;

  std::string mIrcChannelName;
  std::string mUserName;  //!< The cached username.

  ConnectionState mConnectionState;
  Listener* mListener;
  TTV_ErrorCode mConnectionError;

  std::shared_ptr<User> mUser;

  ChannelId mChannelId;
  ChatChannelRestrictions mChatRestrictions;  //!< Stores the state of chat restrictions received by ROOMSTATE

  bool mAnonymous;                  //!< Whether or not connecting anonymously to the channel.
  bool mDisconnectionRequested;     //!< Whether or not expecting a disconnect.
  bool mFireConnectionErrorEvents;  //!< Whether or not to fire a connection error event.
};

/**
 * A lambda proxy for ChatConnection::Listener.
 */
class ttv::chat::ChatConnectionListenerProxy : public ChatConnection::Listener {
 public:
  using OnConnectedFunc = std::function<void(ChatConnection* source)>;
  using OnConnectionFailedFunc = std::function<void(ChatConnection* source, TTV_ErrorCode ec)>;
  using OnConnectionLostFunc = std::function<void(ChatConnection* source, TTV_ErrorCode ec)>;
  using OnClearedFunc = std::function<void(
    ChatConnection* source, const std::string& username, const std::map<std::string, std::string>& messageTags)>;
  using OnPrivateMessageReceivedFunc = std::function<void(ChatConnection* source, const std::string& username,
    const std::string& message, const std::map<std::string, std::string>& messageTags, bool action)>;
  using OnUserNoticeReceivedFunc = std::function<void(
    ChatConnection* source, const std::string& message, const std::map<std::string, std::string>& messageTags)>;
  using OnUserStateChangedFunc =
    std::function<void(ChatConnection* source, const std::map<std::string, std::string>& messageTags)>;
  using OnWhisperReceivedFunc = std::function<void(ChatConnection* source, const std::string& username,
    const std::string& message, const std::map<std::string, std::string>& messageTags, bool action)>;
  using OnWhisperSentFunc = std::function<void(ChatConnection* source, const std::string& receivingUserName,
    const std::string& message, const std::map<std::string, std::string>& messageTags, bool action)>;
  using OnChatRestrictionsChangedFunc = std::function<void(ChatConnection* source, ChatChannelRestrictions)>;
  using OnPermanentBanChangedFunc = std::function<void(ChatConnection* source, bool banned)>;
  using OnTemporaryBanChangedFunc =
    std::function<void(ChatConnection* source, bool temporarilyBanned, uint32_t timeout)>;
  using OnEmoticonSetsChangedFunc = std::function<void(
    ChatConnection* source, const std::string& username, const std::vector<std::string>& emoticonSets)>;
  using OnBadgesChangedFunc =
    std::function<void(ChatConnection* source, const std::string& username, const std::string& badgesMessageTag)>;
  using OnHostTargetChangedFunc =
    std::function<void(ChatConnection* source, const std::string& targetChannel, uint32_t numViewers)>;
  using OnNoticeReceivedFunc = std::function<void(
    ChatConnection* source, const std::string& id, const std::map<std::string, std::string>& params)>;
  using OnIgnoreChangedFunc =
    std::function<void(ChatConnection* source, const std::string& blockUserName, bool ignore)>;
  using OnMessageDeletedFunc = std::function<void(ChatConnection* source, std::string&& deletedMessageId,
    std::string&& senderLoginName, std::string&& deletedMessageContent)>;

 public:
  virtual void OnConnected(ChatConnection* source) override {
    if (mOnConnectedFunc != nullptr) {
      mOnConnectedFunc(source);
    }
  }

  virtual void OnConnectionFailed(ChatConnection* source, TTV_ErrorCode ec) override {
    if (mOnConnectionFailedFunc != nullptr) {
      mOnConnectionFailedFunc(source, ec);
    }
  }

  virtual void OnConnectionLost(ChatConnection* source, TTV_ErrorCode ec) override {
    if (mOnConnectionLostFunc != nullptr) {
      mOnConnectionLostFunc(source, ec);
    }
  }

  virtual void OnCleared(ChatConnection* source, const std::string& username,
    const std::map<std::string, std::string>& messageTags) override {
    if (mOnClearedFunc != nullptr) {
      mOnClearedFunc(source, username, messageTags);
    }
  }

  virtual void OnPrivateMessageReceived(ChatConnection* source, const std::string& username, const std::string& message,
    const std::map<std::string, std::string>& messageTags, bool action) override {
    if (mOnPrivateMessageReceivedFunc != nullptr) {
      mOnPrivateMessageReceivedFunc(source, username, message, messageTags, action);
    }
  }

  virtual void OnUserNoticeReceived(ChatConnection* source, const std::string& message,
    const std::map<std::string, std::string>& messageTags) override {
    if (mOnUserNoticeReceivedFunc != nullptr) {
      mOnUserNoticeReceivedFunc(source, message, messageTags);
    }
  }

  virtual void OnUserStateChanged(
    ChatConnection* source, const std::map<std::string, std::string>& messageTags) override {
    if (mOnUserStateChangedFunc != nullptr) {
      mOnUserStateChangedFunc(source, messageTags);
    }
  }

  virtual void OnChatRestrictionsChanged(ChatConnection* source, const ChatChannelRestrictions& restrictions) override {
    if (mOnChatRestrictionsChangedFunc != nullptr) {
      mOnChatRestrictionsChangedFunc(source, restrictions);
    }
  }

  virtual void OnPermanentBanChanged(ChatConnection* source, bool banned) override {
    if (mOnPermanentBanChangedFunc != nullptr) {
      mOnPermanentBanChangedFunc(source, banned);
    }
  }

  virtual void OnTemporaryBanChanged(ChatConnection* source, bool temporarilyBanned, uint32_t timeout) override {
    if (mOnTemporaryBanChangedFunc != nullptr) {
      mOnTemporaryBanChangedFunc(source, temporarilyBanned, timeout);
    }
  }

  virtual void OnBadgesChanged(
    ChatConnection* source, const std::string& username, const std::string& badgesMessageTag) override {
    if (mOnBadgesChangedFunc != nullptr) {
      mOnBadgesChangedFunc(source, username, badgesMessageTag);
    }
  }

  virtual void OnHostTargetChanged(
    ChatConnection* source, const std::string& targetChannel, uint32_t numViewers) override {
    if (mOnHostTargetChangedFunc != nullptr) {
      mOnHostTargetChangedFunc(source, targetChannel, numViewers);
    }
  }

  virtual void OnNoticeReceived(
    ChatConnection* source, const std::string& id, const std::map<std::string, std::string>& params) override {
    if (mOnNoticeReceivedFunc != nullptr) {
      mOnNoticeReceivedFunc(source, id, params);
    }
  }

  virtual void OnIgnoreChanged(ChatConnection* source, const std::string& blockUserName, bool ignore) override {
    if (mOnIgnoreChangedFunc != nullptr) {
      mOnIgnoreChangedFunc(source, blockUserName, ignore);
    }
  }

  virtual void OnMessageDeleted(ChatConnection* source, std::string&& deletedMessageId, std::string&& senderLoginName,
    std::string&& deletedMessageContent) override {
    if (mOnMessageDeletedFunc != nullptr) {
      mOnMessageDeletedFunc(
        source, std::move(deletedMessageId), std::move(senderLoginName), std::move(deletedMessageContent));
    }
  }

 public:
  OnConnectedFunc mOnConnectedFunc;
  OnConnectionFailedFunc mOnConnectionFailedFunc;
  OnConnectionLostFunc mOnConnectionLostFunc;
  OnClearedFunc mOnClearedFunc;
  OnPrivateMessageReceivedFunc mOnPrivateMessageReceivedFunc;
  OnUserNoticeReceivedFunc mOnUserNoticeReceivedFunc;
  OnUserStateChangedFunc mOnUserStateChangedFunc;
  OnWhisperReceivedFunc mOnWhisperReceivedFunc;
  OnWhisperSentFunc mOnWhisperSentFunc;
  OnChatRestrictionsChangedFunc mOnChatRestrictionsChangedFunc;
  OnPermanentBanChangedFunc mOnPermanentBanChangedFunc;
  OnTemporaryBanChangedFunc mOnTemporaryBanChangedFunc;
  OnBadgesChangedFunc mOnBadgesChangedFunc;
  OnHostTargetChangedFunc mOnHostTargetChangedFunc;
  OnNoticeReceivedFunc mOnNoticeReceivedFunc;
  OnIgnoreChangedFunc mOnIgnoreChangedFunc;
  OnMessageDeletedFunc mOnMessageDeletedFunc;
};
