/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/tokenranges.h"
#include "twitchsdk/core/user/user.h"

#include <regex>
#include <string>
#include <vector>

namespace ttv {
namespace chat {

/**
 * Parses the 'emotes' IRC message tag into the ranges needed to parse the associated message.
 */
bool ParseEmotesMessageTag(const std::string& emotesMessageTag, std::map<std::string, std::vector<EmoteRange>>& result);

/**
 * Parses the 'badges' IRC message tag into the list of badges in the message.
 * The pairs contain badgeSet name and version name, respectively.
 */
bool ParseBadgesMessageTag(const std::string& messageTag, std::vector<std::pair<std::string, std::string>>& result);

/**
 * Parses the 'flags' IRC message tag into a vector of ranges of automod flag levels.
 */
bool ParseAutoModFlagsMessageTag(const std::string& messageTag, std::vector<AutoModFlagsRange>& result);

/**
 * Creates a new IRC message tag string from the passed-in list of badges.
 * The pairs should contain the badgeSet name and version name, respectively.
 */
bool GenerateBadgesMessageTag(const std::vector<std::pair<std::string, std::string>>& badges, std::string& result);

bool TokenizeLocalMessage(const std::shared_ptr<User>& user, ChannelId channelId, const std::string& message,
  std::string& emotesMessageTag, std::string& badgesMessageTag);

void TokenizeServerMessage(const std::string& message, const TokenizationOptions& tokenizationOptions,
  const std::string& emotesMessageTag, const std::shared_ptr<BitsConfiguration>& bitsConfig,
  const std::vector<std::string>& localUserNames, MessageInfo& tokenizedMessage);

void TokenizeServerMessage(const std::string& message, const TokenizationOptions& tokenizationOptions,
  const std::string& emotesMessageTag, const std::string& autoModFlagsMessageTag,
  const std::shared_ptr<BitsConfiguration>& bitsConfig, const std::vector<std::string>& localUserNames,
  MessageInfo& tokenizedMessage);

void TokenizeServerMessage(const std::string& message, const TokenizationOptions& tokenizationOptions,
  const std::map<std::string, std::vector<EmoteRange>>& emoticonRanges, const std::vector<std::string>& localUserNames,
  MessageInfo& tokenizedMessage);

void TokenizeServerMessage(const std::string& message, const TokenizationOptions& tokenizationOptions,
  const std::map<std::string, std::vector<EmoteRange>>& emoticonRanges,
  const std::shared_ptr<BitsConfiguration>& bitsConfig, const std::vector<std::string>& localUserNames,
  MessageInfo& tokenizedMessage);

void TokenizeServerMessage(const std::string& message, const TokenizationOptions& tokenizationOptions,
  const std::map<std::string, std::vector<EmoteRange>>& emoticonRanges,
  const std::vector<AutoModFlagsRange>& autoModFlagsRanges, const std::shared_ptr<BitsConfiguration>& bitsConfig,
  const std::vector<std::string>& localUserNames, std::vector<std::unique_ptr<MessageToken>>& tokens);
}  // namespace chat
}  // namespace ttv
