/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#ifndef TTVSDK_CHATNETWORKEVENT_H
#define TTVSDK_CHATNETWORKEVENT_H

#include "twitchsdk/chat/chattypes.h"

namespace ttv {
namespace chat {
class ChatNetworkEvent;
class IChatReceiveNetworkEvent;
class IChatWriteNetworkEvent;
}  // namespace chat
}  // namespace ttv

/**
 * Wraps an event in the chat system.  Events include sending/receiving messages, pings, etc.
 */
class ttv::chat::ChatNetworkEvent {
 public:
  static std::string EventIDToString(int idEvent);
  static int EventStringToID(const std::string& evt);
  static int CTCPEventStringToID(const std::string& evt, bool bReply = false);

  static bool IsReplyEvent(int idEvent);
  static bool IsCommandEvent(int idEvent);
  static bool IsCTCPEvent(int idEvent);
  static bool IsCTCPReplyEvent(int idEvent);
  static bool IsSystemEvent(int idEvent);

  ChatNetworkEvent();
  ChatNetworkEvent(int idEvent, int nParams, ...);
  ~ChatNetworkEvent();

  void SetEvent(const std::string& evt);
  const std::string& GetEvent() const;

  void SetEventID(int idEvent);
  int GetEventID() const;

  void AddParam(const std::string& sParam);
  const std::string& GetParam(unsigned int iParam) const;
  size_t GetParamCount() const;

  void SetPrefix(const std::string& prefix);
  const std::string& GetPrefix() const;

  void SetMessageTags(const std::string& raw);
  const std::map<std::string, std::string>& GetMessageTags() const;

  bool IsIncoming() const;
  void SetIncoming(bool bIncoming);

  void SetAutoPrefix(bool bAutoPrefix);
  bool GetAutoPrefix() const;

 protected:
  void Clear();

  int32_t mEventId;
  std::string mEventName;
  std::string mPrefix;
  std::map<std::string, std::string> mMessageTags;
  std::vector<std::string> mParams;
  bool mIncoming;
  bool mAutoPrefix;
};

/**
 * Interface for receiving chat notifications.  The implementation handles sending the
 * event to the proper destination.
 */
class ttv::chat::IChatReceiveNetworkEvent {
 public:
  virtual ~IChatReceiveNetworkEvent() = default;

  virtual void ReceiveEvent(const ChatNetworkEvent& evt) = 0;
};

/**
 * Interface for classes which are able to encode and write events to the transport.
 */
class ttv::chat::IChatWriteNetworkEvent {
 public:
  virtual ~IChatWriteNetworkEvent() = default;

  virtual void WriteEvent(const ChatNetworkEvent& evt) = 0;
  virtual void WriteRaw(const std::string& msg) = 0;
};

#define EVENT_OFFSET_COMMAND 1000
#define EVENT_OFFSET_CTCP 2000
#define EVENT_OFFSET_CTCP_REPLY 2500
#define EVENT_OFFSET_SYSTEM 3000

enum EVENT_IRC_REPLY {
  IRC_RPL_WELCOME = 1,
  IRC_RPL_PROTOCTL = 5,
  IRC_RPL_USERHOST = 302,
  IRC_RPL_LISTSTART = 321,
  IRC_RPL_LIST = 322,
  IRC_RPL_LISTEND = 323,
  IRC_RPL_TOPIC = 332,
  IRC_RPL_NAMREPLY = 353,
  IRC_RPL_ENDOFNAMES = 366,
  IRC_RPL_MOTD = 372,

  IRC_ERR_CANNOTSENDTOCHAN = 404,  //"<channel name> :Cannot send to channel"
  IRC_ERR_USERNOTINCHANNEL = 441,  //"<nick> <channel> :They aren't on that channel"
  IRC_ERR_NOTONCHANNEL = 442,      //"<channel> :You're not on that channel"
  IRC_ERR_USERONCHANNEL = 443,     //"<user> <channel> :is already on channel"
  IRC_ERR_KEYSET = 467,            //"<channel> :Channel key already set"
  IRC_ERR_CHANNELISFULL = 471,     //"<channel> :Cannot join channel (+l)"
  IRC_ERR_UNKNOWNMODE = 472,       //"<char> :is unknown mode char to me for <channel>"
  IRC_ERR_INVITEONLYCHAN = 473,    //"<channel> :Cannot join channel (+i)"
  IRC_ERR_BANNEDFROMCHAN = 474,    //"<channel> :Cannot join channel (+b)"
  IRC_ERR_BADCHANNELKEY = 475,     //"<channel> :Cannot join channel (+k)"
  IRC_ERR_BADCHANMASK = 476,       //"<channel> :Bad Channel Mask"
  IRC_ERR_NOCHANMODES = 477,       //"<channel> :Channel doesn't support modes"
  IRC_ERR_BANLISTFULL = 478,       //"<channel> <char> :Channel list is full"
  IRC_ERR_NOPRIVILEGES = 481,      //":Permission Denied- You're not an IRC operator"
  IRC_ERR_CHANOPRIVSNEEDED = 482,  //"<channel> :You're not channel operator"

  IRC_RPL_UNKNOWN = 999
};

enum EVENT_IRC_COMMAND {
  IRC_CMD_NICK = EVENT_OFFSET_COMMAND,
  IRC_CMD_USER,
  IRC_CMD_PASS,
  IRC_CMD_JOIN,
  IRC_CMD_PART,
  IRC_CMD_KICK,
  IRC_CMD_INVITE,
  IRC_CMD_PRIVMSG,
  IRC_CMD_NOTICE,
  IRC_CMD_MODE,
  IRC_CMD_PING,
  IRC_CMD_PONG,
  IRC_CMD_TOPIC,
  IRC_CMD_WHOIS,
  IRC_CMD_WHOWAS,
  IRC_CMD_NAMES,
  IRC_CMD_LIST,
  IRC_CMD_USERHOST,
  IRC_CMD_QUIT,
  IRC_CMD_AWAY,
  IRC_CMD_ERROR,
  IRC_CMD_TWITCHCLIENT,
  IRC_CMD_CAP,
  IRC_CMD_USERSTATE,
  IRC_CMD_GLOBALUSERSTATE,
  IRC_CMD_ROOMSTATE,
  IRC_CMD_CLEARCHAT,
  IRC_CMD_HOSTTARGET,
  IRC_CMD_USERNOTICE,
  IRC_CMD_CLEARMSG,
  IRC_CMD_UNKNOWN,
  IRC_CMD_INVALID = -1
};

enum EVENT_CTCP_COMMAND {
  IRC_CTCP_ACTION = EVENT_OFFSET_CTCP,
  IRC_CTCP_PING,
  IRC_CTCP_VERSION,
  IRC_CTCP_DCC,
  IRC_CTCP_CLEARCHAT,
  IRC_CTCP_UNKNOWN,
  IRC_CTCP_INVALID = -1
};
// ^- these need to be synced -v
enum EVENT_CTCP_REPLY {
  IRC_CTCP_RPL_ACTION = EVENT_OFFSET_CTCP_REPLY,
  IRC_CTCP_RPL_PING,
  IRC_CTCP_RPL_VERSION,
  IRC_CTCP_RPL_DCC,
  IRC_CTCP_RPL_CLEARCHAT,
  IRC_CTCP_RPL_UNKNOWN,
  IRC_CTCP_RPL_INVALID = -1
};

enum EVENT_SYSTEM_MESSAGE {
  SYS_EVENT_TRYCONNECT = EVENT_OFFSET_SYSTEM,
  SYS_EVENT_WAITCONNECT,
  SYS_EVENT_CONNECTED,
  SYS_EVENT_CONNECTFAILED,
  SYS_EVENT_CLOSE,
  SYS_EVENT_STATUS,
  SYS_EVENT_IDENT,
  SYS_EVENT_ERROR,
  SYS_EVENT_UNKNOWN
};

#endif  // TTVSDK_CHATNETWORKEVENT_H
