/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/ichatroom.h"
#include "twitchsdk/chat/ichatroomlistener.h"
#include "twitchsdk/chat/internal/chatuserblocklist.h"
#include "twitchsdk/core/pubsub/pubsubcomponent.h"
#include "twitchsdk/core/user/user.h"

namespace ttv {
namespace chat {
class ChatRoom;
}
}  // namespace ttv

class ttv::chat::ChatRoom : public IChatRoom, public PubSubComponent<IChatRoomListener> {
 public:
  using DisposerFunc = std::function<void()>;

 public:
  ChatRoom(const std::shared_ptr<User>& user, const std::string& roomId, ChannelId channelId);

  // Component Overrides
  static std::string GetComponentName() { return "ttv::chat::ChatRoom"; }
  virtual std::string GetLoggerName() const override;
  virtual void Update() override;

  // IChatRoom implementation
  virtual TTV_ErrorCode Dispose() override;
  virtual TTV_ErrorCode DeleteRoom(const DeleteRoomCallback& callback) override;
  virtual TTV_ErrorCode SendMessage(
    const std::string& message, ChatRoomMessage& placeholderMessage, const SendMessageCallback& callback) override;
  virtual TTV_ErrorCode EditMessage(const std::string& messageId, const std::string& message,
    ChatRoomMessage& placeholderMessage, const EditMessageCallback& callback) override;
  virtual TTV_ErrorCode DeleteMessage(const std::string& messageId, const DeleteMessageCallback& callback) override;
  virtual TTV_ErrorCode FetchMessagesBeforeCursor(
    const std::string& cursor, uint32_t limit, const FetchMessagesCallback& callback) override;
  virtual TTV_ErrorCode FetchMessagesAfterCursor(
    const std::string& cursor, uint32_t limit, const FetchMessagesCallback& callback) override;
  virtual TTV_ErrorCode FetchMessagesBeforeTimestamp(
    Timestamp timestamp, uint32_t limit, const FetchMessagesCallback& callback) override;
  virtual TTV_ErrorCode FetchMessagesAfterTimestamp(
    Timestamp timestamp, uint32_t limit, const FetchMessagesCallback& callback) override;
  virtual TTV_ErrorCode FetchRoomInfo(const FetchRoomInfoCallback& callback) override;
  virtual TTV_ErrorCode SetRoomName(const std::string& name, const UpdateRoomInfoCallback& callback) override;
  virtual TTV_ErrorCode SetTopic(const std::string& topic, const UpdateRoomInfoCallback& callback) override;
  virtual TTV_ErrorCode SetRoomRolePermissions(
    RoomRolePermissions permissions, const UpdateRoomInfoCallback& callback) override;
  virtual TTV_ErrorCode EnableSlowMode(uint32_t durationSeconds, const UpdateRoomModesCallback& callback) override;
  virtual TTV_ErrorCode DisableSlowMode(const UpdateRoomModesCallback& callback) override;
  virtual TTV_ErrorCode EnableR9kMode(const UpdateRoomModesCallback& callback) override;
  virtual TTV_ErrorCode DisableR9kMode(const UpdateRoomModesCallback& callback) override;
  virtual TTV_ErrorCode EnableEmotesOnlyMode(const UpdateRoomModesCallback& callback) override;
  virtual TTV_ErrorCode DisableEmotesOnlyMode(const UpdateRoomModesCallback& callback) override;
  virtual TTV_ErrorCode SetLastReadAt(Timestamp lastReadAt, const UpdateRoomViewCallback& callback) override;
  virtual TTV_ErrorCode SetMuted(bool isMuted, const UpdateRoomViewCallback& callback) override;
  virtual TTV_ErrorCode SetArchived(bool isArchived, const UpdateRoomViewCallback& callback) override;

  void SetTokenizationOptions(const TokenizationOptions& options) { mTokenizationOptions = options; }

  void OnTopicSubscribeStateChanged(
    const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) override;
  void OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) override;

 private:
  TTV_ErrorCode TokenizeLocalMessage(
    const std::shared_ptr<User>& user, const std::string& message, ChatRoomMessage& chatMessage);
  TTV_ErrorCode SetChatMode(
    ChatMode mode, bool turnOn, uint32_t slowModeDurationSeconds, const UpdateRoomModesCallback& callback);

  void FlagIgnoredMessages(std::vector<ChatRoomMessage>& messages);

  std::unordered_map<std::string, uint64_t>
    mMessageNonces;  //!< Keeps track of sent message nonces to prevent duplicates from pubsub.
  std::vector<MessageBadge>
    mCachedUserBadges;  //!< Cache the user's badges received from the last API callback to use for the next placeholder message's badges
  std::string mPubSubTopic;
  std::string mRoomId;
  TokenizationOptions mTokenizationOptions;
  ChannelId mChannelId;
  Color
    mCachedUserColor;  //!< Cache the user's chat color received from the last API callback to use for the next placeholder message's chat color
};
