/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#ifndef TTVSDK_CHATSESSION_H
#define TTVSDK_CHATSESSION_H

#include "twitchsdk/chat/internal/chatnetworkevent.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
namespace chat {
class ChatSession;
class ChatChannel;
}  // namespace chat
}  // namespace ttv

/**
 * A session wraps the event processing for a single channel.
 */
class ttv::chat::ChatSession : public IChatReceiveNetworkEvent {
 public:
  ChatSession();
  ~ChatSession();

  void SetWriter(std::shared_ptr<IChatWriteNetworkEvent> writer);

  void AddEventHandler(IChatReceiveNetworkEvent* handler);
  void RemoveEventHandler(IChatReceiveNetworkEvent* handler);

  void Cap(const std::string& subCommand,
    const std::string&
      capabilities);  //!< Send a CAP REQ command to the server, capabilities is a space separated list.
  void Pass(const std::string& pass);                           //!< Send a PASS command to the server.
  void User(const std::string& nick, const std::string& name);  //!< Send a USER command to the server.
  void TwitchClient();                                          //!< Send a TWITCHCLIENT command to the server.
  void Nick(const std::string& nick);                           //!< Send a NICK command to the server.
  void Join(
    const std::string& channel, const std::string& key = std::string());  //!< Send a JOIN command to the server.
  void Part(
    const std::string& channel, const std::string& msg = std::string());  //!< Send a PART command to the server.
  void Kick(const std::string& channel, const std::string& user,
    const std::string& reason = std::string());                      //!< Send a KICK command to the server.
  void Invite(const std::string& user, const std::string& channel);  //!< Send an INVITE command to the server.
  void Topic(
    const std::string& channel, const std::string& topic = std::string());  //!< Send a TOPIC command to the server.
  void PrivMsg(const std::string& target, const std::string& msg);          //!< Send a PRIVMSG command to the server.
  void Action(const std::string& target, const std::string& msg);           //!< Send an ACTION command to the server.
  void Notice(const std::string& target, const std::string& msg);           //!< Send a NOTICE command to the server.
  void CTCP(const std::string& target, const std::string& cmd, const std::string& msg = std::string());
  void CTCPReply(const std::string& target, const std::string& cmd, const std::string& msg = std::string());
  void CTCPPing(const std::string& target);
  void Whois(const std::string& target);
  void Mode(const std::string& target, const std::string& modes);
  void UserHost(const std::string& user);
  void Quit(const std::string& msg);  //!< Send a QUIT command to the server.
  void Away(const std::string& msg);
  void Raw(const std::string& text);
  void Ping(const std::string& target);  //!< Send a PING command to the server.
  void Pong(const std::string& target);  //!< Send a PONG command to the server.
  void List();

  const std::string& GetNick() const;
  void SetNick(const std::string& nick);
  bool IsMe(const std::string& nick);

  bool IsConnected() { return mConnected; }
  uint64_t GetIdleTime();

  std::string GetChannelTypes();
  void GetChannelUserModes(std::string* letters, std::string* symbols);

  bool HasServerCapability(const std::string& cap);
  std::string GetServerCapability(const std::string& cap);

  // IChatReceiveNetworkEvent implementation ///////////////////////
  virtual void ReceiveEvent(const ChatNetworkEvent& evt);  //!< Accept an incoming event.

 protected:
  void ResetIdleCounter();

  // DispatchEvent - Send an event to the display handlers
  void DispatchEvent(const ChatNetworkEvent& evt);

  // DoEvent - Send an outgoing event
  void DoEvent(const ChatNetworkEvent& evt);

  void InitDefaultServerCapabilities();

  void OnConnect(const ChatNetworkEvent& evt);
  void OnDisconnect(const ChatNetworkEvent& evt);

  void OnJoin(const ChatNetworkEvent& evt);
  void OnPart(const ChatNetworkEvent& evt);
  void OnKick(const ChatNetworkEvent& evt);
  void OnNick(const ChatNetworkEvent& evt);
  void OnQuit(const ChatNetworkEvent& evt);
  void OnPing(const ChatNetworkEvent& evt);
  void OnMode(const ChatNetworkEvent& evt);

  void OnCTCPPing(const ChatNetworkEvent& evt);
  void OnCTCPVersion(const ChatNetworkEvent& evt);
  void OnCTCPRplPing(const ChatNetworkEvent& evt);

  void OnRplNamReply(const ChatNetworkEvent& evt);
  void OnRplWelcome(const ChatNetworkEvent& evt);
  void OnRplProtoCtl(const ChatNetworkEvent& evt);
  void OnRplListStart(const ChatNetworkEvent& evt);
  void OnRplList(const ChatNetworkEvent& evt);

 private:
  std::map<std::string, std::string> m_mapServerCaps;

  std::string mNick;
  std::vector<IChatReceiveNetworkEvent*> mEventHandlers;  //!< The child event handlers for incoming chat events.

  std::shared_ptr<IChatWriteNetworkEvent> mWriter;  //!< The stream writer.

  uint64_t mLastActivity;
  bool mConnected;
};

#endif  // TTVSDK_CHATSESSION_H
