/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/task/chatgetblocklisttask.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <unordered_set>

namespace ttv {
class TaskRunner;
class User;
class UserRepository;

namespace chat {
class ChatUserBlockList;
class UserBlockListListenerProxy;
class ChatChangeUserBlockTask;
}  // namespace chat
}  // namespace ttv

class ttv::chat::ChatUserBlockList : public ttv::UserComponent {
 public:
  using BlockChangeCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchBlockedUsersCallback = std::function<void(TTV_ErrorCode ec, const std::vector<UserInfo>& blockedUsers)>;

  ChatUserBlockList(std::shared_ptr<User> user);
  virtual ~ChatUserBlockList() override;

  TTV_ErrorCode UpdateList();

  TTV_ErrorCode BlockUser(UserId userId, const std::string& reason, bool whisper, const BlockChangeCallback& = nullptr);
  TTV_ErrorCode BlockUser(
    const std::string& userName, const std::string& reason, bool whisper, const BlockChangeCallback& = nullptr);
  TTV_ErrorCode UnblockUser(UserId userId, const BlockChangeCallback& = nullptr);
  TTV_ErrorCode UnblockUser(const std::string& userName, const BlockChangeCallback& = nullptr);
  TTV_ErrorCode FetchBlockedUsers(const FetchBlockedUsersCallback& callback = nullptr);

  bool IsUserBlocked(UserId userId);
  void GetBlockedUsers(std::unordered_set<UserId>& result) const;

  void SetRefreshInterval(uint64_t milliseconds);
  uint64_t GetRefreshInterval() const { return mRefreshIntervalMilliseconds; }

  std::shared_ptr<User> GetUser() { return mUser.lock(); }

  static std::string GetComponentName() { return "ttv::chat::ChatUserBlockList"; }

  // Component overrides
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;

  // UserComponent overrides
  virtual std::string GetLoggerName() const override;

 private:
  using SetReference = std::shared_ptr<std::unordered_set<UserId>>;

  // Component overrides
  virtual bool CheckShutdown() override;

  bool IsRefreshInProgress() const;
  void FetchBlocks();
  void RefreshComplete(TTV_ErrorCode ec);
  void CancelRequestsForUser(UserId blockUserId);
  void ScheduleRequest(
    UserId blockUserId, bool block, const std::string& reason, bool whisper, const BlockChangeCallback& callback);
  void ProcessNextRequest();
  void SetLocalValue(SetReference set, UserId blockUserId, bool block);

  struct ChangeRequest {
    std::string reason;
    BlockChangeCallback callback;
    UserId userId;
    bool block;
    bool whisper;
  };

  std::weak_ptr<UserRepository> mUserRepository;
  SetReference mBlockedUsers;           //!< The current set of users.
  SetReference mRefreshUsers;           //!< The set of users incrementally built during a refresh.
  std::vector<UserInfo> mUserInfoList;  //!< List of UserInfos fetched
  std::vector<ChangeRequest>
    mQueuedChanges;  //!< The set of server changes that are queued while a refresh is in progress.
  CallbackQueue<FetchBlockedUsersCallback> mBlockedUsersCallbacks;
  std::shared_ptr<ChatChangeUserBlockTask> mRunningChangeTask;
  std::shared_ptr<ChatGetBlockListTask> mRunningGetListTask;
  std::shared_ptr<ChatGetBlockListTask::Result> mRefreshResult;
  WaitForExpiry mRefreshTimer;
  uint64_t mRefreshIntervalMilliseconds;
};
