/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chatlistener.h"
#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/ichatobjectfactory.h"
#include "twitchsdk/chat/internal/bitsconfigrepository.h"
#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"

#include <unordered_set>

#include <atomic>
#include <map>
#include <memory>
#include <string>
#include <vector>

namespace ttv {
class TaskRunner;
class Task;
class User;
class UserRepository;
class SettingRepository;

namespace chat {
class ChatUserThreads;
class BitsConfigRepository;
}  // namespace chat
}  // namespace ttv

/**
 * The collection of conversation threads from the perspective of a user.
 */
class ttv::chat::ChatUserThreads : public UserComponent {
 public:
  ChatUserThreads(std::shared_ptr<User> user);
  virtual ~ChatUserThreads() override;

  void SetTokenizationOptions(const TokenizationOptions& options) { mTokenizationOptions = options; }
  void SetUserRepository(std::shared_ptr<UserRepository> repository) { mUserRepository = repository; }
  void SetSettingRepository(std::shared_ptr<SettingRepository> settings);
  void SetBitsConfigRepository(std::shared_ptr<BitsConfigRepository> repository) { mBitsConfigRepository = repository; }
  void SetListener(std::shared_ptr<IChatUserThreadsListener> listener);

  // Component overrides
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override;
  virtual void OnUserInfoFetchComplete(TTV_ErrorCode ec) override;

  /**
   * Forces a fetch of the user's avaiable emote set ids.
   */
  TTV_ErrorCode FetchEmoteSets();

  static std::string GetComponentName() { return "ttv::chat::ChatUserThreads"; }

 protected:
  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

 private:
  class PubSubTopicListener : public PubSubClient::ITopicListener {
   public:
    PubSubTopicListener(ChatUserThreads* owner);

    // PubSubClient::ITopicListener implementation
    virtual void OnTopicSubscribeStateChanged(PubSubClient* source, const std::string& topic,
      PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) override;
    virtual void OnTopicMessageReceived(
      PubSubClient* source, const std::string& topic, const json::Value& msg) override;
    virtual void OnTopicListenerRemoved(PubSubClient* source, const std::string& topic, TTV_ErrorCode ec) override;

   private:
    ChatUserThreads* mOwner;
  };

  // PubSubClient::ITopicListener implementation
  void OnTopicSubscribeStateChanged(
    const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec);
  void OnTopicMessageReceived(const std::string& topic, const json::Value& msg);

  TTV_ErrorCode SubscribeTopics();
  TTV_ErrorCode TokenizeLocalMessage(const std::shared_ptr<User>& user, const std::string& message,
    const std::string& threadId, WhisperMessage& chatMessage);
  TTV_ErrorCode FetchBitsConfig();

  std::weak_ptr<UserRepository> mUserRepository;
  std::shared_ptr<SettingRepository> mSettingRepository;
  std::shared_ptr<BitsConfigRepository> mBitsConfigRepository;
  std::shared_ptr<BitsConfiguration> mBitsConfiguration;
  EventSource<IChatUserThreadsListener> mListeners;

  std::shared_ptr<PubSubClient> mPubSub;
  std::shared_ptr<PubSubTopicListener> mPubSubTopicListener;
  std::shared_ptr<PubSubTopicListenerHelper> mPubSubTopicListenerHelper;
  std::string mWhisperPubSubTopic;  //!< The topic used to subscribe for realtime whisper events

  RetryTimer mFetchBitsConfigRetryTimer;
  BitsConfigRepository::LookupId mBitsConfigFetchToken;  //!< The cancellation token for fetching the BitsConfiguration.
  TokenizationOptions mTokenizationOptions;
  bool mHasFetchedBitsConfig;
};
