/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/internal/json/chatjsonobjectschemas.h"
#include "twitchsdk/core/json/jsonserialization.h"

namespace ttv {
namespace chat {
namespace json {
namespace description {
struct GraphQLChatRoomMessage;         //!< Description for ChatRoomMessage objects received from GraphQL API callbacks
struct GraphQLChatRoomInfo;            //!< Description for ChatRoomInfo objects received from GraphQL API callbacks
struct GraphQLChatRoomView;            //!< Description for ChatRoomView objects received from GraphQL API callbacks
struct GraphQLUserInfo;                //!< Description for UserInfo objects received from GraphQL API callbacks
struct GraphQLRoomPermissions;         //!< Description for ChatRoomPermissions received from GraphQL API callbacks
struct GraphQLChatMode;                //!< Description for ChatModeInfo received from GraphQL API callbacks
struct PubSubChatRoomMessage;          //!< Description for ChatRoomMessage objects received from pubsub
struct PubSubChatRoomInfo;             //!< Description for ChatRoomInfo objects received from pubsub
struct PubSubChatRoomView;             //!< Description for ChatRoomView objects received from pubsub
struct PubSubRoomPermissions;          //!< Description for ChatRoomPermissions received from pubsub
struct PubSubChatMode;                 //!< Description for ChatModeInfo received from pubsub
struct PubSubHostingInfo;              //!< Description for HostingInfo received from pubsub
struct PubSubBitsSentEvent;            //!< Description for BitsSentEvent received from pubsub
struct PubSubChatChannelRestrictions;  //!< Description for ChatChannelRestrictions received from pubsub
struct ChatRoomRole;                   //!< Default description for RoomRole enum objects
struct ChatRoomRolePermissions;        //!< Default description for RoomRolePermissions
struct ChatRoomMentionInfo;            //!< Default description for ChatRoomMentionInfo
struct GraphQLErrorCode;               //!< Default description for GraphQLErrorCode
struct SendRoomMessageError;           //!< Default description for SendRoomMessageError
struct CreateRoomError;                //!< Default description for CreateRoomError
struct UpdateRoomError;                //!< Default description for UpdateRoomError
struct UpdateRoomModesError;           //!< Default description for UpdateRoomModesError
struct BanUserError;                   //!< Default description for BanUserError
struct UnbanUserError;                 //!< Default description for UnbanUserError
struct EmoticonModifier;               //!< Default description for emoticon modifier
struct Emoticon;                       //!< Default description for emoticons from GraphQL
struct ExtensionMessage;               //!< Default description for messages received from extensions through pubsub
struct Chanlet;                        //!< Default description for chanlet updates through pubsub
struct MultiviewContentAttribute;      //!< Default description for multi-view content attributes through pubsub
}  // namespace description
}  // namespace json
}  // namespace chat
}  // namespace ttv

struct ttv::chat::json::description::GraphQLChatRoomMessage {
  template <typename ChatRoomMessageType>
  static auto BindFields(ChatRoomMessageType& message) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", message.roomMessageId),
      make_field<RequiredField>(MakeKeyPath("room", "id"), message.roomId),
      make_field<DateSchema, OptionalField>("sentAt", message.messageInfo.timestamp),
      make_field<RoomMessageDeletedSchema, OptionalField>("deletedAt", message.messageInfo.flags),
      make_field<GraphQLChatRoomMessageContentSchema, RequiredField>("content", message.messageInfo),
      make_field<GraphQLChatRoomMessageSenderSchema, RequiredField>("sender", message.messageInfo));
  }
};

struct ttv::chat::json::description::GraphQLChatRoomInfo {
  template <typename ChatRoomInfoType>
  static auto BindFields(ChatRoomInfoType& info) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", info.id), make_field<RequiredField>("name", info.name),
      make_field<RequiredField>("topic", info.topic),
      make_field<RequiredField>("rolePermissions", info.rolePermissions),
      make_field<ObjectSchema<GraphQLChatRoomView>, RequiredField>("self", info.view),
      make_field<ObjectSchema<GraphQLUserInfo>, RequiredField>("owner", info.owner),
      make_field<ObjectSchema<GraphQLChatMode>, RequiredField>("modes", info.modes));
  }
};

struct ttv::chat::json::description::GraphQLChatRoomView {
  template <typename ChatRoomViewType>
  static auto BindFields(ChatRoomViewType& view) {
    using namespace ttv::json;

    return std::make_tuple(make_field<DateSchema, OptionalField>("lastReadAt", view.lastReadAt),
      make_field<RequiredField>("isMuted", view.isMuted), make_field<RequiredField>("isArchived", view.isArchived),
      make_field<RequiredField>("isUnread", view.isUnread),
      make_field<RequiredField>("unreadMentionCount", view.unreadMentionCount),
      make_field<ObjectSchema<GraphQLRoomPermissions>, OptionalField>("permissions", view.permissions));
  }
};

struct ttv::chat::json::description::GraphQLUserInfo {
  template <typename UserInfoType>
  static auto BindFields(UserInfoType& info) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", info.userId),
      make_field<RequiredField>("login", info.userName), make_field("displayName", info.displayName),
      make_field("description", info.bio), make_field("profileImageURL", info.logoImageUrl),
      make_field<DateSchema, OptionalField>("createdAt", info.createdTimestamp));
  }
};

struct ttv::chat::json::description::GraphQLRoomPermissions {
  template <typename RoomPermissionsType>
  static auto BindFields(RoomPermissionsType& permissions) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("readMessages", permissions.readMessages),
      make_field<RequiredField>("sendMessages", permissions.sendMessages),
      make_field<RequiredField>("moderate", permissions.moderate));
  }
};

struct ttv::chat::json::description::GraphQLChatMode {
  template <typename ChatModeType>
  static auto BindFields(ChatModeType& mode) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("slowModeDurationSeconds", mode.slowModeDurationSeconds),
      make_field<RequiredField>("r9kModeEnabled", mode.r9kMode),
      make_field<RequiredField>("emotesOnlyModeEnabled", mode.emotesOnlyMode));
  }
};

struct ttv::chat::json::description::PubSubChatRoomMessage {
  template <typename ChatRoomMessageType>
  static auto BindFields(ChatRoomMessageType& message) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", message.roomMessageId),
      make_field<RequiredField>("room", message.roomId),
      make_field<DateSchema, OptionalField>("sent_at", message.messageInfo.timestamp),
      make_field<PubSubChatRoomMessageContentSchema, RequiredField>("content", message.messageInfo),
      make_field<PubSubChatRoomMessageSenderSchema, RequiredField>("sender", message.messageInfo));
  }
};

struct ttv::chat::json::description::PubSubChatRoomInfo {
  template <typename ChatRoomInfoType>
  static auto BindFields(ChatRoomInfoType& roomInfo) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("room_id", roomInfo.id),
      make_field<RequiredField>("name", roomInfo.name), make_field<RequiredField>("topic", roomInfo.topic),
      make_field<RequiredField>("permissions", roomInfo.rolePermissions),
      make_field<ObjectSchema<PubSubChatMode>, RequiredField>("modes", roomInfo.modes));
  }
};

struct ttv::chat::json::description::PubSubChatMode {
  template <typename ChatModeType>
  static auto BindFields(ChatModeType& mode) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("slow_mode_duration_seconds", mode.slowModeDurationSeconds),
      make_field<RequiredField>("r9k_enabled", mode.r9kMode),
      make_field<RequiredField>("emotes_only_mode_enabled", mode.emotesOnlyMode));
  }
};

struct ttv::chat::json::description::PubSubChatRoomView {
  template <typename ChatRoomViewType>
  static auto BindFields(ChatRoomViewType& roomView) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("is_muted", roomView.isMuted),
      make_field<RequiredField>("is_archived", roomView.isArchived),
      make_field<RequiredField>("is_unread", roomView.isUnread),
      make_field<DateSchema, OptionalField>("last_read_at", roomView.lastReadAt),
      make_field<RequiredField>("unread_mention_count", roomView.unreadMentionCount),
      make_field<ObjectSchema<PubSubRoomPermissions>, OptionalField>("permissions", roomView.permissions));
  }
};

struct ttv::chat::json::description::PubSubRoomPermissions {
  template <typename RoomPermissionsType>
  static auto BindFields(RoomPermissionsType& permissions) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("read_messages", permissions.readMessages),
      make_field<RequiredField>("send_messages", permissions.sendMessages),
      make_field<RequiredField>("moderate", permissions.moderate));
  }
};

struct ttv::chat::json::description::PubSubHostingInfo {
  template <typename HostingInfoType>
  static auto BindFields(HostingInfoType& info) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("channel_login", info.hostChannelName),
      make_field<OptionalField>("target_channel_login", info.targetChannelName),
      make_field<RequiredField>("channel_id", info.hostChannelId),
      make_field<OptionalField>("target_channel_id", info.targetChannelId),
      make_field<OptionalField>("previous_target_channel_id", info.previousTargetChannelId),
      make_field<RequiredField>("num_viewers", info.numViewers));
  }
};

struct ttv::chat::json::description::PubSubBitsSentEvent {
  template <typename BitsSentEventType>
  static auto BindFields(BitsSentEventType& bitsSentEvent) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("balance", bitsSentEvent.userBitsBalance),
      make_field<RequiredField>(MakeKeyPath("channel_total", "channel_id"), bitsSentEvent.channelId),
      make_field<RequiredField>(MakeKeyPath("channel_total", "total"), bitsSentEvent.channelBitsTotal));
  }
};

struct ttv::chat::json::description::PubSubChatChannelRestrictions {
  template <typename ChatChannelRestrictions>
  static auto BindFields(ChatChannelRestrictions& chatChannelRestrictions) {
    using namespace ttv::json;

    return std::make_tuple(
      make_field<OptionalField>("followers_only_duration_minutes", chatChannelRestrictions.followersDuration),
      make_field<RequiredField>("emote_only_mode_enabled", chatChannelRestrictions.emoteOnly),
      make_field<RequiredField>("subscribers_only_mode_enabled", chatChannelRestrictions.subscribersOnly),
      make_field<RequiredField>("verified_only_mode_enabled", chatChannelRestrictions.verifiedOnly),
      make_field<RequiredField>("r9k_mode_enabled", chatChannelRestrictions.r9k),
      make_field<OptionalField>("slow_mode_duration_seconds", chatChannelRestrictions.slowModeDuration),
      make_field<DateSchema, OptionalField>("slow_mode_set_at", chatChannelRestrictions.slowModeSetAt));
  }
};

struct ttv::chat::json::description::ChatRoomRole {
  static auto EnumMap() {
    using namespace ttv::json;

    return std::make_tuple(make_enum_mapping("EVERYONE", RoomRole::Everyone),
      make_enum_mapping("SUBSCRIBER", RoomRole::Subscriber), make_enum_mapping("MODERATOR", RoomRole::Moderator),
      make_enum_mapping("BROADCASTER", RoomRole::Broadcaster));
  }

  static auto GetFallbackValue() { return RoomRole::Unknown; }
};

struct ttv::chat::json::description::ChatRoomRolePermissions {
  template <typename PermissionsType>
  static auto BindFields(PermissionsType& permissions) {
    using namespace ttv::json;

    return std::make_tuple(
      make_field<RequiredField>("read", permissions.read), make_field<RequiredField>("send", permissions.send));
  }
};

struct ttv::chat::json::description::ChatRoomMentionInfo {
  template <typename ChatRoomMentionInfoType>
  static auto BindFields(ChatRoomMentionInfoType& mentionInfo) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("channel_id", mentionInfo.roomOwnerId),
      make_field<RequiredField>("channel_display_name", mentionInfo.roomOwnerName),
      make_field<RequiredField>("channel_login", mentionInfo.roomOwnerLogin),
      make_field<RequiredField>("sender_id", mentionInfo.senderId),
      make_field<RequiredField>("sender_display_name", mentionInfo.senderName),
      make_field<RequiredField>("room_id", mentionInfo.roomId),
      make_field<RequiredField>("room_name", mentionInfo.roomName),
      make_field<RequiredField>("message_id", mentionInfo.messageId),
      make_field<DateSchema, RequiredField>("sent_at", mentionInfo.sentAt));
  }
};

struct ttv::chat::json::description::GraphQLErrorCode {
  static auto EnumMap() {
    using namespace ttv::json;

    return std::make_tuple(
      make_enum_mapping("AUTO_MOD_ENFORCEMENT_FAILED", ttv::chat::GraphQLErrorCode::AUTO_MOD_ENFORCEMENT_FAILED),
      make_enum_mapping("CHANNEL_NOT_FOUND", ttv::chat::GraphQLErrorCode::CHANNEL_NOT_FOUND),
      make_enum_mapping("DURATION_INVALID", ttv::chat::GraphQLErrorCode::DURATION_INVALID),
      make_enum_mapping(
        "EMOTE_ONLY_MODE_ENFORCEMENT_FAILED", ttv::chat::GraphQLErrorCode::EMOTE_ONLY_MODE_ENFORCEMENT_FAILED),
      make_enum_mapping("FORBIDDEN", ttv::chat::GraphQLErrorCode::FORBIDDEN),
      make_enum_mapping("MAX_ROOMS_LIMIT_EXCEEDED", ttv::chat::GraphQLErrorCode::MAX_ROOMS_LIMIT_EXCEEDED),
      make_enum_mapping(
        "NAME_CONTAINS_INVALID_CHARACTERS", ttv::chat::GraphQLErrorCode::NAME_CONTAINS_INVALID_CHARACTERS),
      make_enum_mapping("NAME_INAPPROPRIATE", ttv::chat::GraphQLErrorCode::NAME_INAPPROPRIATE),
      make_enum_mapping("NAME_LENGTH_INVALID", ttv::chat::GraphQLErrorCode::NAME_LENGTH_INVALID),
      make_enum_mapping("R9K_MODE_ENFORCEMENT_FAILED", ttv::chat::GraphQLErrorCode::R9K_MODE_ENFORCEMENT_FAILED),
      make_enum_mapping("RATE_LIMIT_FAILED", ttv::chat::GraphQLErrorCode::RATE_LIMIT_FAILED),
      make_enum_mapping("ROLES_INVALID", ttv::chat::GraphQLErrorCode::ROLES_INVALID),
      make_enum_mapping("ROOM_NOT_FOUND", ttv::chat::GraphQLErrorCode::ROOM_NOT_FOUND),
      make_enum_mapping("SLOW_MODE_DURATION_INVALID", ttv::chat::GraphQLErrorCode::SLOW_MODE_DURATION_INVALID),
      make_enum_mapping("SLOW_MODE_ENFORCEMENT_FAILED", ttv::chat::GraphQLErrorCode::SLOW_MODE_ENFORCEMENT_FAILED),
      make_enum_mapping("SPAM_ENFORCEMENT_FAILED", ttv::chat::GraphQLErrorCode::SPAM_ENFORCEMENT_FAILED),
      make_enum_mapping("TARGET_ALREADY_BANNED", ttv::chat::GraphQLErrorCode::TARGET_ALREADY_BANNED),
      make_enum_mapping("TARGET_ALREADY_MOD", ttv::chat::GraphQLErrorCode::TARGET_ALREADY_MOD),
      make_enum_mapping("TARGET_IS_ADMIN", ttv::chat::GraphQLErrorCode::TARGET_IS_ADMIN),
      make_enum_mapping("TARGET_IS_ANONYMOUS", ttv::chat::GraphQLErrorCode::TARGET_IS_ANONYMOUS),
      make_enum_mapping("TARGET_IS_BROADCASTER", ttv::chat::GraphQLErrorCode::TARGET_IS_BROADCASTER),
      make_enum_mapping("TARGET_IS_CHAT_BANNED", ttv::chat::GraphQLErrorCode::TARGET_IS_CHAT_BANNED),
      make_enum_mapping("TARGET_IS_GLOBAL_MOD", ttv::chat::GraphQLErrorCode::TARGET_IS_GLOBAL_MOD),
      make_enum_mapping("TARGET_IS_MOD", ttv::chat::GraphQLErrorCode::TARGET_IS_MOD),
      make_enum_mapping("TARGET_IS_SELF", ttv::chat::GraphQLErrorCode::TARGET_IS_SELF),
      make_enum_mapping("TARGET_IS_STAFF", ttv::chat::GraphQLErrorCode::TARGET_IS_STAFF),
      make_enum_mapping("TARGET_NOT_BANNED", ttv::chat::GraphQLErrorCode::TARGET_NOT_BANNED),
      make_enum_mapping("TARGET_NOT_FOUND", ttv::chat::GraphQLErrorCode::TARGET_NOT_FOUND),
      make_enum_mapping("TARGET_NOT_MOD", ttv::chat::GraphQLErrorCode::TARGET_NOT_MOD),
      make_enum_mapping("TOPIC_INAPPROPRIATE", ttv::chat::GraphQLErrorCode::TOPIC_INAPPROPRIATE),
      make_enum_mapping("TOPIC_LENGTH_INVALID", ttv::chat::GraphQLErrorCode::TOPIC_LENGTH_INVALID),
      make_enum_mapping("USER_CHAT_BANNED", ttv::chat::GraphQLErrorCode::USER_CHAT_BANNED),
      make_enum_mapping("USER_CHAT_TIMED_OUT", ttv::chat::GraphQLErrorCode::USER_CHAT_TIMED_OUT),
      make_enum_mapping("USER_SUSPENDED", ttv::chat::GraphQLErrorCode::USER_SUSPENDED),
      make_enum_mapping("ZALGO_ENFORCEMENT_FAILED", ttv::chat::GraphQLErrorCode::ZALGO_ENFORCEMENT_FAILED));
  }

  static auto GetFallbackValue() { return ttv::chat::GraphQLErrorCode::UNKNOWN; }
};

struct ttv::chat::json::description::SendRoomMessageError {
  template <typename SendRoomMessageErrorType>
  static auto BindFields(SendRoomMessageErrorType& error) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("code", error.code),
      make_field<OptionalField>("slowModeDurationSeconds", error.slowModeDurationSeconds),
      make_field<OptionalField>("remainingDurationSeconds", error.remainingDurationSeconds));
  }
};

struct ttv::chat::json::description::CreateRoomError {
  template <typename CreateRoomErrorType>
  static auto BindFields(CreateRoomErrorType& error) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("code", error.code),
      make_field<OptionalField>("maxAllowedRooms", error.maxAllowedRooms),
      make_field<OptionalField>("minLength", error.minLength), make_field<OptionalField>("maxLength", error.maxLength));
  }
};

struct ttv::chat::json::description::UpdateRoomError {
  template <typename UpdateRoomErrorType>
  static auto BindFields(UpdateRoomErrorType& error) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("code", error.code),
      make_field<OptionalField>("minLength", error.minLength), make_field<OptionalField>("maxLength", error.maxLength));
  }
};

struct ttv::chat::json::description::UpdateRoomModesError {
  template <typename UpdateRoomErrorType>
  static auto BindFields(UpdateRoomErrorType& error) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("code", error.code),
      make_field<OptionalField>("minimumSlowModeDurationSeconds", error.minimumSlowModeDurationSeconds),
      make_field<OptionalField>("maximumSlowModeDurationSeconds", error.maximumSlowModeDurationSeconds));
  }
};

struct ttv::chat::json::description::BanUserError {
  template <typename BanUserErrorType>
  static auto BindFields(BanUserErrorType& error) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("code", error.code));
  }
};

struct ttv::chat::json::description::UnbanUserError {
  template <typename UnbanUserErrorType>
  static auto BindFields(UnbanUserErrorType& error) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("code", error.code));
  }
};

struct ttv::chat::json::description::EmoticonModifier {
  template <typename EmoticonModifierType>
  static auto BindFields(EmoticonModifierType& emoticonModifier) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("code", emoticonModifier.code));
  }
};

struct ttv::chat::json::description::Emoticon {
  template <typename EmoticonType>
  static auto BindFields(EmoticonType& emoticon) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", emoticon.emoticonId),
      make_field<EmoticonTokenSchema, RequiredField>("token", emoticon),
      make_field<OptionalField>("modifiers", emoticon.modifiers));
  }
};

struct ttv::chat::json::description::ExtensionMessage {
  template <typename ExtensionMessageType>
  static auto BindFields(ExtensionMessageType& message) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", message.messageId),
      make_field<DateSchema, RequiredField>("sent_at", message.sentAt),
      make_field<PubSubMessageFragmentsSchema, RequiredField>(MakeKeyPath("content", "fragments"), message.tokens),
      make_field<RequiredField>(MakeKeyPath("sender", "extension_client_id"), message.extensionClientId),
      make_field<RequiredField>(MakeKeyPath("sender", "extension_version"), message.extensionVersion),
      make_field<RequiredField>(MakeKeyPath("sender", "display_name"), message.extensionDisplayName),
      make_field<ColorSchema, RequiredField>(MakeKeyPath("sender", "chat_color"), message.chatColor),
      make_field<PubSubMessageBadgesSchema, RequiredField>(MakeKeyPath("sender", "badges"), message.badges));
  }
};

struct ttv::chat::json::description::Chanlet {
  template <typename ChanletType>
  static auto BindFields(ChanletType& chanlet) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", chanlet.chanletId),
      make_field<OptionalField>("contentAttributes", chanlet.attributes));
  }
};

struct ttv::chat::json::description::MultiviewContentAttribute {
  template <typename AttributeType>
  static auto BindFields(AttributeType& attribute) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", attribute.attributeId),
      make_field<RequiredField>("key", attribute.key), make_field<RequiredField>("name", attribute.name),
      make_field<RequiredField>(MakeKeyPath("ownerChannel", "id"), attribute.ownerChannelId),
      make_field<OptionalField>("parentID", attribute.parentId),
      make_field<OptionalField>("parentKey", attribute.parentKey), make_field<RequiredField>("value", attribute.value),
      make_field<OptionalField>("valueShortName", attribute.valueShortName),
      make_field<OptionalField>("imageURL", attribute.imageUrl),
      make_field<DateSchema, OptionalField>("createdAt", attribute.createdAt),
      make_field<DateSchema, OptionalField>("updatedAt", attribute.updatedAt));
  }
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::BitsSentEvent> {
  using Type = ObjectSchema<ttv::chat::json::description::PubSubBitsSentEvent>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::RoomRole> {
  using Type = EnumSchema<ttv::chat::json::description::ChatRoomRole>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::RoomMentionInfo> {
  using Type = ObjectSchema<ttv::chat::json::description::ChatRoomMentionInfo>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::GraphQLErrorCode> {
  using Type = EnumSchema<ttv::chat::json::description::GraphQLErrorCode>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::SendRoomMessageError> {
  using Type = ObjectSchema<ttv::chat::json::description::SendRoomMessageError>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::CreateRoomError> {
  using Type = ObjectSchema<ttv::chat::json::description::CreateRoomError>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::UpdateRoomError> {
  using Type = ObjectSchema<ttv::chat::json::description::UpdateRoomError>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::UpdateRoomModesError> {
  using Type = ObjectSchema<ttv::chat::json::description::UpdateRoomModesError>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::BanUserError> {
  using Type = ObjectSchema<ttv::chat::json::description::BanUserError>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::UnbanUserError> {
  using Type = ObjectSchema<ttv::chat::json::description::UnbanUserError>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::EmoticonModifier> {
  using Type = ObjectSchema<ttv::chat::json::description::EmoticonModifier>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::Emoticon> {
  using Type = ObjectSchema<ttv::chat::json::description::Emoticon>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::RoomRolePermissions> {
  using Type = ObjectSchema<ttv::chat::json::description::ChatRoomRolePermissions>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::ExtensionMessage> {
  using Type = ObjectSchema<ttv::chat::json::description::ExtensionMessage>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::Chanlet> {
  using Type = ObjectSchema<ttv::chat::json::description::Chanlet>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::MultiviewContentAttribute> {
  using Type = ObjectSchema<ttv::chat::json::description::MultiviewContentAttribute>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::chat::ChatChannelRestrictions> {
  using Type = ObjectSchema<ttv::chat::json::description::PubSubChatChannelRestrictions>;
};
