/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonserialization.h"
#include "twitchsdk/core/stringutilities.h"

namespace ttv {
namespace chat {
namespace json {
struct PubSubMessageBadgesSchema;
struct PubSubMessageFragmentsSchema;
struct PubSubChatRoomMessageContentSchema;
struct PubSubChatRoomMessageSenderSchema;
struct GraphQLChatRoomMessageContentSchema;
struct GraphQLChatRoomMessageSenderSchema;
struct RoomMessageDeletedSchema;
struct EmoticonTokenSchema;
struct EmoticonSetsSchema;
}  // namespace json
}  // namespace chat
}  // namespace ttv

struct ttv::chat::json::PubSubMessageFragmentsSchema {
  static bool Parse(const ttv::json::Value& value, std::vector<std::unique_ptr<MessageToken>>& fragments) {
    if (value.isNull() || !value.isArray()) {
      return false;
    }

    for (const auto& jFragment : value) {
      const auto& jText = jFragment["text"];
      if (!jText.isNull() && jText.isString()) {
        bool parsed = false;
        std::string text = jText.asString();
        const auto& jEmoticon = jFragment["emoticon"];
        const auto& jMention = jFragment["mention"];

        if (!jEmoticon.isNull() && jEmoticon.isObject()) {
          std::string emoticonId;
          if (ParseEmoticonId(jEmoticon, "id", emoticonId)) {
            fragments.emplace_back(std::make_unique<ttv::chat::EmoticonToken>(text, emoticonId));
            parsed = true;
          }
        } else if (!jMention.isNull() && jMention.isObject()) {
          std::string userName;
          if (ParseString(jMention, "display_name", userName)) {
            fragments.emplace_back(std::make_unique<ttv::chat::MentionToken>(userName, text, false));
            parsed = true;
          }
        } else if (ttv::chat::IsTwitchChatUrl(text)) {
          fragments.emplace_back(std::make_unique<ttv::chat::UrlToken>(text, false));
          parsed = true;
        }

        // If parsing as the other tokens failed, just parse it as normal text.
        if (!parsed) {
          std::string fragmentText = jText.asString();
          fragments.emplace_back(std::make_unique<ttv::chat::TextToken>(fragmentText));
        }
      } else {
        return false;
      }
    }

    return true;
  }
};

struct ttv::chat::json::PubSubMessageBadgesSchema {
  static bool Parse(const ttv::json::Value& value, std::vector<MessageBadge>& badges) {
    if (value.isNull() || !value.isArray()) {
      return false;
    }

    for (const auto& jBadge : value) {
      MessageBadge badge;
      if (ParseString(jBadge, "id", badge.name) && ParseString(jBadge, "version", badge.version)) {
        badges.emplace_back(std::move(badge));
      }
    }

    return true;
  }
};

struct ttv::chat::json::PubSubChatRoomMessageContentSchema {
  static bool Parse(const ttv::json::Value& value, MessageInfo& info) {
    if (value.isNull()) {
      return false;
    }

    const auto& jFragments = value["fragments"];
    if (jFragments.isNull() || !jFragments.isArray()) {
      // This is valid since a deleted message has no fragments
      return true;
    }

    if (!ttv::chat::json::PubSubMessageFragmentsSchema::Parse(jFragments, info.tokens)) {
      return false;
    }

    if (!info.tokens.empty() && info.tokens[0]->GetType() == MessageToken::Type::Text) {
      std::string mePrefix = "/me";

      auto& firstTokenText = static_cast<TextToken*>(info.tokens[0].get())->text;

      if (StartsWith(firstTokenText, mePrefix)) {
        firstTokenText = firstTokenText.substr(mePrefix.size());
        info.flags.action = true;
      }
    }

    return true;
  }
};

struct ttv::chat::json::PubSubChatRoomMessageSenderSchema {
  static bool Parse(const ttv::json::Value& value, ttv::chat::MessageInfo& message) {
    if (value.isNull()) {
      return false;
    }

    // parse badges
    const auto& jBadges = value["badges"];
    if (jBadges.isNull() || !jBadges.isArray()) {
      return false;
    }
    for (const auto& jBadge : jBadges) {
      MessageBadge badge;
      if (ParseString(jBadge, "id", badge.name) && ParseString(jBadge, "version", badge.version)) {
        // Badges are currently the only way to determine UserModes through pubsub
        if (badge.name == "moderator") {
          message.userMode.moderator = true;
        } else if (badge.name == "broadcaster") {
          message.userMode.broadcaster = true;
        } else if (badge.name == "staff") {
          message.userMode.staff = true;
        } else if (badge.name == "admin") {
          message.userMode.administrator = true;
        } else if (badge.name == "global_mod") {
          message.userMode.globalModerator = true;
        } else if (badge.name == "vip") {
          message.userMode.vip = true;
        }

        message.badges.emplace_back(std::move(badge));
      }
    }

    if (!ParseString(value, "display_name", message.displayName)) {
      return false;
    }

    if (!ParseString(value, "login", message.userName)) {
      return false;
    }

    if (!ParseColor(value, "chat_color", message.nameColorARGB)) {
      message.nameColorARGB = GetRandomUserColor(message.userName);
    }

    const auto& jId = value["user_id"];
    if (!ParseUserId(jId, message.userId)) {
      return false;
    }

    return true;
  }
};

struct ttv::chat::json::GraphQLChatRoomMessageContentSchema {
  static bool Parse(const ttv::json::Value& value, MessageInfo& info) {
    if (value.isNull()) {
      return false;
    }

    const auto& jFragments = value["fragments"];
    if (jFragments.isNull() || !jFragments.isArray()) {
      return false;
    }

    bool firstFragment = true;
    for (const auto& jFragment : jFragments) {
      const auto& jText = jFragment["text"];
      if (!jText.isNull() && jText.isString()) {
        bool parsed = false;
        std::string text = jText.asString();
        const auto& jContent = jFragment["content"];
        if (!jContent.isNull() && jContent.isObject()) {
          std::string type;
          ParseString(jContent, "__typename", type);
          if (type == "Emote") {
            std::string emoticonId;
            if (ParseEmoticonId(jContent, "id", emoticonId)) {
              info.tokens.emplace_back(std::make_unique<ttv::chat::EmoticonToken>(text, emoticonId));
              parsed = true;
            }
          } else if (type == "User") {
            std::string userName;
            if (ParseString(jContent, "displayName", userName)) {
              info.tokens.emplace_back(std::make_unique<ttv::chat::MentionToken>(userName, text, false));
              parsed = true;
            }
          }
        } else if (ttv::chat::IsTwitchChatUrl(text)) {
          info.tokens.emplace_back(std::make_unique<ttv::chat::UrlToken>(text, false));
          parsed = true;
        }

        // If parsing as the other tokens failed, just parse it as normal text.
        if (!parsed) {
          // If the message starts with "/me ", handle the command.
          if (firstFragment) {
            std::string mePrefix = "/me ";

            if (StartsWith(text, mePrefix)) {
              text = text.substr(mePrefix.size());
              info.flags.action = true;
            }
          }
          info.tokens.emplace_back(std::make_unique<ttv::chat::TextToken>(text));
        }

        firstFragment = false;
      } else {
        return false;
      }
    }

    return true;
  }
};

struct ttv::chat::json::GraphQLChatRoomMessageSenderSchema {
  static bool Parse(const ttv::json::Value& value, ttv::chat::MessageInfo& message) {
    if (value.isNull()) {
      return false;
    }

    // Parse badges
    const auto& jBadges = value["displayBadges"];
    if (jBadges.isNull() || !jBadges.isArray()) {
      return false;
    }
    for (auto& jBadge : jBadges) {
      MessageBadge badge;
      if (ParseString(jBadge, "setID", badge.name) && ParseString(jBadge, "version", badge.version)) {
        // Badges are currently the only way to check if the sender is a moderator or broadcaster of the channel
        if (badge.name == "moderator") {
          message.userMode.moderator = true;
        } else if (badge.name == "broadcaster") {
          message.userMode.broadcaster = true;
        } else if (badge.name == "vip") {
          message.userMode.vip = true;
        }

        message.badges.emplace_back(std::move(badge));
      }
    }

    if (!ParseString(value, "displayName", message.displayName)) {
      return false;
    }

    if (!ParseString(value, "login", message.userName)) {
      return false;
    }

    if (!ParseColor(value, "chatColor", message.nameColorARGB)) {
      message.nameColorARGB = GetRandomUserColor(message.userName);
    }

    const auto& jId = value["id"];
    if (!ParseUserId(jId, message.userId)) {
      return false;
    }

    // Parse UserMode
    const auto& jRoles = value["roles"];
    bool isMode;
    ParseBool(jRoles, "isGlobalMod", isMode, false);
    message.userMode.globalModerator = isMode;
    ParseBool(jRoles, "isSiteAdmin", isMode, false);
    message.userMode.administrator = isMode;
    ParseBool(jRoles, "isStaff", isMode, false);
    message.userMode.staff = isMode;

    return true;
  }
};

struct ttv::chat::json::RoomMessageDeletedSchema {
  static bool Parse(const ttv::json::Value& value, MessageInfo::Flags& flags) {
    if (value.isNull()) {
      return false;
    }

    Timestamp time;
    if (ParseTimestamp(value, time)) {
      flags.deleted = true;
      return true;
    } else {
      return false;
    }
  }
};

struct ttv::chat::json::EmoticonTokenSchema {
  static bool Parse(const ttv::json::Value& value, Emoticon& emoticon) {
    if (value.isNull() || !value.isString()) {
      return false;
    }

    std::string token = value.asString();
    if (token == "") {
      return false;
      ;
    }

    UnescapeEmoticonToken(token);

    // Determine if this is a regex or a simple string match
    // See web/tmi/tmi/clue/emoticons.py
    static std::regex isRegexRegex = std::regex("[\\|\\\\\\^\\$\\*\\+\\?\\:\\#]");

    if (std::regex_search(token, isRegexRegex)) {
      try {
        emoticon.regex = std::regex(std::string("^") + token + std::string("$"));
        emoticon.match = token;
        emoticon.isRegex = true;
      } catch (std::exception& ex) {
        ttv::trace::Message("regex", ttv::MessageLevel::Error,
          "[EmoticonTokenSchema - Regex Error] [Pattern] %s [Error] %s", token.c_str(), ex.what());
        TTV_ASSERT(false, "regex", "Bad Regex");
        emoticon.match = token;
        emoticon.isRegex = false;
      }
    } else {
      emoticon.match = token;
      emoticon.isRegex = false;
    }

    return true;
  }
};

struct ttv::chat::json::EmoticonSetsSchema {
  static bool Parse(const ttv::json::Value& value, std::vector<EmoticonSet>& sets) {
    if (value.isNull() || !value.isArray()) {
      return false;
    }

    for (const auto& jEmoteSet : value) {
      EmoticonSet set;

      if (!ParseEmoticonId(jEmoteSet, "emoteSetID", set.emoticonSetId)) {
        break;
      }

      const auto& jEmotes = jEmoteSet["emotes"];
      if (jEmotes.isNull() || !jEmotes.isArray()) {
        break;
      }

      for (const auto& jEmote : jEmotes) {
        Emoticon emote;
        if (ttv::json::ToObject(jEmote, emote)) {
          set.emoticons.emplace_back(std::move(emote));
        }
      }

      sets.emplace_back(std::move(set));
    }

    return true;
  }
};
