/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/core/pubsub/pubsubcomponent.h"
#include "twitchsdk/core/user/user.h"

namespace ttv {
class User;

namespace chat {
class UserEmoticonSets;
class UserEmoticonSetsListener;
class UserEmoticonSetsListenerProxy;
}  // namespace chat
}  // namespace ttv

/**
 * Fetches and keeps track of the list of emoticons available for a user.
 */
class ttv::chat::UserEmoticonSets : public PubSubComponent<UserEmoticonSetsListener> {
 public:
  using FetchEmoticonSetsCallback = std::function<void(TTV_ErrorCode ec, const std::vector<EmoticonSet>& result)>;

  UserEmoticonSets(const std::shared_ptr<User>& user, const TokenizationOptions& options);

  // Component overrides
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;

  virtual std::string GetLoggerName() const override;
  static std::string GetComponentName() { return "ttv::chat::UserEmoticonSets"; }

  /**
   * Returns the user's currently available emoticon sets.
   */
  TTV_ErrorCode GetUserEmoticonSets(std::vector<EmoticonSet>& sets);

  /**
   * Fetches and returns the user's currently available emoticon sets.
   *
   * @param[in] forceFetch If true, will force a refetch for the latest values. If false, will returned currently stored
   * values.
   * @param[in] callback Function to be called that will take in the list of user emoticon sets.
   */
  TTV_ErrorCode FetchUserEmoticonSets(bool forceFetch, const FetchEmoticonSetsCallback& callback);

  /**
   * Returns the currently configured tokenization options.
   */
  TokenizationOptions GetTokenizationOptions() const { return mTokenizationOptions; }

  void OnTopicSubscribeStateChanged(
    const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) override;

  /**
   * Listens for new subscriptions to kick off a fetch for the newest emoticons.
   */
  void OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) override;

 private:
  CallbackQueue<FetchEmoticonSetsCallback> mFetchUserEmoticonSetsCallbacks;
  std::vector<EmoticonSet> mUserEmoticonSets;

  std::string mPubSubTopic;
  RetryTimer mFetchRetryTimer;
  TokenizationOptions mTokenizationOptions;

  bool mEmoticonFetchInFlight;
};

/**
 * The listener interface for events originating in UserEmoticonSets.
 */
class ttv::chat::UserEmoticonSetsListener {
 public:
  virtual ~UserEmoticonSetsListener() = default;
  virtual void OnUserEmoticonSetsChanged(UserId userId, const std::vector<EmoticonSet>& sets) = 0;
};

/**
 * A lambda proxy for UserEmoticonSets::Listener.
 */
class ttv::chat::UserEmoticonSetsListenerProxy : public UserEmoticonSetsListener {
 public:
  using UserEmoticonSetsChangedFunc = std::function<void(UserId userId, const std::vector<EmoticonSet>& emoticonSet)>;

 public:
  virtual void OnUserEmoticonSetsChanged(UserId userId, const std::vector<EmoticonSet>& emoticonSets) override {
    if (mUserEmoticonSetsChangedFunc != nullptr) {
      mUserEmoticonSetsChangedFunc(userId, emoticonSets);
    }
  }

 public:
  UserEmoticonSetsChangedFunc mUserEmoticonSetsChangedFunc;
};
