/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/
#pragma once

#include "twitchsdk/chat/chattypes.h"

#include <map>
#include <string>
#include <vector>

namespace ttv {
namespace chat {

enum TokenRangeEnum {
  TOKEN_RANGE_TYPE_URL,
  TOKEN_RANGE_TYPE_MENTION,
  TOKEN_RANGE_TYPE_EMOTICON,
  TOKEN_RANGE_TYPE_BITS
};

struct EmoticonRange {
  std::string emoticonId;
};

struct MentionRange {};

struct UrlRange {};

struct BitsRange {
  uint32_t prefixLength;
  uint32_t numBits;
};

struct TokenRange {
  TokenRangeEnum type;
  int startIndex;
  int endIndex;  // inclusive
  int rangeNum;  // To stable sort
  union {
    MentionRange mention;
    UrlRange url;
    BitsRange bits;
  } data;

  // string in union needs additional overloaded cases
  EmoticonRange emoticon;
};

struct EmoteRange {
  int startIndex;
  int endIndex;
};

struct AutoModFlagsRange {
  AutoModFlags flags;
  uint32_t startIndex;
  //!< Inclusive, the character at this index is part of the flagged section
  uint32_t endIndex;
};

struct RangeBase {
  RangeBase(int start, int stop) : startIndex(start), stopIndex(stop) {}
  int startIndex;
  int stopIndex;
};

namespace tokenranges {

/*
 * Converts ranges from utf8 indicies to byte indicies inclusive
 * startIndex = 0, stopEnd = 0, length = 1
 */
template <typename T>
std::vector<T> ConvertUtf8RangesToByteRanges(const std::vector<T>& utf8Ranges, const std::string& message);

extern template std::vector<AutoModFlagsRange> ConvertUtf8RangesToByteRanges(
  const std::vector<AutoModFlagsRange>& utf8Ranges, const std::string& message);

extern template std::vector<TokenRange> ConvertUtf8RangesToByteRanges(
  const std::vector<TokenRange>& utf8Ranges, const std::string& message);
/*
 * Converts ranges from byte indicies to utf8 indicies inclusive
 * startIndex = 0, stopEnd = 0, length = 1
 */
template <typename T>
std::vector<T> ConvertByteRangesToUtf8Ranges(const std::vector<T>& byteRanges, const std::string& message);

extern template std::vector<TokenRange> ConvertByteRangesToUtf8Ranges(
  const std::vector<TokenRange>& utf8Ranges, const std::string& message);

/**
 * Converts EmoticonRange sets to TokenRange.  The index number will be the same order as |ranges|.  |messageLength| is
 * the length of the original message body equivlent to message.length()
 * The returned TokenRanges are sorted by startIndex
 */
std::vector<TokenRange> ConvertToTokenRanges(
  const std::map<std::string, std::vector<EmoteRange>>& emoticonCollections, size_t messageLength);

bool SortTokenRangesByTypeThenRangeNum(const TokenRange& a, const TokenRange& b);
bool SortTokenRangesByStartIndex(const TokenRange& a, const TokenRange& b);
bool SortEmoticonRangesByEmoticonId(const TokenRange& a, const TokenRange& b);

void RemoveOverlappingRanges(std::vector<TokenRange>& ranges);

}  // namespace tokenranges
}  // namespace chat
}  // namespace ttv
