#include "context.h"
#include "staticinit.h"
#include "stringutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ibitslistener.h"
#include "twitchsdk/chat/ibitsstatus.h"
#include "twitchsdk/chat/ichannelchatroommanager.h"
#include "twitchsdk/chat/ichannelchatroommanagerlistener.h"
#include "twitchsdk/chat/ichatchannel.h"
#include "twitchsdk/chat/ichatchannelproperties.h"
#include "twitchsdk/chat/ichatchannelpropertylistener.h"
#include "twitchsdk/chat/ichatcommentmanager.h"
#include "twitchsdk/chat/ichatraid.h"
#include "twitchsdk/chat/ichatraidlistener.h"
#include "twitchsdk/chat/ichatroom.h"
#include "twitchsdk/chat/ichatroomlistener.h"
#include "twitchsdk/chat/ichatroomnotifications.h"
#include "twitchsdk/chat/ichatroomnotificationslistener.h"
#include "twitchsdk/chat/ifollowerslistener.h"
#include "twitchsdk/chat/ifollowersstatus.h"
#include "twitchsdk/chat/ifollowinglistener.h"
#include "twitchsdk/chat/imultiviewnotifications.h"
#include "twitchsdk/chat/imultiviewnotificationslistener.h"
#include "twitchsdk/chat/internal/graphql/fetchchannelcheermotesqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/grantvipqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/revokevipqueryinfo.h"
#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"
#include "twitchsdk/chat/isquadnotifications.h"
#include "twitchsdk/chat/isquadnotificationslistener.h"
#include "twitchsdk/chat/isubscriberslistener.h"
#include "twitchsdk/chat/isubscribersstatus.h"
#include "twitchsdk/chat/isubscriptionsnotifications.h"
#include "twitchsdk/chat/isubscriptionsnotificationslistener.h"

#include <sstream>

using namespace ttv;
using namespace ttv::chat;

namespace ttv {
std::string ToString(ChatChannelState state) {
  switch (state) {
    case ChatChannelState::Disconnected:
      return "disconnected";
    case ChatChannelState::Connecting:
      return "connecting";
    case ChatChannelState::Connected:
      return "connected";
    case ChatChannelState::Disconnecting:
      return "disconnecting";
    default:
      return "???";
  }
}

std::string ToString(const UserMode& mode) {
  return CommaSeparatedStrings(
    {
      {mode.moderator, "mod"},
      {mode.broadcaster, "broadcaster"},
      {mode.administrator, "admin"},
      {mode.staff, "staff"},
      {mode.system, "system"},
      {mode.globalModerator, "globalmod"},
      {mode.banned, "banned"},
      {mode.subscriber, "sub"},
      {mode.vip, "vip"},
    },
    "viewer");
}

std::string ToString(const RestrictionReason& restriction) {
  return CommaSeparatedStrings(
    {{restriction.anonymous, "anonymous"}, {restriction.subscribersOnly, "subscribersonly"},
      {restriction.slowMode, "slowmode"}, {restriction.timeout, "timeout"}, {restriction.banned, "banned"}},
    "none");
}

std::string ToString(const ChatRoomPermissions& permissions) {
  return CommaSeparatedStrings({{permissions.readMessages, "read messages"},
                                 {permissions.sendMessages, "send messages"}, {permissions.moderate, "moderate"}},
    "none");
}

std::string ToString(BadgeVersion::Action action) {
  switch (action) {
    case BadgeVersion::Action::None:
      return "none";
    case BadgeVersion::Action::Subscribe:
      return "subscribe_to_channel";
    case BadgeVersion::Action::VisitUrl:
      return "visit_url";
    default:
      return "???";
  }
}

std::string ToString(SubscriptionNotice::Type type) {
  switch (type) {
    case SubscriptionNotice::Type::Sub:
      return "sub";
    case SubscriptionNotice::Type::Resub:
      return "resub";
    case SubscriptionNotice::Type::SubGift:
      return "subgift";
    case SubscriptionNotice::Type::Charity:
      return "charity";
    case SubscriptionNotice::Type::ExtendSub:
      return "extendsub";
    default:
      return "unknown";
  }
}

std::string ToString(SubscriptionNotice::Plan plan) {
  switch (plan) {
    case SubscriptionNotice::Plan::Prime:
      return "Prime";
    case SubscriptionNotice::Plan::Sub1000:
      return "1000";
    case SubscriptionNotice::Plan::Sub2000:
      return "2000";
    case SubscriptionNotice::Plan::Sub3000:
      return "3000";
    default:
      return "unknown";
  }
}

std::string ToString(ChatCommentSource source) {
  switch (source) {
    case ChatCommentSource::Comment:
      return "comment";
    case ChatCommentSource::Chat:
      return "chat";
    default:
      return "unknown";
  }
}

std::string ToString(ChatCommentPublishedState state) {
  switch (state) {
    case ChatCommentPublishedState::Published:
      return "published";
    case ChatCommentPublishedState::Unpublished:
      return "unpublished";
    case ChatCommentPublishedState::PendingReview:
      return "pending review";
    case ChatCommentPublishedState::PendingReviewSpam:
      return "pending review spam";
    case ChatCommentPublishedState::Deleted:
      return "deleted";
    default:
      return "unknown";
  }
}

std::string ToString(IChatCommentManager::PlayingState action) {
  switch (action) {
    case IChatCommentManager::PlayingState::Paused:
      return "paused";
    case IChatCommentManager::PlayingState::Playing:
      return "playing";
    case IChatCommentManager::PlayingState::Buffering:
      return "buffering";
    case IChatCommentManager::PlayingState::Finished:
      return "finished";
    default:
      return "unknown";
  }
}

std::string ToString(CommentPublishingMode mode) {
  switch (mode) {
    case CommentPublishingMode::Open:
      return "open";
    case CommentPublishingMode::Review:
      return "review";
    case CommentPublishingMode::Disabled:
      return "disabled";
    default:
      return "unknown";
  }
}

std::string ToString(RoomRole role) {
  switch (role) {
    case RoomRole::Everyone:
      return "Everyone";
    case RoomRole::Subscriber:
      return "Subscriber";
    case RoomRole::Moderator:
      return "Moderator";
    case RoomRole::Broadcaster:
      return "Broadcaster";
    default:
      return "Unknown";
  }
}

std::string ToString(BitsConfiguration::Cheermote::Type type) {
  switch (type) {
    case BitsConfiguration::Cheermote::Type::Custom:
      return "custom";
    case BitsConfiguration::Cheermote::Type::Sponsored:
      return "sponsored";
    case BitsConfiguration::Cheermote::Type::FirstParty:
      return "first party";
    case BitsConfiguration::Cheermote::Type::ThirdParty:
      return "third party";
    case BitsConfiguration::Cheermote::Type::DisplayOnly:
      return "display only";
    default:
      return "unknown";
  }
}

std::string ToString(BitsConfiguration::CheermoteImage::Theme theme) {
  switch (theme) {
    case BitsConfiguration::CheermoteImage::Theme::Dark:
      return "dark";
    case BitsConfiguration::CheermoteImage::Theme::Light:
      return "light";
    default:
      return "unknown";
  }
}

void Print(const std::unique_ptr<MessageToken>& token, const std::string& /*indent*/) {
  switch (token->GetType()) {
    case MessageToken::Type::Text: {
      TextToken* textToken = static_cast<TextToken*>(token.get());
      std::cout << "[Text: \"" << textToken->text << "\"";

      if (textToken->autoModFlags.HasFlag()) {
        std::cout << " Flags: I." << textToken->autoModFlags.identityLevel << "/S."
                  << textToken->autoModFlags.sexualLevel << "/A." << textToken->autoModFlags.aggressiveLevel << "/P."
                  << textToken->autoModFlags.profanityLevel;
      }

      std::cout << "]";
      break;
    }
    case MessageToken::Type::Emoticon:
      std::cout << "[Emote: " << static_cast<EmoticonToken*>(token.get())->emoticonId << " "
                << static_cast<EmoticonToken*>(token.get())->emoticonText << "]";
      break;
    case MessageToken::Type::Mention:
      std::cout << "[Mention: " << static_cast<MentionToken*>(token.get())->userName << "]";
      break;
    case MessageToken::Type::Url:
      std::cout << "[Url: " << static_cast<UrlToken*>(token.get())->url
                << " Hidden: " << static_cast<UrlToken*>(token.get())->hidden << "]";
      break;
    case MessageToken::Type::Bits:
      std::cout << "[Bits: (" << static_cast<BitsToken*>(token.get())->numBits << ") "
                << static_cast<BitsToken*>(token.get())->prefix << "]";
      break;
  }
}

void PrintCommonMessageFields(const MessageInfo& msg, std::string indent) {
  std::cout << indent << PRINT_FIELD(msg, displayName) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(msg, timestamp) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, userId) << std::endl;
  std::cout << indent << "modes: " << ToString(msg.userMode) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, numBitsSent) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, messageType) << std::endl;

  // Badges
  if (msg.badges.size() > 0) {
    std::cout << indent << "badges[" << msg.badges.size() << "]: ";
    bool isFirst = true;
    for (const auto& badge : msg.badges) {
      if (isFirst) {
        isFirst = false;
      } else {
        std::cout << ", ";
      }

      std::cout << badge.name << "/" << badge.version;
    }

    std::cout << std::endl;
  } else {
    std::cout << indent << "badges[0]" << std::endl;
  }

  // Tokens
  std::cout << indent << "message: ";
  for (const auto& token : msg.tokens) {
    Print(token, indent);
  }

  std::cout << std::endl;

  // Flags
  std::cout << indent << "flags: ";
  if (msg.flags.action) {
    std::cout << "action ";
  }
  if (msg.flags.notice) {
    std::cout << "notice ";
  }
  if (msg.flags.ignored) {
    std::cout << "ignored ";
  }
  if (msg.flags.deleted) {
    std::cout << "deleted ";
  }
  if (msg.flags.containsBits) {
    std::cout << "containsBits ";
  }
  std::cout << std::endl;
}

void Print(const MessageInfo& msg, std::string indent) {
  std::cout << indent << "MessageInfo: " << msg.userName << '<' << msg.userId << '>' << std::endl;
  indent += "  ";

  PrintCommonMessageFields(msg, indent);
}

void Print(const LiveChatMessage& msg, std::string indent) {
  std::cout << indent << "LiveChatMessage: " << msg.messageInfo.userName << '<' << msg.messageInfo.userId << '>'
            << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(msg, messageId) << std::endl;

  PrintCommonMessageFields(msg.messageInfo, indent);
}

void Print(const std::vector<LiveChatMessage>& list, const std::string& indent) {
  std::cout << indent << "Live Chat Messages (" << list.size() << "): " << std::endl;
  for (const auto& message : list) {
    Print(message, indent + "  ");
  }
}

void Print(const WhisperMessage& msg, std::string indent) {
  std::cout << indent << "WhisperMessage: " << msg.messageInfo.userName << '<' << msg.messageInfo.userId << '>'
            << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(msg, threadId) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, messageId) << std::endl;

  PrintCommonMessageFields(msg.messageInfo, indent);
}

void Print(const std::vector<WhisperMessage>& list, const std::string& indent) {
  std::cout << indent << "Whispers (" << list.size() << "): " << std::endl;
  for (const auto& message : list) {
    Print(message, indent + "  ");
  }
}

void Print(const SubscriptionNotice& notice, std::string indent) {
  std::cout << indent << "SubscriptionNotice:" << std::endl;
  indent += "  ";

  std::cout << indent << "type: " << ToString(notice.type) << std::endl;
  std::cout << indent << "plan: " << ToString(notice.plan) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, planDisplayName) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, messageId) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, subCumulativeMonthCount) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, subStreakMonthCount) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, systemMessage) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, shouldShowSubStreak) << std::endl;

  if (notice.type == SubscriptionNotice::Type::SubGift) {
    std::cout << indent << PRINT_FIELD(notice, recipient.displayName) << std::endl;
    std::cout << indent << PRINT_FIELD(notice, recipient.userName) << std::endl;
    std::cout << indent << PRINT_FIELD(notice, recipient.userId) << std::endl;
    std::cout << indent << PRINT_FIELD(notice, senderCount) << std::endl;
  }

  if (notice.userMessage != nullptr) {
    std::cout << indent << "message:" << std::endl;

    indent += "  ";
    Print(*notice.userMessage, indent);
  }
}

void Print(const FirstTimeChatterNotice& notice, std::string indent) {
  std::cout << indent << "FirstTimeChatterNotice:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(notice, messageId) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, systemMessage) << std::endl;

  std::cout << indent << "message:" << std::endl;

  indent += "  ";
  Print(notice.userMessage, indent);
}

void Print(const RaidNotice& notice, std::string indent) {
  std::cout << indent << "RaidNotice:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(notice, systemMessage) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, profileImageUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(notice, viewerCount) << std::endl;
  std::cout << indent << "raidingUserInfo:" << std::endl;
  indent += "  ";
  Print(notice.raidingUserInfo, indent);
}

void Print(const UnraidNotice& notice, std::string indent) {
  std::cout << indent << "UnraidNotice:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(notice, systemMessage) << std::endl;
}

void Print(const GenericMessageNotice& notice, std::string indent) {
  std::cout << indent << "GenericNotice:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(notice, messageId) << std::endl;
  PrintCommonMessageFields(notice.messageInfo, indent);
}

void Print(const ChatComment& msg, std::string indent) {
  std::cout << indent << "ChatComment: " << msg.messageInfo.userName << '<' << msg.messageInfo.userId << '>'
            << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(msg, commentId) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, channelId) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, contentId) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, parentCommentId) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, timestampMilliseconds) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(msg, updatedAt) << std::endl;
  std::cout << indent << "comment source: " << ToString(msg.commentSource) << std::endl;
  std::cout << indent << "published state: " << ToString(msg.publishedState) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, moreReplies) << std::endl;

  PrintCommonMessageFields(msg.messageInfo, indent);

  std::cout << indent << "replies: " << std::endl;
  for (const auto& reply : msg.replies) {
    Print(reply, indent + "  ");
  }
}

void Print(const std::vector<ChatComment>& list, const std::string& indent) {
  std::cout << indent << "VOD Comments (" << list.size() << "): " << std::endl;
  for (const auto& message : list) {
    Print(message, indent + "  ");
  }
}

void Print(const ChannelVodCommentSettings& settings, std::string indent) {
  std::cout << indent << "VOD Comment Settings for channel " << settings.channelId << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(settings, channelId) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(settings, createdAt) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(settings, updatedAt) << std::endl;
  std::cout << indent << PRINT_FIELD(settings, followersOnlyDurationSeconds) << std::endl;
  std::cout << indent << "channel comment publishing mode: " << ToString(settings.publishingMode) << std::endl;
}

void Print(const ChatRoomMessage& msg, std::string indent) {
  std::cout << indent << "ChatRoomMessage: " << msg.messageInfo.userName << '<' << msg.messageInfo.userId << '>'
            << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(msg, roomMessageId) << std::endl;
  std::cout << indent << PRINT_FIELD(msg, roomId) << std::endl;

  PrintCommonMessageFields(msg.messageInfo, indent);
}

void Print(const std::vector<ChatRoomMessage>& messages, const std::string& indent) {
  std::cout << indent << "ChatRoomMessages (" << messages.size() << "): " << std::endl;
  for (const auto& message : messages) {
    Print(message, indent + "  ");
  }
}

void Print(const ChatModeInfo mode, std::string& indent) {
  std::cout << indent << "ChatModeInfo:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(mode, slowModeDurationSeconds) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, r9kMode) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, emotesOnlyMode) << std::endl;
}

void Print(const ChatRoomView& view, std::string indent) {
  std::cout << indent << "ChatRoomView: " << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_TIMESTAMP_FIELD(view, lastReadAt) << std::endl;
  std::cout << indent << PRINT_FIELD(view, isMuted) << std::endl;
  std::cout << indent << PRINT_FIELD(view, isArchived) << std::endl;
  std::cout << indent << PRINT_FIELD(view, isUnread) << std::endl;
  std::cout << indent << PRINT_FIELD(view, unreadMentionCount) << std::endl;
  std::cout << indent << "permissions: " << ToString(view.permissions) << std::endl;
}

void Print(const ChatRoomInfo& info, std::string indent) {
  std::cout << indent << "ChatRoomInfo: " << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(info, id) << std::endl;
  std::cout << indent << PRINT_FIELD(info, name) << std::endl;
  std::cout << indent << PRINT_FIELD(info, topic) << std::endl;
  std::cout << indent << "Minimum read room role: " << ToString(info.rolePermissions.read) << std::endl;
  std::cout << indent << "Minimum send room role: " << ToString(info.rolePermissions.send) << std::endl;

  Print(info.modes, indent);
  Print(info.view, indent);
  Print(info.owner, indent);
}

void Print(const std::vector<ChatRoomInfo>& infos, const std::string& indent) {
  std::cout << indent << "ChatRoomInfos (" << infos.size() << "): " << std::endl;
  for (const auto& info : infos) {
    Print(info, indent + "  ");
  }
}

void Print(const RoomMentionInfo& mentionInfo, std::string indent) {
  std::cout << indent << "RoomMentionInfo: " << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(mentionInfo, roomOwnerId) << std::endl;
  std::cout << indent << PRINT_FIELD(mentionInfo, roomOwnerName) << std::endl;
  std::cout << indent << PRINT_FIELD(mentionInfo, roomOwnerLogin) << std::endl;
  std::cout << indent << PRINT_FIELD(mentionInfo, senderId) << std::endl;
  std::cout << indent << PRINT_FIELD(mentionInfo, senderName) << std::endl;
  std::cout << indent << PRINT_FIELD(mentionInfo, roomId) << std::endl;
  std::cout << indent << PRINT_FIELD(mentionInfo, roomName) << std::endl;
  std::cout << indent << PRINT_FIELD(mentionInfo, messageId) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(mentionInfo, sentAt) << std::endl;
}

void Print(const EmoticonSet& emoticonSet, std::string indent) {
  std::cout << indent << "EmoticonSet: " << emoticonSet.emoticonSetId << std::endl;
  indent += "  ";

  std::cout << indent << "emoticons:" << std::endl;

  for (const auto& emote : emoticonSet.emoticons) {
    std::cout << indent << "  " << emote.emoticonId << ": " << emote.match;
    if (emote.isRegex) {
      std::cout << " (regex)";
    }
    std::cout << std::endl;
  }
}

void Print(const std::vector<EmoticonSet>& emoticonSets, std::string indent) {
  std::cout << indent << "EmoticonSets: [" << emoticonSets.size() << "]" << std::endl;
  indent += "  ";

  for (const auto& set : emoticonSets) {
    Print(set, indent);
  }
}

void Print(const ChatChannelInfo& info, std::string indent) {
  std::cout << indent << "ChannelInfo: " << info.name << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(info, broadcasterLanguage) << std::endl;
  std::cout << indent << "localUserRestriction: " << ToString(info.localUserRestriction) << std::endl;
  std::cout << indent << PRINT_FIELD(info, broadcasterLanguage) << std::endl;
}

void Print(const ChatChannelRestrictions& restrictions, std::string indent) {
  std::cout << indent << "ChatRestrictions: " << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(restrictions, emoteOnly) << std::endl;
  std::cout << indent << PRINT_FIELD(restrictions, verifiedOnly) << std::endl;
  std::cout << indent << PRINT_FIELD(restrictions, followersOnly) << std::endl;
  std::cout << indent << PRINT_FIELD(restrictions, followersDuration) << std::endl;
  std::cout << indent << PRINT_FIELD(restrictions, subscribersOnly) << std::endl;
  std::cout << indent << PRINT_FIELD(restrictions, slowMode) << std::endl;
  std::cout << indent << PRINT_FIELD(restrictions, slowModeDuration) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(restrictions, slowModeSetAt) << std::endl;
  std::cout << indent << PRINT_FIELD(restrictions, r9k) << std::endl;
}

void Print(const UserMode& mode, std::string indent) {
  std::cout << indent << "UserModes: " << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(mode, moderator) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, broadcaster) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, administrator) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, staff) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, system) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, globalModerator) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, banned) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, subscriber) << std::endl;
  std::cout << indent << PRINT_FIELD(mode, vip) << std::endl;
}

void Print(const ChatUserInfo& user, std::string indent) {
  std::cout << indent << "User: " << user.userName << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(user, displayName) << std::endl;
  std::cout << indent << PRINT_FIELD(user, nameColorARGB) << std::endl;
  std::cout << indent << PRINT_FIELD(user, userId) << std::endl;

  Print(user.userMode, indent);
}

void Print(const std::vector<ChatUserInfo>& list, const std::string& indent) {
  std::cout << indent << "Users (" << list.size() << "): " << std::endl;
  for (const auto& info : list) {
    Print(info, indent + "  ");
  }
}

void Print(const BitsConfiguration::CheermoteTier tier, const std::string& indent) {
  std::cout << indent << PRINT_FIELD(tier, tierID) << std::endl;
  std::cout << indent << PRINT_FIELD(tier, bits) << std::endl;
  std::cout << indent << PRINT_FIELD(tier, color) << std::endl;
  std::cout << indent << PRINT_FIELD(tier, canCheer) << std::endl;
  std::cout << indent << PRINT_FIELD(tier, canShowInBitsCard) << std::endl;
  std::cout << indent << "Number of images: " << tier.images.size() << std::endl;
}

void Print(const BadgeImage& image, const std::string& indent) {
  std::cout << indent << "scale, url: " << image.scale << ", " << image.url << std::endl;
}

void Print(const BadgeVersion& version, std::string indent) {
  std::cout << indent << PRINT_FIELD(version, name) << std::endl;
  std::cout << indent << PRINT_FIELD(version, title) << std::endl;
  std::cout << indent << PRINT_FIELD(version, description) << std::endl;
  std::cout << indent << "clickAction: " << ToString(version.clickAction) << std::endl;
  std::cout << indent << PRINT_FIELD(version, clickUrl) << std::endl;

  std::cout << indent << "images[" << version.images.size() << "]:" << std::endl;
  indent += "  ";

  for (const auto& image : version.images) {
    Print(image, indent);
  }
}

void Print(const Badge& badge, std::string indent) {
  std::cout << indent << PRINT_FIELD(badge, name) << std::endl;

  std::cout << indent << "versions[" << badge.versions.size() << "]:" << std::endl;
  indent += "  ";

  for (const auto& kvp : badge.versions) {
    Print(kvp.second, indent);
  }
}

void Print(const BadgeSet& badgeSet, std::string indent) {
  std::cout << indent << PRINT_FIELD(badgeSet, language) << std::endl;

  std::cout << indent << "badges[" << badgeSet.badges.size() << "]:" << std::endl;
  indent += "  ";

  for (const auto& kvp : badgeSet.badges) {
    Print(kvp.second, indent);
  }
}

void Print(const std::shared_ptr<const BitsConfiguration>& config, std::string indent) {
  if (config == nullptr) {
    std::cout << indent << "null" << std::endl;
    return;
  }

  std::cout << indent << "user: " << config->GetUserId() << ", channel: " << config->GetChannelId() << std::endl;

  std::cout << indent << "cheermotes: " << std::endl;
  indent += "  ";

  for (const auto& cheermote : config->GetCheermotes()) {
    std::cout << indent << PRINT_FIELD(cheermote, prefix) << std::endl;
    std::cout << indent << "imageTiers[" << cheermote.tiers.size() << "]" << std::endl;
    for (const auto& tier : cheermote.tiers) {
      Print(tier, indent + "  ");
    }

    std::cout << indent << "type: " << ToString(cheermote.type) << std::endl;

    std::cout << std::endl << std::endl;
  }
}

void Print(const UnreadThreadCounts& counts, std::string indent) {
  std::cout << indent << "UnreadThreadCounts:" << std::endl;

  indent += "  ";

  std::cout << indent << PRINT_VARIABLE(counts.unreadThreadCount) << std::endl;
  std::cout << indent << PRINT_VARIABLE(counts.unreadMessageCount) << std::endl;
  std::cout << indent << PRINT_VARIABLE(counts.exhaustive) << std::endl;
}

void Print(const RaidStatus& status, std::string indent) {
  std::cout << indent << "RaidStatus:" << std::endl;

  indent += "  ";

  std::cout << indent << PRINT_VARIABLE(status.raidId) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.creatorUserId) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.sourceChannelId) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.targetChannelId) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.targetUserLogin) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.targetUserDisplayName) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.targetUserProfileImageUrl) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.transitionJitterSeconds) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.numUsersInRaid) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.forceRaidNowSeconds) << std::endl;
  std::cout << indent << PRINT_VARIABLE(status.joined) << std::endl;
}

void Print(const BadgeEntitlement& badgeEntitlement, std::string indent) {
  std::cout << indent << PRINT_FIELD(badgeEntitlement, isNewBadgeLevel) << std::endl;
  std::cout << indent << PRINT_FIELD(badgeEntitlement, previousLevel) << std::endl;
  std::cout << indent << PRINT_FIELD(badgeEntitlement, newLevel) << std::endl;
}

void Print(const BitsReceivedEvent& bitsReceivedEvent, std::string indent) {
  std::cout << indent << PRINT_FIELD(bitsReceivedEvent, channelName) << std::endl;
  std::cout << indent << PRINT_FIELD(bitsReceivedEvent, context) << std::endl;
  std::cout << indent << PRINT_FIELD(bitsReceivedEvent, channelId) << std::endl;
  std::cout << indent << PRINT_FIELD(bitsReceivedEvent, bitsUsed) << std::endl;
  std::cout << indent << PRINT_FIELD(bitsReceivedEvent, totalBitsUsed) << std::endl;

  Print(bitsReceivedEvent.badge, indent);

  std::cout << indent << "Message: " << std::endl;
  indent += "  ";
  Print(bitsReceivedEvent.message, indent);
}

void Print(const BitsSentEvent& bitsSentEvent, std::string indent) {
  std::cout << indent << "BitsSentEvent: " << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(bitsSentEvent, channelId) << std::endl;
  std::cout << indent << PRINT_FIELD(bitsSentEvent, userBitsBalance) << std::endl;
  std::cout << indent << PRINT_FIELD(bitsSentEvent, channelBitsTotal) << std::endl;
}

void Print(const FollowerAddedEvent& followerAddedEvent, std::string indent) {
  std::cout << indent << PRINT_FIELD(followerAddedEvent, userId) << std::endl;
  std::cout << indent << PRINT_FIELD(followerAddedEvent, displayName) << std::endl;
  std::cout << indent << PRINT_FIELD(followerAddedEvent, userName) << std::endl;
}

void Print(const SubscriberAddedEvent& subscriberAddedEvent, std::string indent) {
  std::cout << indent << PRINT_FIELD(subscriberAddedEvent, userName) << std::endl;
  std::cout << indent << PRINT_FIELD(subscriberAddedEvent, displayName) << std::endl;
  std::cout << indent << PRINT_FIELD(subscriberAddedEvent, channelName) << std::endl;

  std::cout << indent << PRINT_FIELD(subscriberAddedEvent, userId) << std::endl;
  std::cout << indent << PRINT_FIELD(subscriberAddedEvent, channelId) << std::endl;
  std::cout << indent << PRINT_FIELD(subscriberAddedEvent, timestamp) << std::endl;

  indent += "  ";
  Print(subscriberAddedEvent.subNotice, indent);
}

void Print(const SendRoomMessageError& error, std::string indent) {
  std::cout << indent << "SendRoomMessageError:" << std::endl;
  indent += "  ";

  std::string errorString;
  ttv::json::ToJsonString(error.code, errorString);
  std::cout << indent << "error: " << errorString << std::endl;
  std::cout << indent << PRINT_FIELD(error, remainingDurationSeconds) << std::endl;
  std::cout << indent << PRINT_FIELD(error, slowModeDurationSeconds) << std::endl;
}

void Print(const CreateRoomError& error, std::string indent) {
  std::cout << indent << "CreateRoomError:" << std::endl;
  indent += "  ";

  std::string errorString;
  ttv::json::ToJsonString(error.code, errorString);
  std::cout << indent << "error: " << errorString;
  std::cout << indent << PRINT_FIELD(error, maxAllowedRooms) << std::endl;
  std::cout << indent << PRINT_FIELD(error, maxLength) << std::endl;
  std::cout << indent << PRINT_FIELD(error, minLength) << std::endl;
}

void Print(const UpdateRoomError& error, std::string indent) {
  std::cout << indent << "UpdateRoomError:" << std::endl;
  indent += "  ";

  std::string errorString;
  ttv::json::ToJsonString(error.code, errorString);
  std::cout << indent << "error: " << errorString;
  std::cout << indent << PRINT_FIELD(error, minLength) << std::endl;
  std::cout << indent << PRINT_FIELD(error, maxLength) << std::endl;
}

void Print(const BanUserError& error, std::string indent) {
  std::cout << indent << "BanUserError:" << std::endl;
  indent += "  ";

  std::string errorString;
  ttv::json::ToJsonString(error.code, errorString);
  std::cout << indent << "error: " << errorString << std::endl;
}

void Print(const UnbanUserError& error, std::string indent) {
  std::cout << indent << "UnbanUserError:" << std::endl;
  indent += "  ";

  std::string errorString;
  ttv::json::ToJsonString(error.code, errorString);
  std::cout << indent << "error: " << errorString << std::endl;
}

void Print(const ModUserError& error, std::string indent) {
  std::cout << indent << "ModUserError:" << std::endl;
  indent += "  ";

  std::string errorString;
  ttv::json::ToJsonString(error.code, errorString);
  std::cout << indent << "error: " << errorString << std::endl;
}

void Print(const UnmodUserError& error, std::string indent) {
  std::cout << indent << "UnmodUserError:" << std::endl;
  indent += "  ";

  std::string errorString;
  ttv::json::ToJsonString(error.code, errorString);
  std::cout << indent << "error: " << errorString << std::endl;
}

void Print(GrantVIPErrorCode error, std::string indent) {
  std::string errorString;
  ttv::json::ToJsonString(error, errorString);
  std::cout << indent << "GrantVIPErrorCode: " << errorString << std::endl;
}

void Print(RevokeVIPErrorCode error, std::string indent) {
  std::string errorString;
  ttv::json::ToJsonString(error, errorString);
  std::cout << indent << "RevokeVIPErrorCode: " << errorString << std::endl;
}

void Print(const UpdateRoomModesError& error, std::string indent) {
  std::cout << indent << "UpdateRoomModesError:" << std::endl;
  indent += "  ";

  std::string errorString;
  ttv::json::ToJsonString(error.code, errorString);
  std::cout << indent << "error: " << errorString;
  std::cout << indent << PRINT_FIELD(error, minimumSlowModeDurationSeconds) << std::endl;
  std::cout << indent << PRINT_FIELD(error, maximumSlowModeDurationSeconds) << std::endl;
}

void Print(const ModerationActionInfo& info, std::string indent) {
  std::cout << indent << "ModerationActionInfo:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(info, moderatorId) << std::endl;
  std::cout << indent << PRINT_FIELD(info, moderatorName) << std::endl;
  std::cout << indent << PRINT_FIELD(info, targetId) << std::endl;
  std::cout << indent << PRINT_FIELD(info, targetName) << std::endl;
}

void Print(const ExtensionMessage& message, std::string indent) {
  std::cout << indent << "ExtensionMessage:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(message, messageId) << std::endl;
  std::cout << indent << PRINT_FIELD(message, chatColor) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(message, sentAt) << std::endl;

  // Badges
  if (message.badges.size() > 0) {
    std::cout << indent << "badges[" << message.badges.size() << "]: ";
    bool isFirst = true;
    for (const auto& badge : message.badges) {
      if (isFirst) {
        isFirst = false;
      } else {
        std::cout << ", ";
      }

      std::cout << badge.name << "/" << badge.version;
    }

    std::cout << std::endl;
  } else {
    std::cout << indent << "badges[0]" << std::endl;
  }

  // Tokens
  std::cout << indent << "message: ";
  for (const auto& token : message.tokens) {
    Print(token, indent);
  }

  std::cout << std::endl;
}

void Print(const MultiviewContentAttribute& attribute, std::string indent) {
  std::cout << indent << "MultiviewContentAttribute:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(attribute, attributeId) << std::endl;
  std::cout << indent << PRINT_FIELD(attribute, key) << std::endl;
  std::cout << indent << PRINT_FIELD(attribute, name) << std::endl;
  std::cout << indent << PRINT_FIELD(attribute, parentId) << std::endl;
  std::cout << indent << PRINT_FIELD(attribute, parentKey) << std::endl;
  std::cout << indent << PRINT_FIELD(attribute, value) << std::endl;
  std::cout << indent << PRINT_FIELD(attribute, valueShortName) << std::endl;
  std::cout << indent << PRINT_FIELD(attribute, imageUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(attribute, ownerChannelId) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(attribute, createdAt) << std::endl;
  std::cout << indent << PRINT_TIMESTAMP_FIELD(attribute, updatedAt) << std::endl;
}

void Print(const Chanlet& chanlet, std::string indent) {
  std::cout << indent << "Chanlet:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(chanlet, chanletId) << std::endl;

  std::cout << indent << "Content Attributes:" << std::endl;
  for (const auto& attribute : chanlet.attributes) {
    Print(attribute, indent + "  ");
  }
}

void Print(const UserList& userList, std::string indent) {
  std::cout << indent << "User List:" << std::endl;
  indent += "  ";

  std::cout << indent << "Total users: " << userList.totalUserCount << std::endl;

  std::cout << indent << "Moderators: ";
  for (const auto& user : userList.moderators) {
    std::cout << user << " ";
  }
  std::cout << std::endl;

  std::cout << indent << "Global Moderators: ";
  for (const auto& user : userList.globalModerators) {
    std::cout << user << " ";
  }
  std::cout << std::endl;

  std::cout << indent << "Staff: ";
  for (const auto& user : userList.staff) {
    std::cout << user << " ";
  }
  std::cout << std::endl;

  std::cout << indent << "Admins: ";
  for (const auto& user : userList.admins) {
    std::cout << user << " ";
  }
  std::cout << std::endl;

  std::cout << indent << "VIPs: ";
  for (const auto& user : userList.vips) {
    std::cout << user << " ";
  }
  std::cout << std::endl;

  std::cout << indent << "Viewers: ";
  for (const auto& user : userList.viewers) {
    std::cout << user << " ";
  }
  std::cout << std::endl;
}

std::string GetChatRaidMapKey(UserId userId, ChannelId channelId) {
  return std::to_string(userId) + ":" + std::to_string(channelId);
}
}  // namespace ttv

namespace {
const char* kTraceComponents[] = {
  "Chat",
  "ChatAPI",
  "ChatUserList",
  "ChatUserBlockList",
  "ChatThread",
  "ChatUserThreads",
  "ChatChangeUserBlockTask",
  "ChatChannelUsersTask",
  "ChatGenericApiTask",
  "ChatGetBlockListTask",
  "ChatGetThreadMessagesTask",
  "ChatGetUnreadMessageCountTask",
  "ChatGetUserTask",
  "ChatGetUserThreadsTask",
  "ChatPropertiesTask",
  "ChatUpdateUserThreadTask",
  "ChatTransport",
};

class ChatApiListener;
class ChatChannelListener;
class ChatChannelPropertyListener;
class ChatUserThreadsListener;
class ChatCommentListener;
class ChatRaidListener;
class ChatRoomListener;
class ChannelChatRoomManagerListener;
class ChatRoomNotificationsListener;
class FollowersListener;
class FollowingListener;
class BitsListener;
class SubscribersListener;
class SubscriptionsNotificationsListener;
class SquadNotificationsListener;
class MultiviewNotificationsListener;

std::shared_ptr<ChatAPI> gChatApi;
std::shared_ptr<ChatApiListener> gChatApiListener;
std::shared_ptr<ChatChannelListener> gChatChannelListener;
std::shared_ptr<ChatChannelListener> gChatWhisperListener;
std::shared_ptr<ChatUserThreadsListener> gChatUserThreadsListener;
std::shared_ptr<ChatCommentListener> gChatCommentListener;
std::shared_ptr<ChatRaidListener> gChatRaidListener;
std::shared_ptr<ChatRoomListener> gChatRoomListener;
std::shared_ptr<ChannelChatRoomManagerListener> gChannelChatRoomManagerListener;
std::shared_ptr<ChatRoomNotificationsListener> gChatRoomNotificationsListener;
std::map<std::string, std::shared_ptr<IChatCommentManager>> gChatCommentMap;
std::map<std::string, std::shared_ptr<IChatRaid>> gChatRaidMap;
std::string gLastRaidId;
std::map<UserId, std::shared_ptr<IBitsStatus>> gBitsStatusMap;
std::map<std::pair<UserId, ChannelId>, std::shared_ptr<IFollowersStatus>> gFollowersStatusMap;
std::map<UserId, std::shared_ptr<IFollowingStatus>> gFollowingStatusMap;
std::map<UserId, std::shared_ptr<ISubscribersStatus>> gSubscribersStatusMap;
std::map<std::pair<UserId, ChannelId>, std::shared_ptr<IChatChannelProperties>> gChatChannelPropertiesMap;
std::map<std::pair<UserId, ChannelId>, std::shared_ptr<IChatChannel>> gChatChannelMap;
std::map<std::pair<UserId, ChannelId>, std::shared_ptr<IChannelChatRoomManager>> gChannelChatRoomManagerMap;
std::map<std::pair<UserId, std::string>, std::shared_ptr<IChatRoom>> gChatRoomMap;
std::map<UserId, std::shared_ptr<IChatRoomNotifications>> gChatRoomNotificationsMap;
std::map<UserId, std::shared_ptr<ISubscriptionsNotifications>> gSubscriptionsNotificationsMap;
std::map<std::string, std::shared_ptr<ISquadNotifications>> gSquadNotificationsMap;
std::map<ChannelId, std::shared_ptr<IMultiviewNotifications>> gMultiviewNotificationsMap;

class ChatApiListener : public ttv::chat::IChatAPIListener {
 public:
  virtual void ModuleStateChanged(IModule* /*source*/, IModule::State state, TTV_ErrorCode ec) override {
    std::cout << "ModuleStateChanged: " << ::ToString(state) << " " << ErrorToString(ec) << std::endl;
  }

  virtual void ChatUserEmoticonSetsChanged(UserId userId, const std::vector<EmoticonSet>& sets) override {
    std::cout << "ChatUserEmoticonSetsChanged: " << userId << std::endl;
    Print(sets, "  ");
  }
};

class ChatChannelListener : public ttv::chat::IChatChannelListener {
 private:
  std::string GetChannelId(ChannelId channelId) const {
    return (channelId == 0) ? "<whisper>" : std::to_string(channelId);
  }

 public:
  virtual void ChatChannelStateChanged(
    UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec) override {
    std::cout << "ChatChannelStateChanged: " << userId << " " << GetChannelId(channelId) << " " << ToString(state)
              << " " << ErrorToString(ec) << std::endl;
  }

  virtual void ChatChannelInfoChanged(UserId userId, ChannelId channelId, const ChatChannelInfo& info) override {
    std::cout << "ChatChannelInfoChanged: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(info, "  ");
  }

  virtual void ChatChannelRestrictionsChanged(
    UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions) override {
    std::cout << "ChatChannelRestrictionsChanged: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(restrictions, "  ");
  }

  virtual void ChatChannelLocalUserChanged(UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) override {
    std::cout << "ChatChannelLocalUserChanged: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(userInfo, "  ");
  }

  virtual void ChatChannelMessagesReceived(
    UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) override {
    std::cout << "ChatChannelMessagesReceived: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(messageList, "  ");
  }

  virtual void ChatChannelSubscriptionNoticeReceived(
    UserId userId, ChannelId channelId, const SubscriptionNotice& notice) override {
    std::cout << "ChatChannelSubscriptionNoticeReceived: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(notice, "  ");
  }

  virtual void ChatChannelFirstTimeChatterNoticeReceived(
    UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) override {
    std::cout << "ChatChannelFirstTimeChatterNoticeReceived: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(notice, "  ");
  }

  virtual void ChatChannelRaidNoticeReceived(UserId userId, ChannelId channelId, const RaidNotice& notice) override {
    std::cout << "ChatChannelRaidNoticeReceived: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(notice, "  ");
  }

  virtual void ChatChannelUnraidNoticeReceived(
    UserId userId, ChannelId channelId, const UnraidNotice& notice) override {
    std::cout << "ChatChannelUnraidNoticeReceived: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(notice, "  ");
  }

  virtual void ChatChannelGenericNoticeReceived(
    UserId userId, ChannelId channelId, const GenericMessageNotice& notice) override {
    std::cout << "ChatChannelGenericNoticeReceived: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(notice, "  ");
  }

  virtual void ChatChannelMessagesCleared(UserId userId, ChannelId channelId) override {
    std::cout << "ChatChannelMessagesCleared: " << userId << " " << GetChannelId(channelId) << std::endl;
  }

  virtual void ChatChannelUserMessagesCleared(UserId userId, ChannelId channelId, UserId clearUserId) override {
    std::cout << "ChatChannelUserMessagesCleared: " << userId << " " << GetChannelId(channelId) << " " << clearUserId
              << std::endl;
  }

  virtual void ChatChannelMessageDeleted(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& senderLoginName, std::string&& deletedMessageContent) override {
    std::cout << "ChatChannelMessageDeleted: " << userId << " " << GetChannelId(channelId) << ", message " << messageId
              << " by " << senderLoginName << " content " << deletedMessageContent << std::endl;
  }

  virtual void ChatChannelModNoticeUserTimedOut(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) override {
    std::cout << "ChatChannelModNoticeUserTimedOut: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(modActionInfo, "  ");
    std::cout << "  Timeout duration seconds: " << timeoutDurationSeconds << std::endl;
    std::cout << "  Reason: " << reason << std::endl;
  }

  virtual void ChatChannelModNoticeUserBanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason) override {
    std::cout << "ChatChannelModNoticeUserBanned: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(modActionInfo, "  ");
    std::cout << "  Reason: " << reason << std::endl;
  }

  virtual void ChatChannelModNoticeUserUntimedOut(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) override {
    std::cout << "ChatChannelModNoticeUserUntimedOut: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(modActionInfo, "  ");
  }

  virtual void ChatChannelModNoticeUserUnbanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) override {
    std::cout << "ChatChannelModNoticeUserUnbanned: " << userId << " " << GetChannelId(channelId) << std::endl;
    Print(modActionInfo, "  ");
  }

  virtual void ChatChannelModNoticeMessageDeleted(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) override {
    std::cout << "ChatChannelModNoticeMessageDeleted: " << userId << " " << GetChannelId(channelId) << " " << messageId
              << " " << message << std::endl;
    Print(modActionInfo, "  ");
  }

  virtual void ChatChannelModNoticeClearChat(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeClearChat: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << std::endl;
  }

  virtual void ChatChannelModNoticeEmoteOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeEmoteOnly: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << std::endl;
  }

  virtual void ChatChannelModNoticeEmoteOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeEmoteOnlyOff: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << std::endl;
  }

  virtual void ChatChannelModNoticeFollowersOnly(UserId userId, ChannelId channelId, UserId modId,
    std::string&& modName, uint32_t minimumFollowingDurationMinutes) override {
    std::cout << "ChatChannelModNoticeFollowersOnly: " << userId << " " << GetChannelId(channelId) << " " << modId
              << " " << modName << " " << minimumFollowingDurationMinutes << std::endl;
  }

  virtual void ChatChannelModNoticeFollowersOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeFollowersOnlyOff: " << userId << " " << GetChannelId(channelId) << " " << modId
              << " " << modName << std::endl;
  }

  virtual void ChatChannelModNoticeR9K(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeR9K: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << std::endl;
  }

  virtual void ChatChannelModNoticeR9KOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeR9KOff: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << std::endl;
  }

  virtual void ChatChannelModNoticeSlow(UserId userId, ChannelId channelId, UserId modId, std::string&& modName,
    uint32_t slowModeDurationSeconds) override {
    std::cout << "ChatChannelModNoticeSlow: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << " " << slowModeDurationSeconds << std::endl;
  }

  virtual void ChatChannelModNoticeSlowOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeSlowOff: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << std::endl;
  }

  virtual void ChatChannelModNoticeSubsOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeSubsOnly: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << std::endl;
  }

  virtual void ChatChannelModNoticeSubsOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    std::cout << "ChatChannelModNoticeSubsOnlyOff: " << userId << " " << GetChannelId(channelId) << " " << modId << " "
              << modName << std::endl;
  }

  virtual void ChatChannelHostTargetChanged(
    UserId userId, ChannelId channelId, const std::string& targetChannel, uint32_t /*numViewers*/) override {
    std::cout << "ChatChannelHostTargetChanged: " << userId << " " << GetChannelId(channelId) << " " << targetChannel
              << std::endl;
  }

  virtual void ChatChannelNoticeReceived(UserId userId, ChannelId channelId, const std::string& id,
    const std::map<std::string, std::string>& params) override {
    std::cout << "ChatChannelNoticeReceived: " << userId << " " << GetChannelId(channelId) << " " << id << std::endl;
    for (auto kvp : params) {
      std::cout << "  " << kvp.first << " = " << kvp.second << std::endl;
    }
  }

  virtual void AutoModCaughtSentMessage(UserId userId, ChannelId channelId) override {
    std::cout << "AutoModCaughtSentMessage: " << userId << " " << GetChannelId(channelId) << std::endl;
  }

  virtual void AutoModDeniedSentMessage(UserId userId, ChannelId channelId) override {
    std::cout << "AutoModDeniedSentMessage: " << userId << " " << GetChannelId(channelId) << std::endl;
  }

  virtual void AutoModApprovedSentMessage(UserId userId, ChannelId channelId) override {
    std::cout << "AutoModApprovedSentMessage: " << userId << " " << GetChannelId(channelId) << std::endl;
  }

  virtual void AutoModCaughtMessageForMods(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) override {
    std::cout << "AutoModCaughtMessageForMods: " << userId << " " << GetChannelId(channelId) << std::endl;
    std::cout << "  Sender: " << senderName << " (" << senderId << ")" << std::endl;
    std::cout << "  Message: \"" << message << "\" (" << messageId << ")" << std::endl;
    std::cout << "  Reason: " << reason << std::endl;
  }

  virtual void AutoModMessageApprovedByMod(UserId userId, ChannelId channelId, std::string&& messageId,
    UserId moderatorId, std::string&& moderatorName) override {
    std::cout << "AutoModMessageApprovedByMod: " << userId << " " << GetChannelId(channelId) << std::endl;
    std::cout << "  Message id: " << messageId << std::endl;
    std::cout << "  Moderator that approved: " << moderatorName << " (" << moderatorId << ")" << std::endl;
  }

  virtual void AutoModMessageDeniedByMod(UserId userId, ChannelId channelId, std::string&& messageId,
    UserId moderatorId, std::string&& moderatorName) override {
    std::cout << "AutoModMessageDeniedByMod: " << userId << " " << GetChannelId(channelId) << std::endl;
    std::cout << "  Message id: " << messageId << std::endl;
    std::cout << "  Moderator that denied: " << moderatorName << " (" << moderatorId << ")" << std::endl;
  }

  virtual void AutoModDeniedSentCheer(UserId userId, ChannelId channelId) override {
    std::cout << "AutoModDeniedSentCheer: " << userId << " " << GetChannelId(channelId) << std::endl;
  }

  virtual void AutoModTimedOutSentCheer(UserId userId, ChannelId channelId) override {
    std::cout << "AutoModTimedOutSentCheer: " << userId << " " << GetChannelId(channelId) << std::endl;
  }

  virtual void AutoModCaughtCheerForMods(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) override {
    std::cout << "AutoModCaughtCheerForMods: " << userId << " " << GetChannelId(channelId) << std::endl;
    std::cout << "  Sender: " << senderName << " (" << senderId << ")" << std::endl;
    std::cout << "  Message: \"" << message << "\" (" << messageId << ")" << std::endl;
    std::cout << "  Reason: " << reason << std::endl;
  }
};

class ChatUserThreadsListener : public ttv::chat::IChatUserThreadsListener {
 public:
  virtual void ChatThreadRealtimeMessageReceived(
    UserId userId, const std::string& threadId, const WhisperMessage& message) override {
    std::cout << "ChatThreadRealtimeMessageReceived: " << userId << " " << threadId << " " << std::endl;
    Print(message, "  ");
  }
};

class ChatCommentListener : public ttv::chat::IChatCommentListener {
  virtual void ChatCommentManagerStateChanged(
    UserId userId, const std::string& vodId, IChatCommentManager::PlayingState state) override {
    std::cout << "ChatCommentManagerStateChanged: " << userId << " " << vodId << " " << ToString(state) << std::endl;
  }

  virtual void ChatCommentsReceived(
    UserId userId, const std::string& vodId, std::vector<ChatComment>&& messageList) override {
    std::cout << "ChatCommentsReceived: " << userId << " " << vodId << std::endl;
    Print(messageList, "  ");
  }

  virtual void ChatCommentsErrorReceived(const std::string& errorMessage, TTV_ErrorCode ec) override {
    std::cout << "ChatCommentsErrorReceived: " << errorMessage << " " << ec << std::endl;
  }
};

class ChatRaidListener : public ttv::chat::IChatRaidListener {
  virtual void RaidStarted(const RaidStatus& status) override {
    gLastRaidId = status.raidId;

    std::cout << "ChatRaidListener::RaidStarted: " << std::endl;
    Print(status, "  ");
  }

  virtual void RaidUpdated(const RaidStatus& status) override {
    std::cout << "ChatRaidListener::RaidUpdated: " << std::endl;
    Print(status, "  ");
  }

  virtual void RaidFired(const RaidStatus& status) override {
    std::cout << "ChatRaidListener::RaidCompleted: " << std::endl;
    Print(status, "  ");
  }

  virtual void RaidCancelled(const RaidStatus& status) override {
    std::cout << "ChatRaidListener::RaidCancelled: " << std::endl;
    Print(status, "  ");
  }
};

class ChatRoomListener : public ttv::chat::IChatRoomListener {
  virtual void MessageReceived(const std::string& roomId, ChatRoomMessage&& message) override {
    std::cout << "ChatRoomListener::MessageReceived: " << std::endl;
    Print(message, "  ");
  }

  virtual void MessageEdited(const std::string& roomId, ChatRoomMessage&& message) override {
    std::cout << "ChatRoomListener::MessageEdited: " << std::endl;
    Print(message, "  ");
  }

  virtual void MessageDeleted(const std::string& roomId, ChatRoomMessage&& message) override {
    std::cout << "ChatRoomListener::MessageDeleted: " << std::endl;
    Print(message, "  ");
  }

  virtual void RoomUpdated(ChatRoomInfo&& roomInfo) override {
    std::cout << "ChatRoomListener::RoomUpdated: " << std::endl;
    Print(roomInfo, "  ");
  }
};

class ChannelChatRoomManagerListener : public ttv::chat::IChannelChatRoomManagerListener {
  virtual void PurgeMessages(UserId userId, ChannelId channelId, Timestamp purgeAfter) override {
    std::cout << "ChannelChatRoomManagerListener::PurgeMessages: " << std::endl;
    std::cout << "User id: " << userId << std::endl;
    std::cout << "Channel id: " << channelId << std::endl;
    std::cout << "Purge after: " << purgeAfter << std::endl;
  }

  virtual void RoomCreated(ChannelId ownerId, ChatRoomInfo&& roomInfo) override {
    std::cout << "ChannelChatRoomManagerListener::RoomCreated: " << std::endl;
    std::cout << "Owner id: " << ownerId << std::endl;
    Print(roomInfo, "  ");
  }

  virtual void RoomDeleted(ChannelId ownerId, ChatRoomInfo&& roomInfo) override {
    std::cout << "ChannelChatRoomManagerListener::RoomDeleted: " << std::endl;
    std::cout << "Owner id: " << ownerId << std::endl;
    Print(roomInfo, "  ");
  }
};

class ChatRoomNotificationsListener : public ttv::chat::IChatRoomNotificationsListener {
  virtual void UserTimedOut(UserId userId, ChannelId ownerId, Timestamp expiresAt) override {
    std::cout << "ChatRoomNotificationsListener::UserTimedOut: " << std::endl;
    std::cout << "User " << userId << " timed out from " << ownerId << std::endl;
  }

  virtual void UserBanned(UserId userId, ChannelId ownerId) override {
    std::cout << "ChatRoomNotificationsListener::UserBanned: " << std::endl;
    std::cout << "User " << userId << " banned from " << ownerId << std::endl;
  }

  virtual void UserUnbanned(UserId userId, ChannelId ownerId) override {
    std::cout << "ChatRoomNotificationsListener::UserUnbanned: " << std::endl;
    std::cout << "User " << userId << " unbanned from " << ownerId << std::endl;
  }

  virtual void RoomViewUpdated(
    UserId userId, ChannelId ownerId, const std::string& roomId, ChatRoomView&& roomViewInfo) override {
    std::cout << "ChatRoomNotificationsListener::RoomViewUpdated:" << std::endl;
    std::cout << "User id: " << userId << std::endl;
    std::cout << "Owner channel id: " << ownerId << std::endl;
    std::cout << "Room id: " << roomId << std::endl;
    Print(roomViewInfo, "  ");
  }

  virtual void RoomMentionReceived(UserId userId, RoomMentionInfo&& mentionInfo) override {
    std::cout << "ChatRoomNotificationsListener::RoomMention:" << std::endl;
    std::cout << "User id: " << userId << std::endl;
    Print(mentionInfo, "  ");
  }
};

class BitsListener : public ttv::chat::IBitsListener {
 public:
  BitsListener(UserId userId) : mUserId(userId) {}

  virtual void UserReceivedBits(const BitsReceivedEvent& bitsReceivedEvent) override {
    std::cout << "BitsListener:UserReceivedBits (User: " << mUserId << "):" << std::endl;
    Print(bitsReceivedEvent, "  ");
  }

  virtual void UserSentBits(const BitsSentEvent& bitsSentEvent) override {
    std::cout << "BitsListener:UserSentBits (User: " << mUserId << "):" << std::endl;
    Print(bitsSentEvent, "  ");
  }

  virtual void UserGainedBits(const uint32_t bitsBalance) override {
    std::cout << "BitsListener:UserGainedBits (User: " << mUserId << "):" << std::endl;
    std::cout << "Bits balance: " << bitsBalance << std::endl;
  }

 private:
  const UserId mUserId;
};

class FollowersListener : public ttv::chat::IFollowersListener {
 public:
  FollowersListener(UserId userId, ChannelId channelId) : mUserId(userId), mChannelId(channelId) {}

  virtual void NewFollowerAdded(const FollowerAddedEvent& followerAddedEvent) override {
    std::cout << "FollowersListener:NewFollowerAdded (User: " << mUserId << " / Channel: " << mChannelId
              << "):" << std::endl;
    Print(followerAddedEvent, "  ");
  }

 private:
  const UserId mUserId;
  const ChannelId mChannelId;
};

class FollowingListener : public ttv::chat::IFollowingListener {
 public:
  FollowingListener() {}

  virtual void FollowedChannel(UserId userId, ChannelId channelId) override {
    std::cout << "FollowingListener:FollowedChannel (User: " << userId << " / Channel: " << channelId
              << "):" << std::endl;
  }

  virtual void UnfollowedChannel(UserId userId, ChannelId channelId) override {
    std::cout << "FollowingListener:UnfollowedChannel (User: " << userId << " / Channel: " << channelId
              << "):" << std::endl;
  }
};

class SubscribersListener : public ttv::chat::ISubscribersListener {
 public:
  SubscribersListener(UserId userId) : mUserId(userId) {}

  virtual void NewSubscriberAdded(const SubscriberAddedEvent& subscriberAddedEvent) override {
    std::cout << "SubscribersListener:NewSubscriberAdded (User: " << mUserId << "):" << std::endl;
    Print(subscriberAddedEvent, "  ");
  }

 private:
  const UserId mUserId;
};

class SubscriptionsNotificationsListener : public ttv::chat::ISubscriptionsNotificationsListener {
 public:
  virtual void SubscribedToChannel(UserId userId, ChannelId channelId) {
    std::cout << "SubscriptionsNotificationsListener:SubscribedtoChannel (User: " << userId
              << ", Channel: " << channelId << ")" << std::endl;
  }
};

class SquadNotificationsListener : public ttv::chat::ISquadNotificationsListener {
 public:
  virtual void SquadUpdated(SquadInfo&& squad) {
    std::cout << "SquadNotificationsListener:SquadUpdated: " << std::endl;
    Print(std::move(squad), "  ");
  }

  virtual void SquadEnded() { std::cout << "SquadNotificationsListener:SquadEnded" << std::endl; }
};

class MultiviewNotificationsListener : public ttv::chat::IMultiviewNotificationsListener {
 public:
  virtual void ChanletUpdated(UserId userId, ChannelId channelId, Chanlet&& chanlet) {
    std::cout << "MultiviewNotificationsListener:ChanletUpdated: " << std::endl;
    Print(std::move(chanlet), "  ");
  }
};

class ChatChannelPropertyListener : public ttv::chat::IChatChannelPropertyListener {
 public:
  ChatChannelPropertyListener(UserId userId, ChannelId channelId) : mUserId(userId), mChannelId(channelId) {}

  virtual void RitualsEnabled(bool ritualsEnabled) override {
    std::cout << "ChatChannelPropertyListener:RitualsEnabled (User: " << mUserId << " / Channel: " << mChannelId
              << "): "
              << "ritualsEnabled: " << ritualsEnabled << std::endl;
  }

  virtual void OutgoingHostChanged(ChannelId channelId, ChannelId previousTarget, ChannelId currentTarget,
    std::string&& currentTargetName, uint32_t numViewers) override {
    std::cout << "ChatChannelPropertyListener::OutgoingHostChanged: (User: " << mUserId << " / Channel: " << mChannelId
              << "):" << std::endl;
    std::cout << "  Previous target: " << previousTarget << std::endl;
    std::cout << "  Current target: " << currentTarget << ", " << currentTargetName << std::endl;
    std::cout << "  Viewer Count: " << numViewers << std::endl;
  }

  virtual void IncomingHostStarted(
    ChannelId channelId, ChannelId hostChannelId, std::string&& hostChannelName, uint32_t numViewers) override {
    std::cout << "ChatChannelPropertyListener::HostingChannelChangedTarget: (User: " << mUserId
              << " / Channel: " << mChannelId << "):" << std::endl;
    std::cout << "  Host channel: " << hostChannelId << ", " << hostChannelName << std::endl;
    std::cout << "  Viewer Count: " << numViewers << std::endl;
  }

  virtual void IncomingHostEnded(ChannelId channelId, ChannelId hostChannelId, std::string&& hostChannelName) override {
    std::cout << "ChatChannelPropertyListener::HostingChannelChangedTarget: (User: " << mUserId
              << " / Channel: " << mChannelId << "):" << std::endl;
    std::cout << "  Host channel: " << hostChannelId << ", " << hostChannelName << std::endl;
  }

  virtual void ExtensionMessageReceived(ExtensionMessage&& message) override {
    std::cout << "ChatChannelPropertyListener::ExtensionMessageReceived: (User: " << mUserId
              << " / Channel: " << mChannelId << "):" << std::endl;
    Print(std::move(message), "  ");
  }

  virtual void ChatChannelRestrictionsReceived(ChatChannelRestrictions&& chatChannelRestrictions) override {
    std::cout << "ChatChannelPropertyListener::ChatChannelRestrictionsReceived: (User: " << mUserId
              << " / Channel: " << mChannelId << "):" << std::endl;
    Print(std::move(chatChannelRestrictions), "  ");
  }

 private:
  const UserId mUserId;
  const ChannelId mChannelId;
};

std::shared_ptr<CommandCategory> RegisterChatCommands(std::shared_ptr<Context> /*context*/) {
  std::shared_ptr<CommandCategory> category =
    std::make_shared<CommandCategory>("Chat", "The ttv::chat::ChatAPI command interface");

  category->AddCommand("ChatInit")
    .AddFunction()
    .Description("Calls ChatAPI::Initialize")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      // TODO: Maybe make these properties to be settable
      gChatApi->SetTokenizationOptions(TokenizationOptions::All());
      gChatApi->SetEnabledFeatures(FeatureFlags::All());
      gChatApi->SetListener(gChatApiListener);

      TTV_ErrorCode ec = gChatApi->Initialize([](TTV_ErrorCode ec) {
        std::cout << "ChatInit completed";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatGetState")
    .AddFunction()
    .Description("Calls ChatAPI::GetState")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      IModule::State state = gChatApi->GetState();
      std::cout << "  " << ::ToString(state) << std::endl;
      return true;
    });

  category->AddCommand("ChatShutdown")
    .AddFunction()
    .Description("Calls ChatAPI::Shutdown")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gChatApi->Shutdown([](TTV_ErrorCode ec) {
        std::cout << "ChatShutdown completed";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchUserEmoticonSets")
    .AddFunction()
    .Description("Calls ChatAPI::FetchUserEmoticonSets")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("forceRefetch", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::vector<EmoticonSet> sets;

      TTV_ErrorCode ec = gChatApi->FetchUserEmoticonSets(
        params[0], params[1], [](TTV_ErrorCode ec, const std::vector<EmoticonSet>& sets) {
          std::cout << "FetchUserEmoticonSetIds completed";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            Print(sets, "  ");
          }
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("FetchGlobalBadges")
    .AddFunction()
    .Description("Calls ChatAPI::FetchGlobalBadges.")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gChatApi->FetchGlobalBadges([](TTV_ErrorCode ec, BadgeSet&& badgeSet) {
        std::cout << "FetchGlobalBadges completed";
        ReportCommandResult(ec);
        Print(std::move(badgeSet), "  ");
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("FetchChannelBadges")
    .AddFunction()
    .Description("Calls ChatAPI::FetchChannelBadges for the given channel.")
    .AddParam("channel", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->FetchChannelBadges(params[0], [](TTV_ErrorCode ec, BadgeSet&& badgeSets) {
        std::cout << "FetchChannelBadges completed";
        ReportCommandResult(ec);
        Print(std::move(badgeSets), "  ");
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatConnect")
    .AddAlias("connect")
    .AddFunction()
    .Description("Calls ChatAPI::Connect with the given userId")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->Connect(params[0], params[1], gChatChannelListener);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatDisconnect")
    .AddAlias("disconnect")
    .AddFunction()
    .Description("Calls ChatAPI::Disconnect")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->Disconnect(params[0], params[1]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CreateChatChannel")
    .AddFunction()
    .Description("Calls ChatAPI::CreateChatChannel with the given userId/channeId")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChatChannelMap.find(pair);
      if (iter != gChatChannelMap.end()) {
        std::cout << "Already created ChatChannel for UserId/ChannelId pair." << std::endl;
        return true;
      }

      std::shared_ptr<IChatChannelListener> listener = std::make_shared<ChatChannelListener>();
      std::shared_ptr<IChatChannel> chatChannel;

      TTV_ErrorCode ec = gChatApi->CreateChatChannel(pair.first, pair.second, listener, chatChannel);
      if (TTV_SUCCEEDED(ec)) {
        ec = chatChannel->Connect();
        if (TTV_SUCCEEDED(ec)) {
          std::cout << "Successfully connected to ChatChannel: (" << pair.first << "/" << pair.second << ")"
                    << std::endl;
          gChatChannelMap.insert({pair, chatChannel});
        }
      }

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeChatChannel")
    .AddFunction()
    .Description("Disposes a previously created IChatChannel instance with the given userId/channelId")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      std::shared_ptr<IChatChannel> chatChannel;
      {
        auto iter = gChatChannelMap.find(pair);
        if (iter == gChatChannelMap.end()) {
          std::cout << "DisposeChatChannel called on a IChatChannel with UserId/ChannelId pair that doesn't exist."
                    << std::endl;
          return true;
        }
        chatChannel = iter->second;
        gChatChannelMap.erase(iter);
      }
      chatChannel->Disconnect();
      chatChannel.reset();

      std::cout << "Successfully disconnected from ChatChannel: (" << pair.first << "/" << pair.second << ")"
                << std::endl;
      return true;
    });

  category->AddCommand("ChatConnectAnonymous")
    .AddAlias("connectanonymous")
    .AddAlias("connectanon")
    .AddFunction()
    .Description("Calls ChatAPI::Connect anonymously")
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->Connect(0, params[0], gChatChannelListener);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatSendMessageToChannel")
    .AddAlias("send")
    .AddFunction()
    .Description("Calls ChatAPI::SendChatMessage")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("message", ParamType::RemainderAsString)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->SendChatMessage(params[0], params[1], params[2]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatBlockUser")
    .AddAlias("block")
    .AddFunction()
    .Description("Calls ChatAPI::BlockUser")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("otherUserId", ParamType::UInt32)
    .AddParam("reason", ParamType::String)
    .AddParam("whisper", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->BlockUser(params[0], params[1], params[2], params[3], [](TTV_ErrorCode ec) {
        std::cout << "BlockUser completed";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatUnblockUser")
    .AddAlias("unblock")
    .AddFunction()
    .Description("Calls ChatAPI::UnblockUser")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("otherUserId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->UnblockUser(params[0], params[1], [](TTV_ErrorCode ec) {
        std::cout << "UnblockUser completed";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchBlockedUsers")
    .AddAlias("fetchblockedusers")
    .AddFunction()
    .Description("Calls ChatAPI::FetchBlockedUsers")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gChatApi->FetchBlockedUsers(params[0], [](TTV_ErrorCode ec, const std::vector<UserInfo>& blockedUsers) {
          std::cout << "Blocked users: " << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            for (const UserInfo& user : blockedUsers) {
              std::cout << "  " << user.userId << " / " << user.userName << std::endl;
            }
          }
          ReportCommandResult(ec);
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatGetEmoticonUrl")
    .AddFunction()
    .Description("Calls ChatAPI::GetEmoticonUrl")
    .AddParam("emoticonId", ParamType::String)
    .AddParam("scale", ParamType::Float)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string url;
      TTV_ErrorCode ec = gChatApi->GetEmoticonUrl(params[0], params[1], url);

      if (TTV_SUCCEEDED(ec)) {
        std::cout << "  Emoticon URL: " << url << std::endl;
      }

      return true;
    });

  category->AddCommand("ChatGetBitsImageUrl")
    .AddFunction()
    .Description("Calls ChatAPI::GetBitsImageUrl")
    .AddParam("prefix", ParamType::String)
    .AddParam("numBits", ParamType::UInt32)
    .AddParam("theme", ParamType::String)
    .AddParam("dpiScale", ParamType::Float)
    .AddParam("isAnimated", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::shared_ptr<BitsConfiguration> config = std::make_shared<BitsConfiguration>();
      Print(config, "    ");

      BitsConfiguration::CheermoteImage::Theme theme;
      ttv::json::ToObject(ttv::json::Value(params[2].GetString()), theme);

      if (theme == BitsConfiguration::CheermoteImage::Theme::Unknown) {
        std::cout << "Theme must be one of [DARK, LIGHT]" << std::endl;
        return true;
      }

      TTV_ErrorCode ec = gChatApi->FetchGlobalBitsConfiguration(
        [params, theme](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
          if (TTV_FAILED(ec)) {
            std::cout << "FetchBitsConfiguration called back";
          }
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            Print(config, "    ");

            std::string url;
            Color color = 0xFF000000;

            ec = config->GetBitsImageUrl(params[0].GetString(), params[1].GetUInt32(), theme, params[3].GetFloat(),
              params[4].GetBool(), url, color);

            if (TTV_SUCCEEDED(ec)) {
              std::cout << "  Bits color: " << color << ", URL: " << url << std::endl;
            }
          }
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatGetHighestDpiBitsImageUrl")
    .AddFunction()
    .Description("Calls ChatAPI::GetBitsImageUrl")
    .AddParam("prefix", ParamType::String)
    .AddParam("numBits", ParamType::UInt32)
    .AddParam("theme", ParamType::String)
    .AddParam("dpiScaleLimit", ParamType::Float)
    .AddParam("isAnimated", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::shared_ptr<BitsConfiguration> config = std::make_shared<BitsConfiguration>();
      Print(config, "    ");

      BitsConfiguration::CheermoteImage::Theme theme;
      ttv::json::ToObject(ttv::json::Value(params[2].GetString()), theme);

      if (theme == BitsConfiguration::CheermoteImage::Theme::Unknown) {
        std::cout << "Theme must be one of [DARK, LIGHT]" << std::endl;
        return true;
      }

      TTV_ErrorCode ec = gChatApi->FetchGlobalBitsConfiguration(
        [params, theme](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
          if (TTV_FAILED(ec)) {
            std::cout << "FetchBitsConfiguration called back";
          }
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            Print(config, "    ");

            std::string url;
            Color color = 0xFF000000;

            ec = config->GetHighestDpiBitsImageUrl(params[0].GetString(), params[1].GetUInt32(), theme,
              params[3].GetFloat(), params[4].GetBool(), url, color);

            if (TTV_SUCCEEDED(ec)) {
              std::cout << "  Bits color: " << color << ", URL: " << url << std::endl;
            }
          }
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatGlobalFetchBitsConfiguration")
    .AddFunction()
    .Description("Calls ChatAPI::FetchBitsConfiguration")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec =
        gChatApi->FetchGlobalBitsConfiguration([](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
          if (TTV_FAILED(ec)) {
            std::cout << "FetchBitsConfiguration called back";
          }
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Config:" << std::endl;
            Print(config, "    ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatChannelFetchBitsConfiguration")
    .AddFunction()
    .Description("Calls ChatAPI::FetchChannelBitsConfiguration")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->FetchChannelBitsConfiguration(params[0].GetUInt32(), params[1].GetUInt32(),
        [](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
          if (TTV_FAILED(ec)) {
            std::cout << "FetchBitsConfiguration called back";
          }
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Config:" << std::endl;
            Print(config, "    ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatCreateChatCommentManager")
    .AddFunction()
    .Description("Calls ChatAPI::CreateChatCommentManager")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::shared_ptr<IChatCommentManager> chatCommentManager;

      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto& chatCommentEntry = gChatCommentMap[key];
      if (chatCommentEntry != nullptr) {
        ReportCommandResult(TTV_EC_ALREADY_INITIALIZED);
        return true;
      }
      auto result = gChatApi->CreateChatCommentManager(params[0], params[1], gChatCommentListener);
      if (!result.IsError()) {
        chatCommentEntry = result.GetResult();
      }

      ReportCommandResult(result.GetErrorCode());

      return true;
    });

  category->AddCommand("ChatCommentManagerDispose")
    .AddFunction()
    .Description("Calls ChatCommentManager::Dispose")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }

      auto chatCommentManager = chatCommentIterator->second;
      TTV_ErrorCode ec = chatCommentManager->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gChatCommentMap.erase(chatCommentIterator);
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatCommentPlay")
    .AddAlias("CCPlay")
    .AddFunction()
    .Description("Calls ChatCommentManager::Play")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = chatCommentIterator->second->Play();
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatCommentPause")
    .AddAlias("CCPause")
    .AddFunction()
    .Description("Calls ChatCommentManager::Pause")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = chatCommentIterator->second->Pause();
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatCommentUpdatePlayhead")
    .AddAlias("CCUpdatePlayhead")
    .AddFunction()
    .Description("Calls ChatCommentManager::UpdatePlayhead")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("timestampMilliseconds", ParamType::UInt64)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = chatCommentIterator->second->UpdatePlayhead(params[2].GetUInt64());
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatCommentGetPlayheadTime")
    .AddAlias("CCCurrentTime")
    .AddFunction()
    .Description("Calls ChatCommentManager::GetPlayheadTime")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      auto result = chatCommentIterator->second->GetPlayheadTime();
      if (!result.IsError()) {
        std::cout << "Current VOD time in milliseconds: " << result.GetResult() << std::endl;
      }

      ReportCommandResult(result.GetErrorCode());

      return true;
    });

  category->AddCommand("ChatCommentGetChannel")
    .AddFunction()
    .Description("Calls ChatCommentManager::GetChannel")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }

      auto result = chatCommentIterator->second->GetChannelId();
      if (!result.IsError()) {
        std::cout << "ChannelId for vodId: VOD " << params[1].GetString() << " is channel " << result.GetResult()
                  << std::endl;
      }

      ReportCommandResult(result.GetErrorCode());

      return true;
    });

  category->AddCommand("ChatCommentGetPlayingState")
    .AddFunction()
    .Description("Calls ChatCommentManager::GetPlayingState")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      auto playingStateResult = chatCommentIterator->second->GetPlayingState();
      auto playheadTimeResult = chatCommentIterator->second->GetPlayheadTime();

      if (!playingStateResult.IsError() && !playheadTimeResult.IsError()) {
        std::cout << "State for vodId: " << params[1].GetString() << " is " << ToString(playingStateResult.GetResult())
                  << " at time (ms): " << playheadTimeResult.GetResult() << std::endl;
      }

      ReportCommandResult(playingStateResult.GetErrorCode());
      return true;
    });

  category->AddCommand("ChatFetchVodCommentsByTimestamp")
    .AddAlias("CVodCommentsSeconds")
    .AddFunction()
    .Description("Calls ChatCommentManager::FetchComments")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("timestampMilliseconds", ParamType::UInt64)
    .AddParam("limit", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = chatCommentIterator->second->FetchComments(params[2].GetUInt64(), params[3],
        [](TTV_ErrorCode ec, std::vector<ChatComment>&& result, std::string&& nextCursor) {
          std::cout << "Got ChatCommentManager::FetchComments reponse: " << ErrorToString(ec) << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(result, "  ");

            std::cout << "Next cursor url: " << nextCursor << std::endl;
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchVodCommentsByCursor")
    .AddAlias("CVodCommentsCursor")
    .AddFunction()
    .Description("Calls ChatCommentManager::FetchComments")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("cursor", ParamType::String)
    .AddParam("limit", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = chatCommentIterator->second->FetchComments(params[2].GetString(), params[3],
        [](TTV_ErrorCode ec, std::vector<ChatComment>&& result, std::string&& nextCursor) {
          std::cout << "Got ChatCommentManager::FetchComments reponse: " << ErrorToString(ec) << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(result, "  ");

            std::cout << "Next cursor url: " << nextCursor << std::endl;
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchComment")
    .AddAlias("CFetchComment")
    .AddFunction()
    .Description("Calls ChatCommentManager::FetchComment")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("commentId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec =
        chatCommentIterator->second->FetchComment(params[2], [](TTV_ErrorCode ec, ChatComment&& result) {
          std::cout << "Got ChatCommentManager::FetchComment reponse: " << ErrorToString(ec) << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(result, "  ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatDeleteComment")
    .AddAlias("CDeleteComment")
    .AddFunction()
    .Description("Calls ChatCommentManager::DeleteComment")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("commentId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = chatCommentIterator->second->DeleteComment(params[2], [](TTV_ErrorCode ec) {
        std::cout << "Got ChatCommentManager::DeleteComment reponse: " << ErrorToString(ec) << std::endl;
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatPostComment")
    .AddAlias("CPostComment")
    .AddFunction()
    .Description("Calls ChatCommentManager::PostComment")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("message", ParamType::String)
    .AddParam("timestampMilliseconds", ParamType::UInt64)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = chatCommentIterator->second->PostComment(
        params[2], params[3], [](TTV_ErrorCode ec, ChatComment&& result, std::string&& errorMessage) {
          std::cout << "Got ChatCommentManager::PostComment reponse: " << ErrorToString(ec) << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(result, "  ");
          } else {
            std::cout << errorMessage << std::endl;
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatReportComment")
    .AddAlias("CReportComment")
    .AddFunction()
    .Description("Calls ChatCommentManager::ReportComment")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("commentId", ParamType::String)
    .AddParam("reason", ParamType::String)
    .AddParam("description", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec =
        chatCommentIterator->second->ReportComment(params[2], params[3], params[4], [](TTV_ErrorCode ec) {
          std::cout << "Got ChatCommentManager::ReportComment reponse: " << ErrorToString(ec) << std::endl;
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchCommentReplies")
    .AddAlias("CFetchCommentReplies")
    .AddFunction()
    .Description("Calls ChatCommentManager::FetchCommentReplies")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("commentId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = chatCommentIterator->second->FetchCommentReplies(
        params[2], [](TTV_ErrorCode ec, std::vector<ChatComment>&& result) {
          std::cout << "Got ChatCommentManager::FetchCommentReplies reponse: " << ErrorToString(ec) << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(result, "  ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatPostCommentReply")
    .AddAlias("CPostCommentReply")
    .AddFunction()
    .Description("Calls ChatCommentManager::PostCommentReply")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("vodId", ParamType::String)
    .AddParam("parentCommentId", ParamType::String)
    .AddParam("message", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = std::to_string(params[0].GetUInt32()) + ":" + params[1].GetString();
      auto chatCommentIterator = gChatCommentMap.find(key);
      if (chatCommentIterator == gChatCommentMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec =
        chatCommentIterator->second->PostCommentReply(params[2], params[3], [](TTV_ErrorCode ec, ChatComment&& result) {
          std::cout << "Got ChatCommentManager::PostCommentReply reponse: " << ErrorToString(ec) << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(result, "  ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchChannelVodCommentSettings")
    .AddAlias("CFetchCommentSettings")
    .AddFunction()
    .Description("Calls ChatAPI::FetchChannelVodCommentSettings")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->FetchChannelVodCommentSettings(
        params[0], params[1], [](TTV_ErrorCode ec, ChannelVodCommentSettings&& settings) {
          std::cout << "Got ChatAPI::FetchChannelVodCommentSettings reponse: " << ErrorToString(ec) << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(settings, "  ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatSetChannelVodFollowersOnlyDuration")
    .AddAlias("CSetFollowersOnlyDuration")
    .AddFunction()
    .Description("Calls ChatAPI::SetChannelVodFollowersOnlyDuration")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("followersOnlyDuration", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gChatApi->SetChannelVodFollowersOnlyDuration(params[0], params[1], params[2], [](TTV_ErrorCode ec) {
          std::cout << "Got ChatAPI::SetChannelVodFollowersOnlyDuration reponse: " << ErrorToString(ec) << std::endl;
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatSetChannelVodPublishingMode")
    .AddAlias("CSetChannelVodPublishingMode")
    .AddFunction()
    .Description("Calls ChatAPI::SetChannelVodPublishingMode")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("publishingMode", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      CommentPublishingMode mode;
      std::string modeStr(params[2]);

      if (modeStr == "open") {
        mode = CommentPublishingMode::Open;
      } else if (modeStr == "review") {
        mode = CommentPublishingMode::Review;
      } else if (modeStr == "disabled") {
        mode = CommentPublishingMode::Disabled;
      } else {
        std::cout << "Publishing must be one of [open, review, disabled]." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = gChatApi->SetChannelVodPublishingMode(params[0], params[1], mode, [](TTV_ErrorCode ec) {
        std::cout << "Got ChatAPI::SetChannelVodPublishingMode reponse: " << ErrorToString(ec) << std::endl;
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatRaidCreate")
    .AddFunction()
    .Description("Calls ChatAPI::CreateChatRaid")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::shared_ptr<IChatRaid> chatRaid;

      std::string key = GetChatRaidMapKey(params[0].GetUInt32(), params[1].GetUInt32());
      auto& entry = gChatRaidMap[key];
      if (entry != nullptr) {
        ReportCommandResult(TTV_EC_ALREADY_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = gChatApi->CreateChatRaid(params[0], params[1], gChatRaidListener, chatRaid);
      entry = chatRaid;
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatRaidDispose")
    .AddFunction()
    .Description("Calls ChatRaid::Dispose")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = GetChatRaidMapKey(params[0].GetUInt32(), params[1].GetUInt32());
      auto iter = gChatRaidMap.find(key);
      if (iter == gChatRaidMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }

      auto raid = iter->second;
      TTV_ErrorCode ec = raid->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gChatRaidMap.erase(iter);
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRaidJoin")
    .AddFunction()
    .Description("Calls ChatRaid::Join")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("raidId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = GetChatRaidMapKey(params[0].GetUInt32(), params[1].GetUInt32());
      auto iter = gChatRaidMap.find(key);
      if (iter == gChatRaidMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }

      std::string raidId = params[2].GetString();
      if (raidId == "_") {
        raidId = gLastRaidId;
      }

      TTV_ErrorCode ec = iter->second->Join(raidId,
        [](TTV_ErrorCode ec) { std::cout << "Got ChatRaid::Join reponse: " << ErrorToString(ec) << std::endl; });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatRaidLeave")
    .AddFunction()
    .Description("Calls ChatRaid::Leave")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("raidId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = GetChatRaidMapKey(params[0].GetUInt32(), params[1].GetUInt32());
      auto iter = gChatRaidMap.find(key);
      if (iter == gChatRaidMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }

      std::string raidId = params[2].GetString();
      if (raidId == "_") {
        raidId = gLastRaidId;
      }

      TTV_ErrorCode ec = iter->second->Leave(raidId,
        [](TTV_ErrorCode ec) { std::cout << "Got ChatRaid::Leave reponse: " << ErrorToString(ec) << std::endl; });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatRaidStart")
    .AddFunction()
    .Description("Calls ChatRaid::Start")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("targetChannelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = GetChatRaidMapKey(params[0].GetUInt32(), params[1].GetUInt32());
      auto iter = gChatRaidMap.find(key);
      if (iter == gChatRaidMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = iter->second->Start(params[2].GetUInt32(),
        [](TTV_ErrorCode ec) { std::cout << "Got ChatRaid::Start reponse: " << ErrorToString(ec) << std::endl; });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatRaidRaidNow")
    .AddFunction()
    .Description("Calls ChatRaid::RaidNow")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = GetChatRaidMapKey(params[0].GetUInt32(), params[1].GetUInt32());
      auto iter = gChatRaidMap.find(key);
      if (iter == gChatRaidMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = iter->second->RaidNow(
        [](TTV_ErrorCode ec) { std::cout << "Got ChatRaid::Start reponse: " << ErrorToString(ec) << std::endl; });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatRaidCancel")
    .AddFunction()
    .Description("Calls ChatRaid::Cancel")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string key = GetChatRaidMapKey(params[0].GetUInt32(), params[1].GetUInt32());
      auto iter = gChatRaidMap.find(key);
      if (iter == gChatRaidMap.end()) {
        ReportCommandResult(TTV_EC_NOT_INITIALIZED);
        return true;
      }
      TTV_ErrorCode ec = iter->second->Cancel(
        [](TTV_ErrorCode ec) { std::cout << "Got ChatRaid::Cancel reponse: " << ErrorToString(ec) << std::endl; });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CreateBitsStatus")
    .AddFunction()
    .Description("Calls ChatAPI::CreateBitsStatus")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      auto iter = gBitsStatusMap.find(userId);
      if (iter != gBitsStatusMap.end()) {
        std::cout << "Already created BitsStatus for UserId." << std::endl;
        return true;
      }

      std::shared_ptr<IBitsListener> listener = std::make_shared<BitsListener>(userId);
      std::shared_ptr<IBitsStatus> bitsStatus;

      TTV_ErrorCode ec = gChatApi->CreateBitsStatus(userId, listener, bitsStatus);
      gBitsStatusMap.insert({userId, bitsStatus});
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeBitsStatus")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeBitsStatus")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      auto iter = gBitsStatusMap.find(userId);
      if (iter == gBitsStatusMap.end()) {
        std::cout << "DisposeBitsStatus called on a BitsStatus with UserId that doesn't exist." << std::endl;
        return true;
      }

      std::shared_ptr<IBitsStatus> bitsStatus = iter->second;
      TTV_ErrorCode ec = bitsStatus->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gBitsStatusMap.erase(iter);
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("CreateFollowersStatus")
    .AddFunction()
    .Description("Calls ChatAPI::CreateFollowersStatus")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gFollowersStatusMap.find(pair);
      if (iter != gFollowersStatusMap.end()) {
        std::cout << "Already created FollowersStatus for UserId/ChannelId pair." << std::endl;
        return true;
      }

      std::shared_ptr<IFollowersListener> listener = std::make_shared<FollowersListener>(pair.first, pair.second);
      std::shared_ptr<IFollowersStatus> followersStatus;

      TTV_ErrorCode ec = gChatApi->CreateFollowersStatus(pair.first, pair.second, listener, followersStatus);
      gFollowersStatusMap.insert({pair, followersStatus});
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeFollowersStatus")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeFollowersStatus")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gFollowersStatusMap.find(pair);
      if (iter == gFollowersStatusMap.end()) {
        std::cout << "DisposeFollowersStatus called on a FollowersStatus with UserId/ChannelId pair that doesn't exist."
                  << std::endl;
        return true;
      }

      std::shared_ptr<IFollowersStatus> followersStatus = iter->second;
      TTV_ErrorCode ec = followersStatus->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gFollowersStatusMap.erase(iter);
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("CreateFollowingStatus")
    .AddFunction()
    .Description("Calls ChatAPI::CreateFollowingStatus")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      auto iter = gFollowingStatusMap.find(userId);
      if (iter != gFollowingStatusMap.end()) {
        std::cout << "Already created FollowingStatus for UserId." << std::endl;
        return true;
      }

      std::shared_ptr<IFollowingListener> listener = std::make_shared<FollowingListener>();
      std::shared_ptr<IFollowingStatus> followingStatus;

      TTV_ErrorCode ec = gChatApi->CreateFollowingStatus(userId, listener, followingStatus);
      gFollowingStatusMap.insert({userId, followingStatus});
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeFollowingStatus")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeFollowingStatus")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      auto iter = gFollowingStatusMap.find(userId);
      if (iter == gFollowingStatusMap.end()) {
        std::cout << "DisposeFollowingStatus called on a FollowingStatus with UserId that doesn't exist." << std::endl;
        return true;
      }

      std::shared_ptr<IFollowingStatus> followingStatus = iter->second;
      TTV_ErrorCode ec = followingStatus->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gFollowingStatusMap.erase(iter);
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("CreateSubscribersStatus")
    .AddFunction()
    .Description("Calls ChatAPI::CreateSubscribersStatus")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      auto iter = gSubscribersStatusMap.find(userId);
      if (iter != gSubscribersStatusMap.end()) {
        std::cout << "Already created SubscribersStatus for UserId." << std::endl;
        return true;
      }

      std::shared_ptr<ISubscribersListener> listener = std::make_shared<SubscribersListener>(userId);
      std::shared_ptr<ISubscribersStatus> subscribersStatus;

      TTV_ErrorCode ec = gChatApi->CreateSubscribersStatus(userId, listener, subscribersStatus);
      gSubscribersStatusMap.insert({userId, subscribersStatus});
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeSubscribersStatus")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeSubscribersStatus")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      auto iter = gSubscribersStatusMap.find(userId);
      if (iter == gSubscribersStatusMap.end()) {
        std::cout << "DisposeSubscribersStatus called on a SubscribersStatus with UserId that doesn't exist."
                  << std::endl;
        return true;
      }

      std::shared_ptr<ISubscribersStatus> subscribersStatus = iter->second;
      TTV_ErrorCode ec = subscribersStatus->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gSubscribersStatusMap.erase(iter);
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CreateChatChannelProperties")
    .AddFunction()
    .Description("Calls ChatAPI::CreateChatChannelProperties")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChatChannelPropertiesMap.find(pair);
      if (iter != gChatChannelPropertiesMap.end()) {
        std::cout << "Already created ChatChannelProperties for UserId/ChannelId pair." << std::endl;
        return true;
      }

      std::shared_ptr<IChatChannelPropertyListener> listener =
        std::make_shared<ChatChannelPropertyListener>(pair.first, pair.second);
      std::shared_ptr<IChatChannelProperties> chatChannelProperties;

      TTV_ErrorCode ec =
        gChatApi->CreateChatChannelProperties(pair.first, pair.second, listener, chatChannelProperties);
      gChatChannelPropertiesMap.insert({pair, chatChannelProperties});
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeChatChannelProperties")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeChatChannelProperties")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChatChannelPropertiesMap.find(pair);
      if (iter == gChatChannelPropertiesMap.end()) {
        std::cout
          << "DisposeChatChannelProperties called on a ChatChannelProperties with UserId/ChannelId pair that doesn't exist."
          << std::endl;
        return true;
      }

      std::shared_ptr<IChatChannelProperties> chatChannelProperties = iter->second;
      TTV_ErrorCode ec = chatChannelProperties->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gChatChannelPropertiesMap.erase(iter);
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CreateChannelChatRoomManager")
    .AddFunction()
    .Description("Calls ChatAPI::CreateChannelChatRoomManager")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChannelChatRoomManagerMap.find(pair);
      if (iter != gChannelChatRoomManagerMap.end()) {
        std::cout << "Already created ChannelChatRoomManager for UserId/ChannelId pair." << std::endl;
        return true;
      }

      std::shared_ptr<IChannelChatRoomManager> channelChatRoomManager;

      TTV_ErrorCode ec = gChatApi->CreateChannelChatRoomManager(
        params[0].GetUInt32(), params[1].GetUInt32(), gChannelChatRoomManagerListener, channelChatRoomManager);
      if (TTV_SUCCEEDED(ec)) {
        gChannelChatRoomManagerMap.insert({pair, channelChatRoomManager});
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeChannelChatRoomManager")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeChannelChatRoomManager")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChannelChatRoomManagerMap.find(pair);
      if (iter == gChannelChatRoomManagerMap.end()) {
        std::cout
          << "DisposeUserChatRoomManager called on a ChannelChatRoomManager with UserId/ChannelId pair that doesn't exist."
          << std::endl;
        return true;
      }

      std::shared_ptr<IChannelChatRoomManager> channelChatRoomManager = iter->second;
      TTV_ErrorCode ec = channelChatRoomManager->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gChannelChatRoomManagerMap.erase(iter);
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CreateChatRoomNotifications")
    .AddFunction()
    .Description("Calls ChatAPI::CreateChatRoomNotifications")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto iter = gChatRoomNotificationsMap.find(params[0].GetUInt32());
      if (iter != gChatRoomNotificationsMap.end()) {
        std::cout << "Already created ChatRoomNotifications for UserId." << std::endl;
        return true;
      }

      std::shared_ptr<IChatRoomNotifications> chatRoomNotifications;

      TTV_ErrorCode ec = gChatApi->CreateChatRoomNotifications(
        params[0].GetUInt32(), gChatRoomNotificationsListener, chatRoomNotifications);
      if (TTV_SUCCEEDED(ec)) {
        gChatRoomNotificationsMap.insert({params[0].GetUInt32(), chatRoomNotifications});
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeChatRoomNotifications")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeChatRoomNotifications")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto iter = gChatRoomNotificationsMap.find(params[0].GetUInt32());
      if (iter == gChatRoomNotificationsMap.end()) {
        std::cout << "DisposeChatRoomNotifications called on a ChatRoomNotifications with UserId that doesn't exist."
                  << std::endl;
        return true;
      }

      std::shared_ptr<IChatRoomNotifications> chatRoomNotifications = iter->second;
      TTV_ErrorCode ec = chatRoomNotifications->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gChatRoomNotificationsMap.erase(iter);
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CreateChatRoom")
    .AddFunction()
    .Description("Calls ChatAPI::CreateChatRoom")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter != gChatRoomMap.end()) {
        std::cout << "Already created ChatRoom for UserId/RoomId pair." << std::endl;
        return true;
      }

      std::shared_ptr<IChatRoom> chatRoom;

      TTV_ErrorCode ec = gChatApi->CreateChatRoom(
        params[0].GetUInt32(), params[1].GetString(), params[2].GetUInt32(), gChatRoomListener, chatRoom);
      if (TTV_SUCCEEDED(ec)) {
        gChatRoomMap.insert({pair, chatRoom});
      }
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("DisposeChatRoom")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeChatRoom")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "DisposeChatRoom called on a ChatRoom with UserId/RoomId pair that doesn't exist." << std::endl;
        return true;
      }

      std::shared_ptr<IChatRoom> chatRoom = iter->second;
      TTV_ErrorCode ec = chatRoom->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gChatRoomMap.erase(iter);
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatRoomDelete")
    .AddFunction()
    .Description("Calls ChatRoom::DeleteRoom")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->DeleteRoom([](TTV_ErrorCode ec) {
        std::cout << "ChatRoomDelete callback received" << std::endl;
        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomSendMessage")
    .AddFunction()
    .Description("Calls ChatRoom::SendMessage")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("message", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      ChatRoomMessage placeholder;
      TTV_ErrorCode ec = iter->second->SendMessage(params[2].GetString(), placeholder,
        [](TTV_ErrorCode ec, SendRoomMessageError&& error, ChatRoomMessage&& message) {
          std::cout << "ChatRoomSendMessage callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(message), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomEditMessage")
    .AddFunction()
    .Description("Calls ChatRoom::EditMessage")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("messageId", ParamType::String)
    .AddParam("message", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      ChatRoomMessage placeholder;
      TTV_ErrorCode ec = iter->second->EditMessage(
        params[2].GetString(), params[3].GetString(), placeholder, [](TTV_ErrorCode ec, ChatRoomMessage&& message) {
          std::cout << "ChatRoomEditMessage callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(message), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomDeleteMessage")
    .AddFunction()
    .Description("Calls ChatRoom::DeleteMessage")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("messageId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->DeleteMessage(params[2].GetString(), [params](TTV_ErrorCode ec) {
        if (TTV_SUCCEEDED(ec)) {
          std::cout << "Message " << params[2].GetString() << " deleted" << std::endl;
        }
        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomSetMuted")
    .AddFunction()
    .Description("Calls ChatRoom::SetMuted")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("isMuted", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->SetMuted(params[2].GetBool(), [](TTV_ErrorCode ec, ChatRoomInfo&& info) {
        std::cout << "ChatRoomSetMuted callback received" << std::endl;

        if (TTV_SUCCEEDED(ec)) {
          Print(std::move(info), "  ");
        }
        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomSetArchived")
    .AddFunction()
    .Description("Calls ChatRoom::SetArchived")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("isArchived", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->SetArchived(params[2].GetBool(), [](TTV_ErrorCode ec, ChatRoomInfo&& info) {
        std::cout << "ChatRoomSetArchived callback received" << std::endl;

        if (TTV_SUCCEEDED(ec)) {
          Print(std::move(info), "  ");
        }
        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomSetLastReadAt")
    .AddFunction()
    .Description("Calls ChatRoom::SetLastReadAt")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("lastReadAt", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      Timestamp timestamp;
      RFC3339TimeToUnixTimestamp(params[2].GetString(), timestamp);

      TTV_ErrorCode ec = iter->second->SetLastReadAt(timestamp, [](TTV_ErrorCode ec, ChatRoomInfo&& info) {
        std::cout << "ChatRoomSetLastReadAt callback received" << std::endl;

        if (TTV_SUCCEEDED(ec)) {
          Print(std::move(info), "  ");
        }
        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomSetName")
    .AddFunction()
    .Description("Calls ChatRoom::SetRoomName")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("name", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->SetRoomName(
        params[2].GetString(), [](TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomSetName callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(error, "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomSetTopic")
    .AddFunction()
    .Description("Calls ChatRoom::SetTopic")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("name", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->SetTopic(
        params[2].GetString(), [](TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomSetTopic callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomSetRoomRolePermissions")
    .AddFunction()
    .Description("Calls ChatRoom::SetRoomRolePermissions")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("minimumReadRole", ParamType::String)
    .AddParam("minimumSendRole", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      RoomRolePermissions permissions;
      ttv::json::ToObject(ttv::json::Value(params[2].GetString()), permissions.read);
      ttv::json::ToObject(ttv::json::Value(params[3].GetString()), permissions.send);

      if (permissions.read == RoomRole::Unknown || permissions.send == RoomRole::Unknown) {
        std::cout << "Roles must be one of [EVERYONE, SUBSCRIBER, MODERATOR, BROADCASTER]" << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->SetRoomRolePermissions(
        permissions, [](TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomSetMinimumRole callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomEnableSlowMode")
    .AddFunction()
    .Description("Calls ChatRoom::EnableSlowMode")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("durationSeconds", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->EnableSlowMode(
        params[2].GetUInt32(), [](TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomEnableSlowMode callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomDisableSlowMode")
    .AddFunction()
    .Description("Calls ChatRoom::DisableSlowMode")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec =
        iter->second->DisableSlowMode([](TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomDisableSlowMode callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomEnableR9kMode")
    .AddFunction()
    .Description("Calls ChatRoom::EnableR9kMode")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec =
        iter->second->EnableR9kMode([](TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomEnableR9kMode callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomDisableR9kMode")
    .AddFunction()
    .Description("Calls ChatRoom::DisableR9kMode")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec =
        iter->second->DisableR9kMode([](TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomDisableR9kMode callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomEnableEmotesOnlyMode")
    .AddFunction()
    .Description("Calls ChatRoom::EnableEmotesOnlyMode")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec =
        iter->second->EnableEmotesOnlyMode([](TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomEnableEmotesOnlyMode callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomDisableEmotesOnlyMode")
    .AddFunction()
    .Description("Calls ChatRoom::DisableEmotesOnlyMode")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec =
        iter->second->DisableEmotesOnlyMode([](TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
          std::cout << "ChatRoomDisableEmotesOnlyMode callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomFetchMessagesBeforeCursor")
    .AddFunction()
    .Description("Calls ChatRoom::FetchMessagesBeforeCursor")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("cursor", ParamType::String)
    .AddParam("limit", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->FetchMessagesBeforeCursor(params[2].GetString(), params[3].GetUInt32(),
        [](TTV_ErrorCode ec, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
          std::cout << "ChatRoomFetchMessagesBeforeCursor callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(messages), "  ");
            std::cout << "  Next cursor: " << nextCursor << std::endl;
            std::cout << "  More messages to fetch: " << moreMessages << std::endl;
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomFetchMessagesAfterCursor")
    .AddFunction()
    .Description("Calls ChatRoom::FetchMessagesAfterCursor")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("cursor", ParamType::String)
    .AddParam("limit", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->FetchMessagesAfterCursor(params[2].GetString(), params[3].GetUInt32(),
        [](TTV_ErrorCode ec, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
          std::cout << "ChatRoomFetchMessagesAfterCursor callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(messages), "  ");
            std::cout << "  Next cursor: " << nextCursor << std::endl;
            std::cout << "  More messages to fetch: " << moreMessages << std::endl;
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomFetchMessagesBeforeTimestamp")
    .AddFunction()
    .Description("Calls ChatRoom::FetchMessagesBeforeTimestamp")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("timestamp", ParamType::String)
    .AddParam("limit", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      Timestamp timestamp;
      RFC3339TimeToUnixTimestamp(params[2].GetString(), timestamp);

      TTV_ErrorCode ec = iter->second->FetchMessagesBeforeTimestamp(timestamp, params[3].GetUInt32(),
        [](TTV_ErrorCode ec, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
          std::cout << "ChatRoomFetchMessagesBeforeTimestamp callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(messages), "  ");
            std::cout << "  Next cursor: " << nextCursor << std::endl;
            std::cout << "  More messages to fetch: " << moreMessages << std::endl;
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomFetchMessagesAfterTimestamp")
    .AddFunction()
    .Description("Calls ChatRoom::FetchMessagesAfterTimestamp")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .AddParam("timestamp", ParamType::String)
    .AddParam("limit", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      Timestamp timestamp;
      RFC3339TimeToUnixTimestamp(params[2].GetString(), timestamp);

      TTV_ErrorCode ec = iter->second->FetchMessagesAfterTimestamp(timestamp, params[3].GetUInt32(),
        [](TTV_ErrorCode ec, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
          std::cout << "ChatRoomFetchMessagesAfterTimestamp callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(messages), "  ");
            std::cout << "  Next cursor: " << nextCursor << std::endl;
            std::cout << "  More messages to fetch: " << moreMessages << std::endl;
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChatRoomFetchInfo")
    .AddFunction()
    .Description("Calls ChatRoom::FetchMessages")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("roomId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, std::string> pair = {params[0].GetUInt32(), params[1].GetString()};
      auto iter = gChatRoomMap.find(pair);
      if (iter == gChatRoomMap.end()) {
        std::cout << "Could not find a ChatRoom for given UserId/RoomId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->FetchRoomInfo([](TTV_ErrorCode ec, ChatRoomInfo&& info) {
        std::cout << "ChatRoomFetchInfo callback received" << std::endl;

        if (TTV_SUCCEEDED(ec)) {
          Print(std::move(info), "  ");
        }
        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChannelChatRoomManagerCreateNewRoom")
    .AddAlias("ChatCreateNewRoom")
    .AddFunction()
    .Description("Calls ChannelChatRoomManager::AddNewChatRoom")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("roomName", ParamType::String)
    .AddParam("topic", ParamType::String)
    .AddParam("minimumReadRole", ParamType::String)
    .AddParam("minimumSendRole", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChannelChatRoomManagerMap.find(pair);
      if (iter == gChannelChatRoomManagerMap.end()) {
        std::cout << "Could not find a ChannelChatRoomManager for given UserId/ChannelId pair." << std::endl;
        return true;
      }

      RoomRolePermissions permissions;
      ttv::json::ToObject(ttv::json::Value(params[4].GetString()), permissions.read);
      ttv::json::ToObject(ttv::json::Value(params[5].GetString()), permissions.send);

      if (permissions.read == RoomRole::Unknown || permissions.send == RoomRole::Unknown) {
        std::cout << "Roles must be one of [EVERYONE, SUBSCRIBER, MODERATOR, BROADCASTER]" << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->AddNewChatRoom(params[2].GetString(), params[3].GetString(), permissions,
        [](TTV_ErrorCode ec, CreateRoomError&& error, ChatRoomInfo&& info) {
          std::cout << "AddNewChatRoom callback received" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(info), "  ");
          } else if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChannelChatRoomManagerJoinChatRooms")
    .AddAlias("ChatJoinRooms")
    .AddFunction()
    .Description("Calls ChannelChatRoomManager::JoinChatRooms")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChannelChatRoomManagerMap.find(pair);
      if (iter == gChannelChatRoomManagerMap.end()) {
        std::cout << "Could not find a ChannelChatRoomManager for given UserId/ChannelId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->JoinChatRooms([](TTV_ErrorCode ec) {
        std::cout << "JoinChatRooms callback received" << std::endl;

        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChannelChatRoomManagerLeaveChannel")
    .AddAlias("ChatLeaveRooms")
    .AddFunction()
    .Description("Calls ChannelChatRoomManager::LeaveChannelChatRooms")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChannelChatRoomManagerMap.find(pair);
      if (iter == gChannelChatRoomManagerMap.end()) {
        std::cout << "Could not find a ChannelChatRoomManager for given UserId/ChannelId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->LeaveChatRooms([](TTV_ErrorCode ec) {
        std::cout << "LeaveChannelChatRooms callback received" << std::endl;

        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ChannelChatRoomManagerFetchRooms")
    .AddAlias("ChatFetchChannelRooms")
    .AddFunction()
    .Description("Calls ChannelChatRoomManager::FetchChatRoomsInfo")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::pair<UserId, ChannelId> pair = {params[0].GetUInt32(), params[1].GetUInt32()};
      auto iter = gChannelChatRoomManagerMap.find(pair);
      if (iter == gChannelChatRoomManagerMap.end()) {
        std::cout << "Could not find a ChannelChatRoomManager for given UserId/ChannelId pair." << std::endl;
        return true;
      }

      TTV_ErrorCode ec = iter->second->FetchChatRoomsInfo([](TTV_ErrorCode ec, std::vector<ChatRoomInfo>&& infos) {
        std::cout << "FetchChatRoomsInfo callback received" << std::endl;

        if (TTV_SUCCEEDED(ec)) {
          Print(std::move(infos), "  ");
        }

        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("CreateSubscriptionsNotifications")
    .AddFunction()
    .Description("Calls ChatAPI::CreateSubscriptionsNotifications")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      auto& subscriptionsNotificationsEntry = gSubscriptionsNotificationsMap[userId];
      if (subscriptionsNotificationsEntry != nullptr) {
        ReportCommandResult(TTV_EC_ALREADY_INITIALIZED);
        return true;
      }

      auto listener = std::make_shared<SubscriptionsNotificationsListener>();
      auto result = gChatApi->CreateSubscriptionsNotifications(userId, listener);
      if (!result.IsError()) {
        subscriptionsNotificationsEntry = result.GetResult();
      }

      ReportCommandResult(result.GetErrorCode());

      return true;
    });

  category->AddCommand("DisposeSubscriptionsNotifications")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeSubscriptionsNotifications")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      UserId userId = params[0].GetUInt32();
      auto iter = gSubscriptionsNotificationsMap.find(userId);
      if (iter == gSubscriptionsNotificationsMap.end()) {
        std::cout
          << "DisposeSubscriptionsNotifications called on a SubscriptionsNotifications with UserId that doesn't exist."
          << std::endl;
        return true;
      }

      std::shared_ptr<ISubscriptionsNotifications> subscriptionsNotifications = iter->second;
      TTV_ErrorCode ec = subscriptionsNotifications->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gSubscriptionsNotificationsMap.erase(iter);
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CreateSquadNotifications")
    .AddFunction()
    .Description("Calls ChatAPI::CreateSquadNotifications")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("squadId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string squadId = params[1].GetString();
      auto& squadNotificationsEntry = gSquadNotificationsMap[squadId];
      if (squadNotificationsEntry != nullptr) {
        ReportCommandResult(TTV_EC_ALREADY_INITIALIZED);
        return true;
      }

      auto listener = std::make_shared<SquadNotificationsListener>();
      auto result = gChatApi->CreateSquadNotifications(params[0].GetUInt32(), squadId, listener);
      if (!result.IsError()) {
        squadNotificationsEntry = result.GetResult();
      }

      ReportCommandResult(result.GetErrorCode());

      return true;
    });

  category->AddCommand("DisposeSquadNotifications")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeSquadNotifications")
    .AddParam("squadId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto iter = gSquadNotificationsMap.find(params[0].GetString());
      if (iter == gSquadNotificationsMap.end()) {
        std::cout << "DisposeSquadNotifications called on a SquadNotifications with squad ID that doesn't exist."
                  << std::endl;
        return true;
      }

      std::shared_ptr<ISquadNotifications> squadNotifications = iter->second;
      TTV_ErrorCode ec = squadNotifications->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gSquadNotificationsMap.erase(iter);
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("CreateMultiviewNotifications")
    .AddFunction()
    .Description("Calls ChatAPI::CreateMultiviewNotifications")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      ChannelId channelId = params[1].GetUInt32();
      auto& multiviewNotificationsEntry = gMultiviewNotificationsMap[channelId];
      if (multiviewNotificationsEntry != nullptr) {
        ReportCommandResult(TTV_EC_ALREADY_INITIALIZED);
        return true;
      }

      auto listener = std::make_shared<MultiviewNotificationsListener>();
      auto result = gChatApi->CreateMultiviewNotifications(params[0].GetUInt32(), channelId, listener);
      if (!result.IsError()) {
        multiviewNotificationsEntry = result.GetResult();
      }

      ReportCommandResult(result.GetErrorCode());

      return true;
    });

  category->AddCommand("DisposeMultiviewNotifications")
    .AddFunction()
    .Description("Calls ChatAPI::DisposeMultiviewNotifications")
    .AddParam("squadId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto iter = gSquadNotificationsMap.find(params[0].GetString());
      if (iter == gSquadNotificationsMap.end()) {
        std::cout << "DisposeSquadNotifications called on a SquadNotifications with squad ID that doesn't exist."
                  << std::endl;
        return true;
      }

      std::shared_ptr<ISquadNotifications> squadNotifications = iter->second;
      TTV_ErrorCode ec = squadNotifications->Dispose();
      if (TTV_SUCCEEDED(ec)) {
        gSquadNotificationsMap.erase(iter);
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatBanUser")
    .AddFunction()
    .Description("Calls ChatAPI::BanUser")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("bannedUserName", ParamType::String)
    .AddParam("duration", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->BanUser(params[0].GetUInt32(), params[1].GetUInt32(), params[2].GetString(),
        params[3].GetUInt32(), [](TTV_ErrorCode ec, BanUserError&& error) {
          std::cout << "BanUser completed" << std::endl;

          if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }

          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatUnbanUser")
    .AddFunction()
    .Description("Calls ChatAPI::UnbanUser")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("bannedUserName", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->UnbanUser(params[0].GetUInt32(), params[1].GetUInt32(), params[2].GetString(),
        [](TTV_ErrorCode ec, UnbanUserError&& error) {
          std::cout << "UnbanUser completed" << std::endl;

          if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }

          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatModUser")
    .AddFunction()
    .Description("Calls ChatAPI::ModUser")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("modUserName", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->ModUser(params[0].GetUInt32(), params[1].GetUInt32(), params[2].GetString(),
        [](TTV_ErrorCode ec, ModUserError&& error) {
          std::cout << "ModUser completed" << std::endl;

          if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }

          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatUnmodUser")
    .AddFunction()
    .Description("Calls ChatAPI::UnmodUser")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("unmodUserName", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->UnmodUser(params[0].GetUInt32(), params[1].GetUInt32(), params[2].GetString(),
        [](TTV_ErrorCode ec, UnmodUserError&& error) {
          std::cout << "UnmodUser completed" << std::endl;

          if (error.code != GraphQLErrorCode::SUCCESS) {
            Print(std::move(error), "  ");
          }

          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatGrantVIP")
    .AddFunction()
    .Description("Calls ChatAPI::GrantVIP")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("vipUserName", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->GrantVIP(params[0].GetUInt32(), params[1].GetUInt32(), params[2].GetString(),
        [](TTV_ErrorCode ec, GrantVIPErrorCode error) {
          std::cout << "GrantVIP completed" << std::endl;

          if (error != GrantVIPErrorCode::SUCCESS) {
            Print(error, "  ");
          }

          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatRevokeVIP")
    .AddFunction()
    .Description("Calls ChatAPI::RevokeVIP")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("unvipUserName", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->RevokeVIP(params[0].GetUInt32(), params[1].GetUInt32(), params[2].GetString(),
        [](TTV_ErrorCode ec, RevokeVIPErrorCode error) {
          std::cout << "RevokeVIP completed" << std::endl;

          if (error != RevokeVIPErrorCode::SUCCESS) {
            Print(error, "  ");
          }

          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchChannelVIPs")
    .AddFunction()
    .Description("Calls ChatAPI::FetchChannelVIPs")
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gChatApi->FetchChannelVIPs(params[0].GetUInt32(), [](TTV_ErrorCode ec, std::vector<std::string>&& vipNames) {
          std::cout << "FetchChannelVIPs completed" << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Vips: ";
            for (const auto& name : vipNames) {
              std::cout << name << " ";
            }
            std::cout << std::endl;
          }

          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatUpdateUserColor")
    .AddFunction()
    .Description("Calls ChatAPI::UpdateUserColor")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("color", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->UpdateUserColor(params[0].GetUInt32(), params[1].GetString(), [](TTV_ErrorCode ec) {
        std::cout << "UpdateUserColor completed" << std::endl;
        ReportCommandResult(ec);
      });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchChannelMods")
    .AddFunction()
    .Description("Calls ChatAPI::FetchChannelMods")
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("cursor", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->FetchChannelModerators(params[0].GetUInt32(), params[1].GetString(),
        [](TTV_ErrorCode ec, const std::vector<std::string>& modNames, const std::string& nextCursor) {
          std::cout << "FetchChannelMods completed." << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Mods: ";
            for (const auto& name : modNames) {
              std::cout << name << " ";
            }
            std::cout << std::endl;

            std::cout << "  Next cursor: " << nextCursor << std::endl;
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ChatFetchUserListForChannel")
    .AddFunction()
    .Description("Calls ChatAPI::FetchUserListForChannel")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gChatApi->FetchUserListForChannel(
        params[0].GetUInt32(), params[1].GetUInt32(), [](TTV_ErrorCode ec, UserList&& userList) {
          std::cout << "FetchUserListForChannel completed." << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(std::move(userList), "  ");
          }
          ReportCommandResult(ec);
        });

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("PrintJavaChatErrorCodes")
    .AddFunction()
    .Description("Dumps the Java snippet for updated values in ChatErrorCode.java")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      std::vector<EnumValue> values;
      ttv::chat::GetChatErrorCodeValues(values);

      for (const auto& v : values) {
        std::cout << "public static final ErrorCode " << v.name << " = new ChatErrorCode(0x" << std::hex << v.value
                  << ", \"" << v.name << "\");" << std::endl;
      }

      return true;
    });

  return category;
}

StaticInitializer gStaticInitializer([]() {
  StaticInitializationRegistrar::GetInstance().RegisterInitializer(
    [](std::shared_ptr<Context> context) {
      context->RegisterLoggers(kTraceComponents, sizeof(kTraceComponents) / sizeof(kTraceComponents[0]));

      auto coreApi = std::static_pointer_cast<CoreAPI>(context->GetModule("ttv::CoreAPI"));

      gChatApi = std::make_shared<ChatAPI>();
      gChatApi->SetCoreApi(coreApi);
      context->RegisterModule(gChatApi);

      gChatApiListener = std::make_shared<ChatApiListener>();
      gChatChannelListener = std::make_shared<ChatChannelListener>();
      gChatWhisperListener = std::make_shared<ChatChannelListener>();
      gChatUserThreadsListener = std::make_shared<ChatUserThreadsListener>();
      gChatCommentListener = std::make_shared<ChatCommentListener>();
      gChatRaidListener = std::make_shared<ChatRaidListener>();
      gChatRoomListener = std::make_shared<ChatRoomListener>();
      gChannelChatRoomManagerListener = std::make_shared<ChannelChatRoomManagerListener>();
      gChatRoomNotificationsListener = std::make_shared<ChatRoomNotificationsListener>();

      auto commands = RegisterChatCommands(context);
      context->RegisterCommandCategory(commands);
    },
    1);

  StaticInitializationRegistrar::GetInstance().RegisterUserLoginFunction(
    [](std::shared_ptr<Context> /*context*/, UserId userId) {
      gChatApi->SetUserThreadsListener(userId, gChatUserThreadsListener);
    });
});
}  // namespace
