/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/bitsconfigrepository.h"

#include "twitchsdk/chat/internal/graphql/fetchchannelcheermotesqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatgetbitsconfigtask.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/graphqltask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/user/user.h"

namespace {
const char* kLoggerName = "BitsConfigRepository";
}

ttv::chat::BitsConfigRepository::BitsConfigRepository()
    : mBitsConfigCache(10), mCurrentLookupId(1), mLookupInProgress(false) {}

std::string ttv::chat::BitsConfigRepository::GetLoggerName() const {
  return kLoggerName;
}

void ttv::chat::BitsConfigRepository::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  mCancelledBitsConfigLookups.Flush(TTV_EC_REQUEST_ABORTED, nullptr);

  if (!mLookupInProgress) {
    ScheduleLookup();
  }

  Component::Update();
}

TTV_ErrorCode ttv::chat::BitsConfigRepository::Shutdown() {
  TTV_ErrorCode ec = Component::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    for (auto& pair : mPendingBitsConfigLookups) {
      pair.second->Flush(TTV_EC_SHUTTING_DOWN, nullptr);
    }

    mPendingBitsConfigLookups.clear();
  }

  return ec;
}

bool ttv::chat::BitsConfigRepository::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  if (mLookupInProgress) {
    return false;
  }

  if (!mPendingBitsConfigLookups.empty()) {
    return false;
  }

  return true;
}

void ttv::chat::BitsConfigRepository::CompleteShutdown() {
  Component::CompleteShutdown();

  TTV_ASSERT(!mLookupInProgress);

  mBitsConfigCache.Clear();
}

TTV_ErrorCode ttv::chat::BitsConfigRepository::FetchGlobalBitsConfiguration(LookupCallback callback) {
  return FetchChannelBitsConfiguration(0, 0, callback);
}

TTV_ErrorCode ttv::chat::BitsConfigRepository::FetchChannelBitsConfiguration(
  UserId userId, ChannelId channelId, LookupCallback callback) {
  LookupId tmp;
  return FetchChannelBitsConfiguration(userId, channelId, callback, tmp);
}

TTV_ErrorCode ttv::chat::BitsConfigRepository::FetchGlobalBitsConfiguration(LookupCallback callback, LookupId& token) {
  return FetchChannelBitsConfiguration(0, 0, callback, token);
}

TTV_ErrorCode ttv::chat::BitsConfigRepository::FetchChannelBitsConfiguration(
  UserId userId, ChannelId channelId, LookupCallback callback, LookupId& token) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  std::pair<UserId, ChannelId> userChannelPair = {userId, channelId};
  auto it = mPendingBitsConfigLookups.find(userChannelPair);

  token = mCurrentLookupId;
  mCurrentLookupId++;

  mChannelLookups[token] = userChannelPair;

  if (it != mPendingBitsConfigLookups.end()) {
    it->second->Push(callback, token);
  } else {
    auto queue = std::make_shared<CallbackQueue<LookupCallback>>();
    queue->Push(callback, token);

    mPendingBitsConfigLookups[userChannelPair] = queue;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::BitsConfigRepository::CancelFetch(LookupId token) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  auto lookupIt = mChannelLookups.find(token);
  if (lookupIt != mChannelLookups.end()) {
    auto it = mPendingBitsConfigLookups.find(lookupIt->second);
    if (it != mPendingBitsConfigLookups.end()) {
      // Move the callback to the cancelled lookup queue to be called in Update()
      LookupCallback cancelledCallback = it->second->Erase(token);
      if (cancelledCallback != nullptr) {
        mCancelledBitsConfigLookups.Push(std::move(cancelledCallback));
      }
      return TTV_EC_SUCCESS;
    } else {
      return TTV_EC_NOT_AVAILABLE;
    }
  } else {
    return TTV_EC_NOT_AVAILABLE;
  }
}

TTV_ErrorCode ttv::chat::BitsConfigRepository::ScheduleLookup() {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (mLookupInProgress) {
    return TTV_EC_REQUEST_PENDING;
  }

  if (mPendingBitsConfigLookups.empty()) {
    return TTV_EC_SUCCESS;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto iter = mPendingBitsConfigLookups.begin();
  while (iter != mPendingBitsConfigLookups.end()) {
    UserId userId = iter->first.first;
    ChannelId channelId = iter->first.second;
    std::shared_ptr<CallbackQueue<LookupCallback>> queue = iter->second;

    std::shared_ptr<BitsConfiguration> config;
    if (mBitsConfigCache.GetEntry(iter->first, config)) {
      mPendingBitsConfigLookups.erase(iter);  // Erase before we flush to avoid reentrancy issues
      queue->Flush(TTV_EC_SUCCESS, config);
    } else {
      // Fetch if we don't have cached value
      if (channelId == 0) {
        // Global cheermotes, old v5 endpoint
        auto fetchCallback = [this](ChatGetBitsConfigTask* source, TTV_ErrorCode callbackEc,
                               std::shared_ptr<ChatGetBitsConfigTask::Result> result) {
          CompleteTask(source);

          std::shared_ptr<BitsConfiguration> resultConfig;

          if (TTV_SUCCEEDED(callbackEc) && result != nullptr) {
            resultConfig = std::make_shared<BitsConfiguration>(std::move(result->cheermotes), 0, 0);

            CacheBitsConfig(resultConfig);
          }

          auto channelIter = mPendingBitsConfigLookups.find(std::make_pair(0, 0));
          if (channelIter != mPendingBitsConfigLookups.end()) {
            std::shared_ptr<CallbackQueue<LookupCallback>> callbackQueue = channelIter->second;
            mPendingBitsConfigLookups.erase(channelIter);  // Erase before we flush to avoid reentrancy issues
            callbackQueue->Flush(callbackEc, resultConfig);
          }

          mLookupInProgress = false;
        };

        ec = StartTask(std::make_shared<ChatGetBitsConfigTask>(0, fetchCallback));
      } else {
        graphql::FetchChannelCheermotesQueryInfo::InputParams input;
        input.channelId = std::to_string(channelId);

        // Channel cheermotes, GQL
        if (mUserRepository != nullptr) {
          auto user = mUserRepository->GetUser(userId);
          if (user != nullptr && user->GetUserId() != 0) {
            auto token = user->GetOAuthToken();
            input.authToken = token->GetToken();
          }
        }

        auto task = std::make_shared<GraphQLTask<graphql::FetchChannelCheermotesQueryInfo>>(
          std::move(input), [this, userId, channelId](GraphQLTask<graphql::FetchChannelCheermotesQueryInfo>* source,
                              Result<graphql::FetchChannelCheermotesQueryInfo::PayloadType>&& result) {
            CompleteTask(source);

            TTV_ErrorCode callbackEc = TTV_EC_SUCCESS;
            std::shared_ptr<BitsConfiguration> resultConfig;

            if (result.IsError()) {
              callbackEc = result.GetErrorCode();
            } else {
              auto& payload = result.GetResult();
              resultConfig = std::make_shared<BitsConfiguration>(std::move(payload.cheermotes), userId, channelId);

              CacheBitsConfig(resultConfig);
            }

            auto userChannelPairIter = mPendingBitsConfigLookups.find(std::make_pair(userId, channelId));
            if (userChannelPairIter != mPendingBitsConfigLookups.end()) {
              std::shared_ptr<CallbackQueue<LookupCallback>> callbackQueue = userChannelPairIter->second;
              mPendingBitsConfigLookups.erase(userChannelPairIter);  // Erase before we flush to avoid reentrancy issues
              callbackQueue->Flush(callbackEc, resultConfig);
            }

            mLookupInProgress = false;
          });

        ec = StartTask(task);
      }

      if (TTV_SUCCEEDED(ec)) {
        mLookupInProgress = true;
        break;
      } else {
        mPendingBitsConfigLookups.erase(iter);  // Erase before we flush to avoid reentrancy issues
        queue->Flush(ec, nullptr);
      }
    }
    iter = mPendingBitsConfigLookups
             .begin();  // Create a new iterator that starts at the beginning of mPendingBitsConfigLookups
  }

  return ec;
}

TTV_ErrorCode ttv::chat::BitsConfigRepository::CacheBitsConfig(const std::shared_ptr<BitsConfiguration> bitsConfig) {
  if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  mBitsConfigCache.SetEntry(std::make_pair(bitsConfig->GetUserId(), bitsConfig->GetChannelId()), bitsConfig);

  return TTV_EC_SUCCESS;
}
