/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/channel/bitsstatus.h"

#include "twitchsdk/chat/ibitslistener.h"
#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/stringutilities.h"

// Documentation: https://dev.twitch.tv/docs/pubsub

namespace {
const uint64_t kErrorFetchRetryMaxBackOff = 1000 * 60 * 2;
const uint64_t kErrorFetchRetryJitterMilliseconds = 1000;

const char* kLoggerName = "BitsStatus";
const char* kChannelBitsTopicPrefix = "channel-bits-events-v1.";
const char* kUserBitsTopicPrefix = "user-bits-updates-v1.";
}  // namespace

ttv::chat::BitsStatus::BitsStatus(const std::shared_ptr<User>& user)
    : PubSubComponent(user),
      mFetchBitsConfigRetryTimer(kErrorFetchRetryMaxBackOff, kErrorFetchRetryJitterMilliseconds),
      mChannelBitsPubSubTopic(kChannelBitsTopicPrefix + std::to_string(user->GetUserId())),
      mUserBitsPubSubTopic(kUserBitsTopicPrefix + std::to_string(user->GetUserId())),
      mBitsConfigFetchToken(0),
      mChannelId(user->GetUserId()),
      mHasFetchedBitsConfig(false) {
  AddTopic(mChannelBitsPubSubTopic);
  AddTopic(mUserBitsPubSubTopic);
}

std::string ttv::chat::BitsStatus::GetLoggerName() const {
  return kLoggerName;
}

// Component Overrides
TTV_ErrorCode ttv::chat::BitsStatus::Initialize() {
  TTV_ErrorCode ec = PubSubComponent::Initialize();
  if (TTV_SUCCEEDED(ec)) {
    // Start the retry timer - Need to fetch bitsConfig if we want to tokenize bits
    mFetchBitsConfigRetryTimer.ScheduleNextRetry();
  }

  return ec;
}

bool ttv::chat::BitsStatus::CheckShutdown() {
  if (!PubSubComponent::CheckShutdown()) {
    return false;
  }

  if (mBitsConfigFetchToken != 0) {
    return false;
  }

  return true;
}

TTV_ErrorCode ttv::chat::BitsStatus::Shutdown() {
  TTV_ErrorCode ec = PubSubComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    mFetchBitsConfigRetryTimer.Clear();
  }

  if (mBitsConfigFetchToken != 0 && mBitsConfigRepository != nullptr) {
    mBitsConfigRepository->CancelFetch(mBitsConfigFetchToken);
  }

  return ec;
}

void ttv::chat::BitsStatus::Update() {
  PubSubComponent::Update();

  if (mState == State::Initialized) {
    // Attempt to fetch bits configuration
    if (!mHasFetchedBitsConfig && mFetchBitsConfigRetryTimer.CheckNextRetry()) {
      FetchBitsConfig();
    }
  }
}

TTV_ErrorCode ttv::chat::BitsStatus::Dispose() {
  return PubSubComponent::Dispose();
}

bool ttv::chat::BitsStatus::IsReady() const {
  return mHasFetchedBitsConfig;
}

void ttv::chat::BitsStatus::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "BitsListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::BitsStatus::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  std::string type;
  if (!ParseString(msg, "message_type", type)) {
    Log(MessageLevel::Error, "Couldn't find pubsub message type, dropping");
    return;
  }

  const auto& jData = msg["data"];
  if (jData.isNull() || !jData.isObject()) {
    Log(MessageLevel::Error, "Pub sub message missing data, dropping");
    return;
  }

  if (topic == mChannelBitsPubSubTopic) {
    if (type == "bits_event") {
      BitsReceivedEvent event;

      const auto& jChannelName = jData["channel_name"];
      if (!jChannelName.isNull() && jChannelName.isString()) {
        event.channelName = jChannelName.asString();
      }

      const auto& jContext = jData["context"];
      if (!jContext.isNull() && jContext.isString()) {
        event.context = jContext.asString();
      }

      const auto& jUserName = jData["user_name"];
      if (!jUserName.isNull() && jUserName.isString()) {
        event.message.userName = jUserName.asString();
        event.message.nameColorARGB = GetRandomUserColor(event.message.userName);
      }

      const auto& jUserId = jData["user_id"];
      ParseUserId(jUserId, event.message.userId);

      const auto& jChannelId = jData["channel_id"];
      ParseChannelId(jChannelId, event.channelId);

      const auto& jTimestamp = jData["time"];
      ParseTimestamp(jTimestamp, event.message.timestamp);

      const auto& jBitsUsed = jData["bits_used"];
      ParseUInt32(jBitsUsed, event.bitsUsed);
      event.message.numBitsSent = event.bitsUsed;

      const auto& jTotalBitsUsed = jData["total_bits_used"];
      ParseUInt32(jTotalBitsUsed, event.totalBitsUsed);

      const auto& jBitsMessage = jData["chat_message"];
      ParseBitsReceivedMessage(jBitsMessage, event);

      BadgeEntitlement badge;
      const auto& jBadgeEntitlement = jData["badge_entitlement"];
      if (!jBadgeEntitlement.isNull() && jBadgeEntitlement.isObject()) {
        badge.isNewBadgeLevel = true;

        const auto& jNewLevel = jBadgeEntitlement["new_version"];
        ParseUInt32(jNewLevel, badge.newLevel);

        const auto& jPreviousLevel = jBadgeEntitlement["previous_version"];
        ParseUInt32(jPreviousLevel, badge.previousLevel);
      } else {
        badge.isNewBadgeLevel = false;
      }
      event.badge = badge;

      if (mListener != nullptr) {
        mListener->UserReceivedBits(event);
      }
    } else {
      Log(MessageLevel::Error, "Unrecognized pub-sub message type (%s), dropping", type.c_str());
      return;
    }
  } else if (topic == mUserBitsPubSubTopic) {
    if (type == "balance_update") {
      BitsSentEvent event;
      if (ttv::json::ToObject(jData, event)) {
        if (mListener != nullptr) {
          mListener->UserSentBits(std::move(event));
        }
      } else {
        // If we fail to parse it as a BitsSentEvent, then it's a general balance update not involving sending bits.
        uint32_t bitsBalance;
        if (!ttv::ParseUInt(jData, "balance", bitsBalance)) {
          return;
        }

        mListener->UserGainedBits(bitsBalance);
      }
    } else {
      Log(MessageLevel::Error, "Unrecognized pub-sub message type (%s), dropping", type.c_str());
      return;
    }
  }
}

void ttv::chat::BitsStatus::ParseBitsReceivedMessage(const ttv::json::Value& jBitsMessage, BitsReceivedEvent& event) {
  // Not filled out in Messsage: displayName, userMode

  if (!jBitsMessage.isNull() && jBitsMessage.isString()) {
    std::string message = jBitsMessage.asString();

    mTokenizationOptions.emoticons = false;  // We don't tokenize emotes in the bits message

    std::vector<std::string> localUserNames;
    auto user = mUser.lock();
    if (user != nullptr) {
      localUserNames.emplace_back(user->GetUserName());
      localUserNames.emplace_back(user->GetDisplayName());
    }

    std::map<std::string, std::vector<EmoteRange>> emoticonRanges;

    // We could be passing in a nullptr for mBitsConfiguration here if we haven't finished fetching it
    TokenizeServerMessage(
      message, mTokenizationOptions, emoticonRanges, mBitsConfiguration, localUserNames, event.message);
  }
}

TTV_ErrorCode ttv::chat::BitsStatus::FetchBitsConfig() {
  if (mBitsConfigFetchToken != 0) {
    return TTV_EC_REQUEST_PENDING;
  }

  UserId userId = 0;
  auto user = mUser.lock();
  if (user != nullptr) {
    userId = user->GetUserId();
  }

  TTV_ErrorCode ec = mBitsConfigRepository->FetchChannelBitsConfiguration(userId, mChannelId,
    [this](TTV_ErrorCode callbackEc, const std::shared_ptr<BitsConfiguration>& config) {
      if (TTV_SUCCEEDED(callbackEc)) {
        mFetchBitsConfigRetryTimer.Clear();
        mBitsConfiguration = config;
        mHasFetchedBitsConfig = true;
      } else {
        mFetchBitsConfigRetryTimer.ScheduleNextRetry();
      }

      mBitsConfigFetchToken = 0;
    },
    mBitsConfigFetchToken);

  if (TTV_FAILED(ec)) {
    mFetchBitsConfigRetryTimer.ScheduleNextRetry();
  }

  return ec;
}
