/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/channel/followersstatus.h"

#include "twitchsdk/chat/ifollowerslistener.h"
#include "twitchsdk/core/json/corejsonutil.h"

namespace {
const char* kLoggerName = "FollowersStatus";
const char* kTopicPrefix = "following.";
}  // namespace

ttv::chat::FollowersStatus::FollowersStatus(const std::shared_ptr<User>& user, ChannelId channelId)
    : PubSubComponent(user), mPubSubTopic(kTopicPrefix + std::to_string(channelId)) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::FollowersStatus::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::FollowersStatus::Dispose() {
  return PubSubComponent::Dispose();
}

void ttv::chat::FollowersStatus::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "FollowersListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::FollowersStatus::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    FollowerAddedEvent event;

    const auto& jDisplayName = msg["display_name"];
    if (!jDisplayName.isNull() && jDisplayName.isString()) {
      event.displayName = jDisplayName.asString();
    }

    const auto& jUserName = msg["username"];
    if (!jUserName.isNull() && jUserName.isString()) {
      event.userName = jUserName.asString();
    }
    const auto& jUserId = msg["user_id"];
    ParseUserId(jUserId, event.userId);

    if (mListener != nullptr) {
      mListener->NewFollowerAdded(event);
    }
  }
}
