/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/channel/followingstatus.h"

#include "twitchsdk/chat/ifollowinglistener.h"
#include "twitchsdk/core/json/corejsonutil.h"

namespace {
const char* kLoggerName = "FollowingStatus";
const char* kTopicPrefix = "follows.";
}  // namespace

ttv::chat::FollowingStatus::FollowingStatus(const std::shared_ptr<User>& user)
    : PubSubComponent(user), mPubSubTopic(kTopicPrefix + std::to_string(user->GetUserId())) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::FollowingStatus::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::FollowingStatus::Dispose() {
  return PubSubComponent::Dispose();
}

void ttv::chat::FollowingStatus::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "FollowingListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::FollowingStatus::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    ChannelId channelId;
    const auto& jChannelId = msg["target_user_id"];
    if (!ParseUserId(jChannelId, channelId)) {
      return;
    }

    UserId userId;
    auto user = mUser.lock();
    if (user == nullptr) {
      return;
    }

    userId = user->GetUserId();

    const auto& jType = msg["type"];
    if (jType.isString()) {
      if (jType.asString() == "user-followed") {
        mListener->FollowedChannel(userId, channelId);
      } else if (jType.asString() == "user-unfollowed") {
        mListener->UnfollowedChannel(userId, channelId);
      }
    }
  }
}
