/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/channel/subscribersstatus.h"

#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/isubscriberslistener.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/stringutilities.h"

// Documentation: https://dev.twitch.tv/docs/pubsub

// Can trigger a pub-sub to <channel_id> manually with the following curl command:
// curl -H "Content-Type: application/json" -X POST -d
// '{"user_id":"<user-id>","kind":"3000","product_name":"Subscription Name","tenure_months":5,"custom_message":"Custom
// Message"}' http://subscriptions.internal.justin.tv/internal/channels/<channel_id>/notification

namespace {
const char* kLoggerName = "SubscribersStatus";
const char* kTopicPrefix = "channel-subscribe-events-v1.";
}  // namespace

ttv::chat::SubscribersStatus::SubscribersStatus(const std::shared_ptr<User>& user)
    : PubSubComponent(user), mPubSubTopic(kTopicPrefix + std::to_string(user->GetUserId())) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::SubscribersStatus::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::SubscribersStatus::Dispose() {
  return PubSubComponent::Dispose();
}

void ttv::chat::SubscribersStatus::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "SubsListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::SubscribersStatus::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    SubscriberAddedEvent event;

    const auto& jUserName = msg["user_name"];
    if (!jUserName.isNull() && jUserName.isString()) {
      event.userName = jUserName.asString();
    }

    const auto& jDisplayName = msg["display_name"];
    if (!jDisplayName.isNull() && jDisplayName.isString()) {
      event.displayName = jDisplayName.asString();
    }

    const auto& jChannelName = msg["channel_name"];
    if (!jChannelName.isNull() && jChannelName.isString()) {
      event.channelName = jChannelName.asString();
    }

    const auto& jUserId = msg["user_id"];
    ParseUserId(jUserId, event.userId);

    const auto& jChannelId = msg["channel_id"];
    ParseChannelId(jChannelId, event.channelId);

    const auto& jTimestamp = msg["time"];
    ParseTimestamp(jTimestamp, event.timestamp);

    const auto& jSubMessage = msg["sub_message"];
    ParseSubscriptionMessage(jSubMessage, event);

    // Not filled out in SubscriptionNotice:
    // - systemMessage

    const auto& jSubPlanName = msg["sub_plan_name"];
    if (!jSubPlanName.isNull() && jSubPlanName.isString()) {
      event.subNotice.planDisplayName = jSubPlanName.asString();
    }

    const auto& jSubStreakMonths = msg["streak_months"];
    ParseUInt32(jSubStreakMonths, event.subNotice.subStreakMonthCount);

    const auto& jsubCumulativeMonths = msg["cumulative_months"];
    ParseUInt32(jsubCumulativeMonths, event.subNotice.subCumulativeMonthCount);

    const auto& jSubType = msg["context"];
    if (!jSubType.isNull() && jSubType.isString()) {
      const std::string& subType = jSubType.asString();

      if (subType == "sub") {
        event.subNotice.type = SubscriptionNotice::Type::Sub;
      } else if (subType == "resub") {
        event.subNotice.type = SubscriptionNotice::Type::Resub;
      } else if (subType == "charity") {
        event.subNotice.type = SubscriptionNotice::Type::Charity;
      } else if (subType == "extendsub") {
        event.subNotice.type = SubscriptionNotice::Type::ExtendSub;
      } else {
        event.subNotice.type = SubscriptionNotice::Type::Unknown;
      }
    }

    const auto& jSubPlan = msg["sub_plan"];
    if (!jSubPlan.isNull() && jSubPlan.isString()) {
      const std::string& subPlan = jSubPlan.asString();

      if (subPlan == "Prime") {
        event.subNotice.plan = SubscriptionNotice::Plan::Prime;
      } else if (subPlan == "1000") {
        event.subNotice.plan = SubscriptionNotice::Plan::Sub1000;
      } else if (subPlan == "2000") {
        event.subNotice.plan = SubscriptionNotice::Plan::Sub2000;
      } else if (subPlan == "3000") {
        event.subNotice.plan = SubscriptionNotice::Plan::Sub3000;
      } else {
        event.subNotice.plan = SubscriptionNotice::Plan::Unknown;
      }
    }

    if (mListener != nullptr) {
      mListener->NewSubscriberAdded(event);
    }
  }
}

void ttv::chat::SubscribersStatus::ParseSubscriptionMessage(
  const ttv::json::Value& jSubMessage, SubscriberAddedEvent& event) {
  auto tokenizedMessage = std::make_unique<MessageInfo>();
  tokenizedMessage->userId = event.userId;
  tokenizedMessage->timestamp = event.timestamp;
  tokenizedMessage->userName = event.userName;
  tokenizedMessage->displayName = event.displayName;
  tokenizedMessage->numBitsSent = 0;  // We can't cheer in the subscription message
  tokenizedMessage->nameColorARGB = GetRandomUserColor(event.userName);

  // Not filled out in Messsage: userMode

  if (!jSubMessage.isNull() && jSubMessage.isObject()) {
    const auto& jMessage = jSubMessage["message"];
    if (!jMessage.isNull() && jMessage.isString()) {
      std::string message = jMessage.asString();

      mTokenizationOptions.bits = false;  // We can't cheer in the subscription message

      std::map<std::string, std::vector<EmoteRange>> emoticonRanges;

      // Construct emoticonRanges from the JSON response
      const auto& jEmotes = jSubMessage["emotes"];
      if (!jEmotes.isNull() && jEmotes.isArray() && jEmotes.size() > 0) {
        for (const auto& jEntry : jEmotes) {
          if (!jEntry.isNull() && jEntry.isObject()) {
            std::string id;
            EmoteRange range;

            const auto& jEmoticonStart = jEntry["start"];
            const auto& jEmoticonEnd = jEntry["end"];

            if (ParseEmoticonId(jEntry, "id", id) && jEmoticonStart.isNumeric() && jEmoticonEnd.isNumeric()) {
              int start = static_cast<int>(jEmoticonStart.asInt());
              int end = static_cast<int>(jEmoticonEnd.asInt());

              range.startIndex = start;
              range.endIndex = end;

              auto& emoteList = emoticonRanges[id];
              emoteList.emplace_back(range);
            }
          }
        }
      }

      std::vector<std::string> localUserNames;
      auto user = mUser.lock();
      if (user != nullptr) {
        localUserNames.emplace_back(user->GetUserName());
        localUserNames.emplace_back(user->GetDisplayName());
      }

      TokenizeServerMessage(message, mTokenizationOptions, emoticonRanges, nullptr, localUserNames, *tokenizedMessage);
      event.subNotice.userMessage = std::move(tokenizedMessage);
    }
  }
}
