/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/channelchatroommanager.h"

#include "twitchsdk/chat/internal/chatroom.h"
#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"
#include "twitchsdk/chat/internal/task/chataddnewroomtask.h"
#include "twitchsdk/chat/internal/task/chatfetchchannelroomstask.h"
#include "twitchsdk/chat/internal/task/chatroomjoinchanneltask.h"
#include "twitchsdk/chat/internal/task/chatroomleavechanneltask.h"
#include "twitchsdk/core/mutex.h"

namespace {
const char* kLoggerName = "ChannelChatRoomManager";
const char* kTopicPrefix = "chatrooms-channel-v1.";
}  // namespace

ttv::chat::ChannelChatRoomManager::ChannelChatRoomManager(const std::shared_ptr<User>& user, ChannelId channelId)
    : PubSubComponent(user),
      mPubSubTopic(kTopicPrefix + std::to_string(channelId)),
      mUserId(user->GetUserId()),
      mChannelId(channelId) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::ChannelChatRoomManager::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::ChannelChatRoomManager::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChannelChatRoomManager::AddNewChatRoom(const std::string& roomName, const std::string& topic,
  RoomRolePermissions roomRolePermissions, const AddRoomCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (mUserId != mChannelId) {
    return TTV_EC_INVALID_CHANNEL_ID;
  }

  if (roomRolePermissions.read == RoomRole::Unknown || roomRolePermissions.send == RoomRole::Unknown) {
    return TTV_EC_INVALID_ARG;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatAddNewRoomTask>(roomName, topic, roomRolePermissions, oauthToken->GetToken(),
    [this, user, oauthToken, callback](
      ChatAddNewRoomTask* source, TTV_ErrorCode ec, CreateRoomError&& error, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error), std::move(info));
      }
    });

  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChannelChatRoomManager::JoinChatRooms(const JoinCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomJoinChannelTask>(mChannelId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomJoinChannelTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChannelChatRoomManager::LeaveChatRooms(const LeaveCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomLeaveChannelTask>(mChannelId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomLeaveChannelTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChannelChatRoomManager::FetchChatRoomsInfo(const FetchRoomsInfoCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatFetchChannelRoomsTask>(mChannelId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](
      ChatFetchChannelRoomsTask* source, TTV_ErrorCode ec, std::vector<ChatRoomInfo>&& infos) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(infos));
      }
    });

  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

void ttv::chat::ChannelChatRoomManager::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum /*state*/, TTV_ErrorCode /*ec*/) {}

void ttv::chat::ChannelChatRoomManager::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) {
  if (jVal.isNull() || !jVal.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    std::string type;
    if (!ParseString(jVal, "type", type)) {
      Log(MessageLevel::Error, "Couldn't find pubsub message type, dropping");
      return;
    }

    const auto& jData = jVal["data"];
    if (jData.isNull() || !jData.isObject()) {
      Log(MessageLevel::Error, "Pub sub message missing data, dropping");
      return;
    }

    if (type == "purge_messages_request") {
      UserId userId;
      const auto& jUserId = jData["user_id"];
      if (!ParseUserId(jUserId, userId)) {
        return;
      }

      ChannelId channelId;
      const auto& jChannelId = jData["channel_id"];
      ParseChannelId(jChannelId, channelId);
      if (channelId != mChannelId) {
        return;
      }

      Timestamp purgeAfter;
      if (!ParseTimestamp(jData, "purge_after", purgeAfter)) {
        return;
      }

      mListener->PurgeMessages(userId, channelId, purgeAfter);
    } else if (type == "created_room" || type == "deleted_room") {
      ChatRoomInfo roomInfo;
      const auto& jRoomInfo = jData["room"];
      if (!ttv::json::ObjectSchema<json::description::PubSubChatRoomInfo>::Parse(jRoomInfo, roomInfo)) {
        return;
      }

      ChannelId ownerId;
      const auto& jChannelId = jRoomInfo["owner_id"];
      if (!ParseChannelId(jChannelId, ownerId)) {
        return;
      }

      if (mListener != nullptr) {
        if (type == "created_room") {
          mListener->RoomCreated(ownerId, std::move(roomInfo));
        } else {
          mListener->RoomDeleted(ownerId, std::move(roomInfo));
        }
      }
    } else {
      Log(MessageLevel::Error, "Unrecognized pub-sub message type (%s), dropping", type.c_str());
      return;
    }
  }
}
