/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/chatapi.h"

#include "twitchsdk/chat/internal/bitsconfigrepository.h"
#include "twitchsdk/chat/internal/channel/bitsstatus.h"
#include "twitchsdk/chat/internal/channel/followersstatus.h"
#include "twitchsdk/chat/internal/channel/followingstatus.h"
#include "twitchsdk/chat/internal/channel/subscribersstatus.h"
#include "twitchsdk/chat/internal/channelchatroommanager.h"
#include "twitchsdk/chat/internal/chatapitaskhost.h"
#include "twitchsdk/chat/internal/chatchannelproperties.h"
#include "twitchsdk/chat/internal/chatchannelset.h"
#include "twitchsdk/chat/internal/chatcommentmanager.h"
#include "twitchsdk/chat/internal/chatraid.h"
#include "twitchsdk/chat/internal/chatroom.h"
#include "twitchsdk/chat/internal/chatroomnotifications.h"
#include "twitchsdk/chat/internal/chaturlgenerator.h"
#include "twitchsdk/chat/internal/chatuserbadges.h"
#include "twitchsdk/chat/internal/chatuserblocklist.h"
#include "twitchsdk/chat/internal/chatuserthreads.h"
#include "twitchsdk/chat/internal/defaultchatobjectfactory.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/chat/internal/multiviewnotifications.h"
#include "twitchsdk/chat/internal/squadnotifications.h"
#include "twitchsdk/chat/internal/subscriptionsnotifications.h"
#include "twitchsdk/chat/internal/task/chatbanusertask.h"
#include "twitchsdk/chat/internal/task/chatupdatecolortask.h"
#include "twitchsdk/chat/internal/useremoticonsets.h"
#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/settingrepository.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/types/memorytypes.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

namespace {
static const char* kModuleName = "ttv::chat::ChatAPI";
const char* kRequiredScopes[] = {"chat_login", "user_blocks_read", "user_blocks_edit"};
}  // namespace

struct ttv::chat::ChatAPIInternalData {
  std::unique_ptr<IMutex> mutex;  // The mutex that guards the instance containers here
  std::vector<std::shared_ptr<IChatChannel>> chatChannelInstances;
  std::vector<std::shared_ptr<IChatChannelProperties>> chatChannelPropertiesInstances;
  std::vector<std::shared_ptr<IChatCommentManager>> chatCommentManagerInstances;
  std::vector<std::shared_ptr<IChatRaid>> chatRaidInstances;
  std::vector<std::shared_ptr<IChannelChatRoomManager>> channelChatRoomManagerInstances;
  std::vector<std::shared_ptr<IChatRoom>> chatRoomInstances;
  std::vector<std::shared_ptr<IChatRoomNotifications>> chatRoomNotificationsInstances;
  std::vector<std::shared_ptr<BitsStatus>> bitsStatusInstances;
  std::vector<std::shared_ptr<FollowersStatus>> followersStatusInstances;
  std::vector<std::shared_ptr<FollowingStatus>> followingStatusInstances;
  std::vector<std::shared_ptr<SubscribersStatus>> subscribersStatusInstances;
  std::vector<std::shared_ptr<ISubscriptionsNotifications>> subscriptionsNotificationsInstances;
  std::vector<std::shared_ptr<ISquadNotifications>> squadNotificationsInstances;
  std::vector<std::shared_ptr<IMultiviewNotifications>> multiviewNotificationsInstances;
  std::shared_ptr<UserEmoticonSetsListenerProxy> userEmoticonSetsListenerProxy;
};

std::string ttv::chat::ChatAPI::CoreApiClient::GetClientName() {
  return kModuleName;
}

void ttv::chat::ChatAPI::CoreApiClient::GetRequiredOAuthScopes(std::vector<std::string>& scopes) {
  scopes.insert(scopes.end(), kRequiredScopes, kRequiredScopes + sizeof(kRequiredScopes) / sizeof(kRequiredScopes[0]));
}

ttv::chat::ChatAPI::ChatAPI() {
  mEnabledFeatures = FeatureFlags::All();
  mTokenizationOptions = TokenizationOptions::All();

  // Register the error lookup function for the module
  ttv::RegisterErrorToStringFunction(&ttv::chat::ChatErrorToString);
  ttv::RegisterErrorCodeValueFunction(&ttv::chat::GetChatErrorCodeValues);

  SetChatObjectFactory(nullptr);
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateChatChannel(UserId userId, ChannelId channelId,
  const std::shared_ptr<IChatChannelListener>& listener, std::shared_ptr<IChatChannel>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return TTV_EC_NEED_TO_LOGIN;
    }
  } else {
    user = mUserRepository->GetAnonymousUser();
  }

  std::shared_ptr<ChatChannelWrapper> wrapper = std::make_shared<ChatChannelWrapper>(user, channelId, listener);
  auto channelSet = wrapper->GetChatChannelSet();

  channelSet->SetChatObjectFactory(mChatObjectFactory);
  channelSet->SetTaskRunner(mTaskRunner);
  channelSet->SetMainEventScheduler(mMainEventScheduler);
  channelSet->SetTokenizationOptions(mTokenizationOptions);
  channelSet->SetChannelRepository(mChannelRepository);
  channelSet->SetBitsConfigRepository(mBitsConfigRepository);
  channelSet->SetSettingRepository(mSettingRepository);

  TTV_ErrorCode ec = channelSet->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    wrapper->SetDisposer([wrapper, data = mInternalData]() { ChatAPI::DisposeChatChannel(wrapper, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->chatChannelInstances.push_back(wrapper);
    }

    user->GetComponentContainer()->AddComponent(channelSet);

    result.reset(wrapper.get(), [wrapper](IChatChannel* /*p*/) { wrapper->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeChatChannel(
  const std::shared_ptr<IChatChannel>& chatChannel, const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(chatChannel, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter =
    std::find(internalData->chatChannelInstances.begin(), internalData->chatChannelInstances.end(), chatChannel);
  if (iter != internalData->chatChannelInstances.end()) {
    // Get the concrete object
    auto wrapper = std::static_pointer_cast<ChatChannelWrapper>(chatChannel);
    auto chatChannelSet = wrapper->GetChatChannelSet();
    std::shared_ptr<User> user = chatChannelSet->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(chatChannelSet)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(chatChannelSet);
    }

    internalData->chatChannelInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::GetEmoticonUrl(const std::string& emoticonId, float scale, std::string& result) {
  return ttv::chat::GetEmoticonUrl(emoticonId, scale, result);
}

ttv::Result<std::shared_ptr<ttv::chat::IChatCommentManager>> ttv::chat::ChatAPI::CreateChatCommentManager(
  UserId userId, const std::string& vodId, const std::shared_ptr<IChatCommentListener>& listener) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, MakeErrorResult(TTV_EC_NOT_INITIALIZED));
  TTV_RETURN_ON_EMPTY_STDSTRING(vodId, MakeErrorResult(TTV_EC_INVALID_ARG));
  TTV_RETURN_ON_NULL(listener, MakeErrorResult(TTV_EC_INVALID_ARG));

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return MakeErrorResult(TTV_EC_NEED_TO_LOGIN);
    }
  } else {
    user = mUserRepository->GetAnonymousUser();
  }

  std::shared_ptr<ChatCommentManager> chatCommentManager = std::make_shared<ChatCommentManager>(user, vodId);

  chatCommentManager->SetTaskRunner(mTaskRunner);
  chatCommentManager->SetTokenizationOptions(mTokenizationOptions);
  chatCommentManager->SetChannelRepository(mChannelRepository);
  chatCommentManager->SetBitsConfigRepository(mBitsConfigRepository);
  chatCommentManager->SetListener(listener);

  TTV_ErrorCode ec = chatCommentManager->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    chatCommentManager->SetDisposer(
      [chatCommentManager, data = mInternalData, componentContainer = GetComponentContainer()]() {
        DisposeChatCommentManager(chatCommentManager, data, componentContainer);
      });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->chatCommentManagerInstances.push_back(chatCommentManager);
    }

    GetComponentContainer()->AddComponent(chatCommentManager);

    std::shared_ptr<IChatCommentManager> result;
    result.reset(
      chatCommentManager.get(), [chatCommentManager](IChatCommentManager* /*p*/) { chatCommentManager->Dispose(); });

    return MakeSuccessResult(result);
  } else {
    return MakeErrorResult(ec);
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeChatCommentManager(
  const std::shared_ptr<IChatCommentManager>& chatCommentManager,
  const std::shared_ptr<ChatAPIInternalData>& internalData,
  const std::shared_ptr<ComponentContainer>& componentContainer) {
  TTV_RETURN_ON_NULL(chatCommentManager, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->chatCommentManagerInstances.begin(),
    internalData->chatCommentManagerInstances.end(), chatCommentManager);
  if (iter != internalData->chatCommentManagerInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<ChatCommentManager>(chatCommentManager);

    // Make sure this component is valid and hasn't already been disposed
    if (!componentContainer->ContainsComponent(obj)) {
      return TTV_EC_INVALID_ARG;
    }

    // Dispose of it
    componentContainer->DisposeComponent(obj);
    internalData->chatCommentManagerInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchChannelVodCommentSettings(
  UserId userId, ChannelId channelId, FetchChannelVodCommentSettingsCallback&& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mTaskHost->FetchVodCommentSettings(
    userId, channelId, [callback = std::move(callback)](TTV_ErrorCode ec, ChannelVodCommentSettings&& settings) {
      if (callback != nullptr) {
        callback(ec, std::move(settings));
      }
    });
}

TTV_ErrorCode ttv::chat::ChatAPI::SetChannelVodFollowersOnlyDuration(UserId userId, ChannelId channelId,
  uint32_t followersOnlyDurationSeconds, SetChannelVodCommentSettingsCallback&& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mTaskHost->SetVodCommentFollowersOnlyDurationSeconds(
    userId, channelId, followersOnlyDurationSeconds, [callback = std::move(callback)](TTV_ErrorCode ec) {
      if (callback != nullptr) {
        callback(ec);
      }
    });
}

TTV_ErrorCode ttv::chat::ChatAPI::SetChannelVodPublishingMode(
  UserId userId, ChannelId channelId, CommentPublishingMode mode, SetChannelVodCommentSettingsCallback&& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mTaskHost->SetVodCommentPublishingMode(
    userId, channelId, mode, [callback = std::move(callback)](TTV_ErrorCode ec) {
      if (callback != nullptr) {
        callback(ec);
      }
    });
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateChannelChatRoomManager(UserId userId, ChannelId channelId,
  const std::shared_ptr<IChannelChatRoomManagerListener>& listener, std::shared_ptr<IChannelChatRoomManager>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);
  }
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto channelChatRoomManager = std::make_shared<ChannelChatRoomManager>(user, channelId);

  channelChatRoomManager->SetTaskRunner(mTaskRunner);
  channelChatRoomManager->SetListener(listener);

  TTV_ErrorCode ec = channelChatRoomManager->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    channelChatRoomManager->SetDisposer([channelChatRoomManager, data = mInternalData]() {
      DisposeChannelChatRoomManager(channelChatRoomManager, data);
    });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->channelChatRoomManagerInstances.push_back(channelChatRoomManager);
    }

    user->GetComponentContainer()->AddComponent(channelChatRoomManager);

    result.reset(channelChatRoomManager.get(),
      [channelChatRoomManager](IChannelChatRoomManager* /*p*/) { channelChatRoomManager->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeChannelChatRoomManager(
  const std::shared_ptr<IChannelChatRoomManager>& channelChatRoomManager,
  const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(channelChatRoomManager, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->channelChatRoomManagerInstances.begin(),
    internalData->channelChatRoomManagerInstances.end(), channelChatRoomManager);
  if (iter != internalData->channelChatRoomManagerInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<ChannelChatRoomManager>(channelChatRoomManager);

    std::shared_ptr<User> user = obj->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(obj)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(obj);
    }

    internalData->channelChatRoomManagerInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateChatRoom(UserId userId, const std::string& roomId, ChannelId channelId,
  const std::shared_ptr<IChatRoomListener>& listener, std::shared_ptr<IChatRoom>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(roomId, "", TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);
  }
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto room = std::make_shared<ChatRoom>(user, roomId, channelId);

  room->SetTaskRunner(mTaskRunner);
  room->SetListener(listener);
  room->SetTokenizationOptions(mTokenizationOptions);

  TTV_ErrorCode ec = room->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    room->SetDisposer([room, data = mInternalData]() { DisposeChatRoom(room, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->chatRoomInstances.push_back(room);
    }

    user->GetComponentContainer()->AddComponent(room);

    result.reset(room.get(), [room](IChatRoom* /*p*/) { room->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeChatRoom(
  const std::shared_ptr<IChatRoom>& chatRoom, const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(chatRoom, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->chatRoomInstances.begin(), internalData->chatRoomInstances.end(), chatRoom);
  if (iter != internalData->chatRoomInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<ChatRoom>(chatRoom);

    std::shared_ptr<User> user = obj->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(obj)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(obj);
    }

    internalData->chatRoomInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateChatRoomNotifications(UserId userId,
  const std::shared_ptr<IChatRoomNotificationsListener>& listener, std::shared_ptr<IChatRoomNotifications>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);
  }
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto roomNotifications = std::make_shared<ChatRoomNotifications>(user);

  roomNotifications->SetTaskRunner(mTaskRunner);
  roomNotifications->SetListener(listener);

  TTV_ErrorCode ec = roomNotifications->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    roomNotifications->SetDisposer(
      [roomNotifications, data = mInternalData]() { DisposeChatRoomNotifications(roomNotifications, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->chatRoomNotificationsInstances.push_back(roomNotifications);
    }

    user->GetComponentContainer()->AddComponent(roomNotifications);

    result.reset(
      roomNotifications.get(), [roomNotifications](IChatRoomNotifications* /*p*/) { roomNotifications->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeChatRoomNotifications(
  const std::shared_ptr<IChatRoomNotifications>& chatRoomNotifications,
  const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(chatRoomNotifications, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->chatRoomNotificationsInstances.begin(),
    internalData->chatRoomNotificationsInstances.end(), chatRoomNotifications);
  if (iter != internalData->chatRoomNotificationsInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<ChatRoomNotifications>(chatRoomNotifications);

    std::shared_ptr<User> user = obj->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(obj)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(obj);
    }

    internalData->chatRoomNotificationsInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateChatRaid(UserId userId, ChannelId channelId,
  const std::shared_ptr<IChatRaidListener>& listener, std::shared_ptr<IChatRaid>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);
  }
  if (user == nullptr) {
    user = mUserRepository->GetAnonymousUser();
  }

  auto raid = std::make_shared<ChatRaid>(user, channelId);

  raid->SetTaskRunner(mTaskRunner);
  raid->SetListener(listener);

  TTV_ErrorCode ec = raid->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    raid->SetDisposer([raid, data = mInternalData]() { DisposeChatRaid(raid, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->chatRaidInstances.push_back(raid);
    }

    user->GetComponentContainer()->AddComponent(raid);

    result.reset(raid.get(), [raid](IChatRaid* /*p*/) { raid->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeChatRaid(
  const std::shared_ptr<IChatRaid>& chatRaid, const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(chatRaid, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->chatRaidInstances.begin(), internalData->chatRaidInstances.end(), chatRaid);
  if (iter != internalData->chatRaidInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<ChatRaid>(chatRaid);

    std::shared_ptr<User> user = obj->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(obj)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(obj);
    }

    internalData->chatRaidInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateBitsStatus(
  UserId userId, const std::shared_ptr<IBitsListener>& listener, std::shared_ptr<IBitsStatus>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return TTV_EC_NEED_TO_LOGIN;
    }
  }

  auto bitsStatus = std::make_shared<BitsStatus>(user);

  bitsStatus->SetListener(listener);
  bitsStatus->SetTokenizationOptions(mTokenizationOptions);
  bitsStatus->SetBitsConfigRepository(mBitsConfigRepository);

  TTV_ErrorCode ec = bitsStatus->Initialize();

  if (TTV_SUCCEEDED(ec)) {
    bitsStatus->SetDisposer([bitsStatus, data = mInternalData]() { DisposeBitsStatus(bitsStatus, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->bitsStatusInstances.push_back(bitsStatus);
    }

    user->GetComponentContainer()->AddComponent(bitsStatus);

    result.reset(bitsStatus.get(), [bitsStatus](IBitsStatus* /*p*/) { bitsStatus->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeBitsStatus(
  const std::shared_ptr<IBitsStatus>& bitsStatus, const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(bitsStatus, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->bitsStatusInstances.begin(), internalData->bitsStatusInstances.end(), bitsStatus);
  if (iter != internalData->bitsStatusInstances.end()) {
    std::shared_ptr<User> user = (*iter)->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(*iter)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(*iter);
    }

    internalData->bitsStatusInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateFollowersStatus(UserId userId, ChannelId channelId,
  const std::shared_ptr<IFollowersListener>& listener, std::shared_ptr<IFollowersStatus>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return TTV_EC_NEED_TO_LOGIN;
    }
  } else {
    user = mUserRepository->GetAnonymousUser();
  }

  auto followersStatus = std::make_shared<FollowersStatus>(user, channelId);

  followersStatus->SetListener(listener);
  TTV_ErrorCode ec = followersStatus->Initialize();

  if (TTV_SUCCEEDED(ec)) {
    followersStatus->SetDisposer(
      [followersStatus, data = mInternalData]() { DisposeFollowersStatus(followersStatus, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->followersStatusInstances.push_back(followersStatus);
    }

    user->GetComponentContainer()->AddComponent(followersStatus);

    result.reset(followersStatus.get(), [followersStatus](IFollowersStatus* /*p*/) { followersStatus->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeFollowersStatus(
  const std::shared_ptr<IFollowersStatus>& followersStatus, const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(followersStatus, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(
    internalData->followersStatusInstances.begin(), internalData->followersStatusInstances.end(), followersStatus);
  if (iter != internalData->followersStatusInstances.end()) {
    std::shared_ptr<User> user = (*iter)->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(*iter)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(*iter);
    }

    internalData->followersStatusInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateFollowingStatus(
  UserId userId, const std::shared_ptr<IFollowingListener>& listener, std::shared_ptr<IFollowingStatus>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId == 0) {
    return TTV_EC_INVALID_ARG;
  }

  user = mUserRepository->GetUser(userId);

  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto followingStatus = std::make_shared<FollowingStatus>(user);

  followingStatus->SetListener(listener);
  TTV_ErrorCode ec = followingStatus->Initialize();

  if (TTV_SUCCEEDED(ec)) {
    followingStatus->SetDisposer(
      [followingStatus, data = mInternalData]() { DisposeFollowingStatus(followingStatus, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->followingStatusInstances.push_back(followingStatus);
    }

    user->GetComponentContainer()->AddComponent(followingStatus);

    result.reset(followingStatus.get(), [followingStatus](IFollowingStatus* /*p*/) { followingStatus->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeFollowingStatus(
  const std::shared_ptr<IFollowingStatus>& followingStatus, const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(followingStatus, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(
    internalData->followingStatusInstances.begin(), internalData->followingStatusInstances.end(), followingStatus);
  if (iter != internalData->followingStatusInstances.end()) {
    std::shared_ptr<User> user = (*iter)->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(*iter)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(*iter);
    }

    internalData->followingStatusInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateSubscribersStatus(
  UserId userId, const std::shared_ptr<ISubscribersListener>& listener, std::shared_ptr<ISubscribersStatus>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return TTV_EC_NEED_TO_LOGIN;
    }
  }

  auto subscribersStatus = std::make_shared<SubscribersStatus>(user);

  subscribersStatus->SetListener(listener);
  subscribersStatus->SetTokenizationOptions(mTokenizationOptions);

  TTV_ErrorCode ec = subscribersStatus->Initialize();

  if (TTV_SUCCEEDED(ec)) {
    subscribersStatus->SetDisposer(
      [subscribersStatus, data = mInternalData]() { DisposeSubscribersStatus(subscribersStatus, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->subscribersStatusInstances.push_back(subscribersStatus);
    }

    user->GetComponentContainer()->AddComponent(subscribersStatus);

    result.reset(
      subscribersStatus.get(), [subscribersStatus](ISubscribersStatus* /*p*/) { subscribersStatus->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeSubscribersStatus(const std::shared_ptr<ISubscribersStatus>& subscribersStatus,
  const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(subscribersStatus, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->subscribersStatusInstances.begin(),
    internalData->subscribersStatusInstances.end(), subscribersStatus);
  if (iter != internalData->subscribersStatusInstances.end()) {
    std::shared_ptr<User> user = (*iter)->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(*iter)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(*iter);
    }

    internalData->subscribersStatusInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

ttv::Result<std::shared_ptr<ttv::chat::ISubscriptionsNotifications>>
ttv::chat::ChatAPI::CreateSubscriptionsNotifications(
  UserId userId, const std::shared_ptr<ISubscriptionsNotificationsListener>& listener) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, MakeErrorResult(TTV_EC_NOT_INITIALIZED));
  TTV_RETURN_ON_SAME(userId, 0, MakeErrorResult(TTV_EC_INVALID_ARG));
  TTV_RETURN_ON_NULL(listener, MakeErrorResult(TTV_EC_INVALID_ARG));

  std::shared_ptr<User> user;
  user = mUserRepository->GetUser(userId);

  if (user == nullptr) {
    return MakeErrorResult(TTV_EC_NEED_TO_LOGIN);
  }

  auto subscriptionsNotifications = std::make_shared<SubscriptionsNotifications>(user);

  subscriptionsNotifications->SetTaskRunner(mTaskRunner);
  subscriptionsNotifications->SetListener(listener);

  TTV_ErrorCode ec = subscriptionsNotifications->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    subscriptionsNotifications->SetDisposer([subscriptionsNotifications, data = mInternalData]() {
      DisposeSubscriptionsNotifications(subscriptionsNotifications, data);
    });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->subscriptionsNotificationsInstances.push_back(subscriptionsNotifications);
    }

    user->GetComponentContainer()->AddComponent(subscriptionsNotifications);

    std::shared_ptr<ISubscriptionsNotifications> result;
    result.reset(subscriptionsNotifications.get(),
      [subscriptionsNotifications](ISubscriptionsNotifications* /*p*/) { subscriptionsNotifications->Dispose(); });

    return MakeSuccessResult(result);
  } else {
    return MakeErrorResult(ec);
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeSubscriptionsNotifications(
  const std::shared_ptr<ISubscriptionsNotifications>& subscriptionsNotifications,
  const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(subscriptionsNotifications, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->subscriptionsNotificationsInstances.begin(),
    internalData->subscriptionsNotificationsInstances.end(), subscriptionsNotifications);
  if (iter != internalData->subscriptionsNotificationsInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<SubscriptionsNotifications>(subscriptionsNotifications);

    std::shared_ptr<User> user = obj->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(obj)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(obj);
    }

    internalData->subscriptionsNotificationsInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

ttv::Result<std::shared_ptr<ttv::chat::ISquadNotifications>> ttv::chat::ChatAPI::CreateSquadNotifications(
  UserId userId, const std::string& squadId, const std::shared_ptr<ISquadNotificationsListener>& listener) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, MakeErrorResult(TTV_EC_NOT_INITIALIZED));
  TTV_RETURN_ON_NULL(listener, MakeErrorResult(TTV_EC_INVALID_ARG));

  std::shared_ptr<User> user;
  user = mUserRepository->GetUser(userId);

  if (user == nullptr) {
    return MakeErrorResult(TTV_EC_NEED_TO_LOGIN);
  }

  auto squadNotifications = std::make_shared<SquadNotifications>(user, squadId);

  squadNotifications->SetTaskRunner(mTaskRunner);
  squadNotifications->SetListener(listener);

  TTV_ErrorCode ec = squadNotifications->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    squadNotifications->SetDisposer(
      [squadNotifications, data = mInternalData]() { DisposeSquadNotifications(squadNotifications, data); });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->squadNotificationsInstances.push_back(squadNotifications);
    }

    user->GetComponentContainer()->AddComponent(squadNotifications);

    std::shared_ptr<ISquadNotifications> result;
    result.reset(
      squadNotifications.get(), [squadNotifications](ISquadNotifications* /*p*/) { squadNotifications->Dispose(); });

    return MakeSuccessResult(result);
  } else {
    return MakeErrorResult(ec);
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeSquadNotifications(
  const std::shared_ptr<ISquadNotifications>& squadNotifications,
  const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(squadNotifications, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->squadNotificationsInstances.begin(),
    internalData->squadNotificationsInstances.end(), squadNotifications);
  if (iter != internalData->squadNotificationsInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<SquadNotifications>(squadNotifications);

    std::shared_ptr<User> user = obj->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(obj)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(obj);
    }

    internalData->squadNotificationsInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

ttv::Result<std::shared_ptr<ttv::chat::IMultiviewNotifications>> ttv::chat::ChatAPI::CreateMultiviewNotifications(
  UserId userId, ChannelId channelId, const std::shared_ptr<IMultiviewNotificationsListener>& listener) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, MakeErrorResult(TTV_EC_NOT_INITIALIZED));
  TTV_RETURN_ON_SAME(channelId, 0, MakeErrorResult(TTV_EC_NOT_INITIALIZED));
  TTV_RETURN_ON_NULL(listener, MakeErrorResult(TTV_EC_INVALID_ARG));

  std::shared_ptr<User> user;
  user = mUserRepository->GetUser(userId);

  if (user == nullptr) {
    return MakeErrorResult(TTV_EC_NEED_TO_LOGIN);
  }

  auto multiviewNotifications = std::make_shared<MultiviewNotifications>(user, channelId);

  multiviewNotifications->SetTaskRunner(mTaskRunner);
  multiviewNotifications->SetListener(listener);

  TTV_ErrorCode ec = multiviewNotifications->Initialize();

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    multiviewNotifications->SetDisposer([multiviewNotifications, data = mInternalData]() {
      DisposeMultiviewNotifications(multiviewNotifications, data);
    });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->multiviewNotificationsInstances.push_back(multiviewNotifications);
    }

    user->GetComponentContainer()->AddComponent(multiviewNotifications);

    std::shared_ptr<IMultiviewNotifications> result;
    result.reset(multiviewNotifications.get(),
      [multiviewNotifications](IMultiviewNotifications* /*p*/) { multiviewNotifications->Dispose(); });

    return MakeSuccessResult(result);
  } else {
    return MakeErrorResult(ec);
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeMultiviewNotifications(
  const std::shared_ptr<IMultiviewNotifications>& multiviewNotifications,
  const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(multiviewNotifications, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->multiviewNotificationsInstances.begin(),
    internalData->multiviewNotificationsInstances.end(), multiviewNotifications);
  if (iter != internalData->multiviewNotificationsInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<MultiviewNotifications>(multiviewNotifications);

    std::shared_ptr<User> user = obj->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(obj)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(obj);
    }

    internalData->multiviewNotificationsInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::CreateChatChannelProperties(UserId userId, ChannelId channelId,
  const std::shared_ptr<IChatChannelPropertyListener>& listener, std::shared_ptr<IChatChannelProperties>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return TTV_EC_NEED_TO_LOGIN;
    }
  } else {
    user = mUserRepository->GetAnonymousUser();
  }

  auto chatChannelProperties = std::make_shared<ChatChannelProperties>(user, channelId);
  chatChannelProperties->SetTaskRunner(mTaskRunner);
  chatChannelProperties->SetListener(listener);

  TTV_ErrorCode ec = chatChannelProperties->Initialize();

  if (TTV_SUCCEEDED(ec)) {
    chatChannelProperties->SetDisposer([chatChannelProperties, data = mInternalData]() {
      ChatAPI::DisposeChatChannelProperties(chatChannelProperties, data);
    });

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->chatChannelPropertiesInstances.push_back(chatChannelProperties);
    }

    user->GetComponentContainer()->AddComponent(chatChannelProperties);

    result.reset(chatChannelProperties.get(),
      [chatChannelProperties](IChatChannelProperties* /*p*/) { chatChannelProperties->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::DisposeChatChannelProperties(
  const std::shared_ptr<IChatChannelProperties>& chatChannelProperties,
  const std::shared_ptr<ChatAPIInternalData>& internalData) {
  TTV_RETURN_ON_NULL(chatChannelProperties, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find(internalData->chatChannelPropertiesInstances.begin(),
    internalData->chatChannelPropertiesInstances.end(), chatChannelProperties);
  if (iter != internalData->chatChannelPropertiesInstances.end()) {
    // Get the concrete object
    auto obj = std::static_pointer_cast<ChatChannelProperties>(chatChannelProperties);

    std::shared_ptr<User> user = obj->GetUser();
    if (user != nullptr) {
      auto container = user->GetComponentContainer();
      if (container == nullptr) {
        return TTV_EC_SHUTTING_DOWN;
      }

      // Make sure this component is valid and hasn't already been disposed
      if (!container->ContainsComponent(obj)) {
        return TTV_EC_INVALID_ARG;
      }

      // Dispose of it
      container->DisposeComponent(obj);
    }

    internalData->chatChannelPropertiesInstances.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatAPI::SetChatObjectFactory(const std::shared_ptr<IChatObjectFactory>& factory) {
  // Use the default if the client doesn't want a custom one
  if (factory == nullptr) {
    mChatObjectFactory = std::static_pointer_cast<IChatObjectFactory>(std::make_shared<DefaultChatObjectFactory>());
  } else {
    mChatObjectFactory = factory;
  }

  if (mUserRepository != nullptr) {
    auto anonymousChannelSet =
      mUserRepository->GetAnonymousUser()->GetComponentContainer()->GetComponent<ChatChannelSet>();
    if (anonymousChannelSet != nullptr) {
      anonymousChannelSet->SetChatObjectFactory(factory);
    }

    std::vector<std::shared_ptr<User>> users;
    mUserRepository->GetUsers(users);
    for (auto user : users) {
      auto channelSet = user->GetComponentContainer()->GetComponent<ChatChannelSet>();
      if (channelSet != nullptr) {
        channelSet->SetChatObjectFactory(factory);
      }
    }
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::GetBlockListForUser(UserId userId, std::shared_ptr<ChatUserBlockList>& blockList) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_INVALID_USERID;
  }

  blockList = user->GetComponentContainer()->GetComponent<ChatUserBlockList>();
  if (blockList != nullptr) {
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_SHUT_DOWN;
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::GetUserThreadsForUser(UserId userId, std::shared_ptr<ChatUserThreads>& userThreads) {
  if (!mEnabledFeatures.conversations) {
    return TTV_EC_FEATURE_DISABLED;
  }

  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_INVALID_USERID;
  }

  userThreads = user->GetComponentContainer()->GetComponent<ChatUserThreads>();
  if (userThreads != nullptr) {
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_SHUT_DOWN;
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::SetEnabledFeatures(const FeatureFlags& features) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Uninitialized, TTV_EC_ALREADY_INITIALIZED);

  mEnabledFeatures = features;

  return TTV_EC_SUCCESS;
}

std::string ttv::chat::ChatAPI::GetModuleName() const {
  return kModuleName;
}

TTV_ErrorCode ttv::chat::ChatAPI::SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi) {
  if (mState != State::Uninitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  TTV_ASSERT(coreApi != nullptr);

  mCoreApi = coreApi;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::SetListener(const std::shared_ptr<IChatAPIListener>& listener) {
  if (mState != State::Uninitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  mListeners.ClearListeners();

  if (listener != nullptr) {
    mListeners.AddListener(listener);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::SetTokenizationOptions(const TokenizationOptions& options) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Uninitialized, TTV_EC_ALREADY_INITIALIZED);

  mTokenizationOptions = options;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatAPI::Initialize(const InitializeCallback& callback) {
  TTV_RETURN_ON_NULL(mCoreApi, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec = ModuleBase::Initialize(callback);

  if (TTV_SUCCEEDED(ec)) {
    mState = State::Initializing;

    mCoreApiClient = std::make_shared<CoreApiClient>(this);

    mMainEventScheduler = mCoreApi->GetMainEventScheduler();
    mTaskRunner = std::make_shared<TaskRunner>("ChatAPI");

    mChannelRepository = mCoreApi->GetChannelRepository();
    mUserRepository = mCoreApi->GetUserRepository();
    mSettingRepository = mCoreApi->GetSettingRepository();
    mUserBlocksTaskRunner = std::make_shared<TaskRunner>("ChatAPI-UserBlocks");

    mInternalData = std::make_shared<ChatAPIInternalData>();
    TTV_ErrorCode ret = CreateMutex(mInternalData->mutex, "ChatAPIInternalData");
    ASSERT_ON_ERROR(ret);

    std::string language;
    mCoreApi->GetLocalLanguage(language);

    mTaskHost = std::make_shared<ChatAPITaskHost>();
    mTaskHost->SetTaskRunner(mTaskRunner);
    mTaskHost->SetUserRepository(mUserRepository);
    mTaskHost->Initialize();

    mBitsConfigRepository = std::make_shared<BitsConfigRepository>();
    mBitsConfigRepository->SetTaskRunner(mTaskRunner);
    mBitsConfigRepository->SetUserRepository(mUserRepository);
    mBitsConfigRepository->Initialize();

    auto anonymousUser = mUserRepository->GetAnonymousUser();

    // Configure the anonymous user
    auto channelSet = std::make_shared<ChatChannelSet>(anonymousUser);
    channelSet->SetChannelRepository(mChannelRepository);
    channelSet->SetBitsConfigRepository(mBitsConfigRepository);
    channelSet->SetChatObjectFactory(mChatObjectFactory);
    channelSet->SetTokenizationOptions(mTokenizationOptions);
    channelSet->SetTaskRunner(mTaskRunner);
    channelSet->SetMainEventScheduler(mMainEventScheduler);
    channelSet->SetSettingRepository(mSettingRepository);
    channelSet->Initialize();

    mUserRepository->GetAnonymousUser()->GetComponentContainer()->SetComponent(
      ChatChannelSet::GetComponentName(), channelSet);

    ec = mCoreApi->RegisterClient(mCoreApiClient);
  }

  if (TTV_SUCCEEDED(ec)) {
    RegisterInitializeCallback(callback);
    NotifyStateChange();
  } else {
    CompleteShutdown();
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::Shutdown(const ShutdownCallback& callback) {
  TTV_ErrorCode ec = ModuleBase::Shutdown(callback);

  if (TTV_SUCCEEDED(ec)) {
    // Cleanup data in users
    std::vector<std::shared_ptr<User>> users;
    mUserRepository->GetUsers(users);

    for (auto user : users) {
      CoreUserLoggedOut(user);
    }

    auto anonymousUser = mUserRepository->GetAnonymousUser();
    if (anonymousUser != nullptr) {
      auto container = anonymousUser->GetComponentContainer();
      if (container != nullptr) {
        container->DisposeComponent(ChatChannelSet::GetComponentName());
      }
    }

    // Dispose all outstanding public components
    if (mInternalData != nullptr) {
      while (!mInternalData->chatChannelInstances.empty()) {
        mInternalData->chatChannelInstances[0]->Dispose();
      }

      while (!mInternalData->chatCommentManagerInstances.empty()) {
        mInternalData->chatCommentManagerInstances[0]->Dispose();
      }

      while (!mInternalData->channelChatRoomManagerInstances.empty()) {
        mInternalData->channelChatRoomManagerInstances[0]->Dispose();
      }

      while (!mInternalData->chatRoomInstances.empty()) {
        mInternalData->chatRoomInstances[0]->Dispose();
      }

      while (!mInternalData->chatRoomNotificationsInstances.empty()) {
        mInternalData->chatRoomNotificationsInstances[0]->Dispose();
      }

      while (!mInternalData->chatRaidInstances.empty()) {
        mInternalData->chatRaidInstances[0]->Dispose();
      }

      while (!mInternalData->bitsStatusInstances.empty()) {
        mInternalData->bitsStatusInstances[0]->Dispose();
      }

      while (!mInternalData->followersStatusInstances.empty()) {
        mInternalData->followersStatusInstances[0]->Dispose();
      }

      while (!mInternalData->subscribersStatusInstances.empty()) {
        mInternalData->subscribersStatusInstances[0]->Dispose();
      }
    }

    RegisterShutdownCallback(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::SetUserThreadsListener(
  UserId userId, const std::shared_ptr<IChatUserThreadsListener>& listener) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatUserThreads> userThreads;
  TTV_ErrorCode ec = GetUserThreadsForUser(userId, userThreads);

  if (TTV_SUCCEEDED(ec)) {
    userThreads->SetListener(listener);
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::GetChannelSet(UserId userId, std::shared_ptr<ChatChannelSet>& channelSet) {
  if (userId != 0) {
    auto user = mUserRepository->GetUser(userId);

    if (user == nullptr) {
      return TTV_EC_INVALID_LOGIN;
    } else {
      channelSet = user->GetComponentContainer()->GetComponent<ChatChannelSet>();
    }
  } else {
    channelSet = mUserRepository->GetAnonymousUser()->GetComponentContainer()->GetComponent<ChatChannelSet>();
  }

  if (channelSet == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  } else {
    return TTV_EC_SUCCESS;
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::Connect(
  UserId userId, ChannelId channelId, const std::shared_ptr<IChatChannelListener>& listener) {
  auto start = GetSystemTimeMilliseconds();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatChannelSet> channelSet;
  TTV_ErrorCode ec = GetChannelSet(userId, channelSet);
  if (TTV_SUCCEEDED(ec)) {
    ec = channelSet->Connect(channelId, listener);
    channelSet->SetConnectTrackingStartTime(channelId, start);
  }
  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::Disconnect(UserId userId, ChannelId channelId) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatChannelSet> channelSet;
  TTV_ErrorCode ec = GetChannelSet(userId, channelSet);
  if (TTV_SUCCEEDED(ec)) {
    ec = channelSet->Disconnect(channelId);
  }
  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::SendChatMessage(UserId userId, ChannelId channelId, const std::string& message) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatChannelSet> channelSet;
  TTV_ErrorCode ec = GetChannelSet(userId, channelSet);
  if (TTV_SUCCEEDED(ec)) {
    ec = channelSet->SendChatMessage(channelId, message);
  }
  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchUserListForChannel(
  UserId userId, ChannelId channelId, const FetchUserListCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatChannelSet> channelSet;
  TTV_ErrorCode ec = GetChannelSet(userId, channelSet);
  if (TTV_SUCCEEDED(ec)) {
    ec = channelSet->FetchUserList(channelId, callback);
  }
  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::BlockUser(
  UserId userId, UserId blockUserId, const std::string& reason, bool whisper, const BlockChangeCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatUserBlockList> blockList;
  TTV_ErrorCode ec = GetBlockListForUser(userId, blockList);

  if (TTV_SUCCEEDED(ec)) {
    ec = blockList->BlockUser(blockUserId, reason, whisper, callback);
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::UnblockUser(UserId userId, UserId blockUserId, const BlockChangeCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatUserBlockList> blockList;
  TTV_ErrorCode ec = GetBlockListForUser(userId, blockList);

  if (TTV_SUCCEEDED(ec)) {
    ec = blockList->UnblockUser(blockUserId, callback);
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::GetUserBlocked(UserId userId, UserId blockUserId, bool& blocked) {
  blocked = false;

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatUserBlockList> blockList;
  TTV_ErrorCode ec = GetBlockListForUser(userId, blockList);

  if (TTV_SUCCEEDED(ec)) {
    blocked = blockList->IsUserBlocked(blockUserId);
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchBlockedUsers(UserId userId, const FetchBlockedUsersCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatUserBlockList> blockList;
  TTV_ErrorCode ec = GetBlockListForUser(userId, blockList);

  if (TTV_SUCCEEDED(ec)) {
    ec = blockList->FetchBlockedUsers(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatAPI::Update() {
  if (mState == State::Uninitialized) {
    return TTV_EC_SUCCESS;
  }

  ModuleBase::Update();

  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }

  if (mTaskHost != nullptr) {
    mTaskHost->Update();
  }

  if (mBitsConfigRepository != nullptr) {
    mBitsConfigRepository->Update();
  }

  if (mUserBlocksTaskRunner != nullptr) {
    mUserBlocksTaskRunner->PollTasks();
  }

  switch (mState) {
    case State::Initializing: {
      mState = State::Initialized;
      NotifyStateChange();

      break;
    }
    case State::ShuttingDown: {
      if (CheckShutdown()) {
        CompleteShutdown();

        NotifyStateChange();
      }

      break;
    }
    default: { break; }
  }

  return TTV_EC_SUCCESS;
}

bool ttv::chat::ChatAPI::CheckShutdown() {
  if (!ModuleBase::CheckShutdown()) {
    return false;
  }

  if (mTaskHost != nullptr) {
    if (mTaskHost->GetState() == ChatAPITaskHost::State::Initialized) {
      mTaskHost->Shutdown();
    }

    if (mTaskHost->GetState() != ChatAPITaskHost::State::Uninitialized) {
      return false;
    }
  }

  if (mBitsConfigRepository != nullptr) {
    if (mBitsConfigRepository->GetState() == BitsConfigRepository::State::Initialized) {
      mBitsConfigRepository->Shutdown();
    }

    if (mBitsConfigRepository->GetState() != BitsConfigRepository::State::Uninitialized) {
      return false;
    }
  }

  if (!mTaskRunner->IsShutdown()) {
    mTaskRunner->Shutdown();
    return false;
  }

  if (!mUserBlocksTaskRunner->IsShutdown()) {
    mUserBlocksTaskRunner->Shutdown();
    return false;
  }

  return true;
}

void ttv::chat::ChatAPI::CompleteShutdown() {
  if (mCoreApiClient != nullptr) {
    if (mCoreApi != nullptr) {
      mCoreApi->UnregisterClient(mCoreApiClient);
    }

    mCoreApiClient.reset();
  }

  if (mTaskRunner != nullptr) {
    mTaskRunner->CompleteShutdown();
    mTaskRunner.reset();
  }

  if (mUserBlocksTaskRunner != nullptr) {
    mUserBlocksTaskRunner->CompleteShutdown();
    mUserBlocksTaskRunner.reset();
  }

  mCoreApi.reset();
  mUserRepository.reset();
  mTaskHost.reset();
  mBitsConfigRepository.reset();
  mInternalData.reset();

  ModuleBase::CompleteShutdown();
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchUserEmoticonSets(
  UserId userId, bool forceRefetch, const FetchUserEmoticonSetsCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(mTokenizationOptions.emoticons, false, TTV_EC_FEATURE_DISABLED);

  std::shared_ptr<User> user;
  if (userId != 0) {
    user = mUserRepository->GetUser(userId);
  }

  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto emoticonSets = user->GetComponentContainer()->GetComponent<UserEmoticonSets>();
  if (emoticonSets != nullptr) {
    return emoticonSets->FetchUserEmoticonSets(forceRefetch, callback);
  } else {
    return TTV_EC_FEATURE_DISABLED;
  }
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchGlobalBadges(const FetchBadgesCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mTaskHost->FetchGlobalBadges([callback](TTV_ErrorCode ec, BadgeSet&& badges) {
    if (callback != nullptr) {
      callback(ec, std::move(badges));
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchChannelBadges(ChannelId channelId, const FetchBadgesCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mTaskHost->FetchChannelBadges(channelId, [callback](TTV_ErrorCode ec, BadgeSet&& badges) {
    if (callback != nullptr) {
      callback(ec, std::move(badges));
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchGlobalBitsConfiguration(const FetchBitsConfigurationCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mBitsConfigRepository->FetchGlobalBitsConfiguration(
    [callback](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
      if (callback != nullptr) {
        callback(ec, config);
      }
    });
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchChannelBitsConfiguration(
  UserId userId, ChannelId channelId, const FetchBitsConfigurationCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  return mBitsConfigRepository->FetchChannelBitsConfiguration(
    userId, channelId, [callback](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
      if (callback != nullptr) {
        callback(ec, config);
      }
    });
}

void ttv::chat::ChatAPI::GetRequiredAuthScopes(std::vector<std::string>& scopes) {
  scopes.insert(scopes.end(), kRequiredScopes, kRequiredScopes + sizeof(kRequiredScopes) / sizeof(kRequiredScopes[0]));
}

TTV_ErrorCode ttv::chat::ChatAPI::SetMessageFlushInterval(uint64_t milliseconds) {
  std::vector<std::shared_ptr<User>> users;
  mUserRepository->GetUsers(users);
  for (const auto& user : users) {
    auto channelSet = user->GetComponentContainer()->GetComponent<ChatChannelSet>();
    if (channelSet != nullptr) {
      channelSet->SetMessageFlushInterval(milliseconds);
    }
  }

  return TTV_EC_SUCCESS;
}

uint64_t ttv::chat::ChatAPI::GetMessageFlushInterval() const {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, 0);

  auto channelSet = mUserRepository->GetAnonymousUser()->GetComponentContainer()->GetComponent<ChatChannelSet>();

  TTV_RETURN_ON_NULL(channelSet, 0);

  return channelSet->GetMessageFlushInterval();
}

std::shared_ptr<ttv::chat::ChatUserBlockList> ttv::chat::ChatAPI::CreateUserBlockList(
  const std::shared_ptr<ttv::User>& user) {
  std::shared_ptr<ChatUserBlockList> blockList = std::make_shared<ChatUserBlockList>(user);
  blockList->SetTaskRunner(mUserBlocksTaskRunner);
  blockList->Initialize();

  user->GetComponentContainer()->SetComponent(ChatUserBlockList::GetComponentName(), blockList);

  return blockList;
}

TTV_ErrorCode ttv::chat::ChatAPI::BanUser(UserId userId, ChannelId channelId, const std::string& bannedUserName,
  uint32_t durationSeconds, const BanUserCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_EMPTY_STDSTRING(bannedUserName, TTV_EC_INVALID_ARG);

  return mTaskHost->BanUser(
    userId, channelId, bannedUserName, durationSeconds, [callback](TTV_ErrorCode ec, BanUserError&& error) {
      if (callback != nullptr) {
        callback(ec, std::move(error));
      }
    });
}

TTV_ErrorCode ttv::chat::ChatAPI::UnbanUser(
  UserId userId, ChannelId channelId, const std::string& bannedUserName, const UnbanUserCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_EMPTY_STDSTRING(bannedUserName, TTV_EC_INVALID_ARG);

  return mTaskHost->UnbanUser(userId, channelId, bannedUserName, [callback](TTV_ErrorCode ec, UnbanUserError&& error) {
    if (callback != nullptr) {
      callback(ec, std::move(error));
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::ModUser(
  UserId userId, ChannelId channelId, const std::string& modUserName, const ModUserCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_EMPTY_STDSTRING(modUserName, TTV_EC_INVALID_ARG);

  return mTaskHost->ModUser(userId, channelId, modUserName, [callback](TTV_ErrorCode ec, ModUserError&& error) {
    if (callback != nullptr) {
      callback(ec, std::move(error));
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::UnmodUser(
  UserId userId, ChannelId channelId, const std::string& unmodUserName, const UnmodUserCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_EMPTY_STDSTRING(unmodUserName, TTV_EC_INVALID_ARG);

  return mTaskHost->UnmodUser(userId, channelId, unmodUserName, [callback](TTV_ErrorCode ec, UnmodUserError&& error) {
    if (callback != nullptr) {
      callback(ec, std::move(error));
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::GrantVIP(
  UserId userId, ChannelId channelId, const std::string& vipUserName, const GrantVIPCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_EMPTY_STDSTRING(vipUserName, TTV_EC_INVALID_ARG);

  return mTaskHost->GrantVIP(userId, channelId, vipUserName, [callback](TTV_ErrorCode ec, GrantVIPErrorCode error) {
    if (callback != nullptr) {
      callback(ec, error);
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::RevokeVIP(
  UserId userId, ChannelId channelId, const std::string& unvipUserName, const RevokeVIPCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_EMPTY_STDSTRING(unvipUserName, TTV_EC_INVALID_ARG);

  return mTaskHost->RevokeVIP(userId, channelId, unvipUserName, [callback](TTV_ErrorCode ec, RevokeVIPErrorCode error) {
    if (callback != nullptr) {
      callback(ec, error);
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchChannelVIPs(ChannelId channelId, const FetchChannelVIPsCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);

  return mTaskHost->FetchChannelVIPs(channelId, [callback](TTV_ErrorCode ec, std::vector<std::string>&& vipNames) {
    if (callback != nullptr) {
      callback(ec, std::move(vipNames));
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::UpdateUserColor(
  UserId userId, const std::string& color, const UpdateUserColorCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(userId, 0, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_EMPTY_STDSTRING(color, TTV_EC_INVALID_ARG);

  return mTaskHost->UpdateUserColor(userId, color, [callback](TTV_ErrorCode ec) {
    if (callback != nullptr) {
      callback(ec);
    }
  });
}

TTV_ErrorCode ttv::chat::ChatAPI::FetchChannelModerators(
  ChannelId channelId, const std::string& cursor, const FetchChannelModeratorsCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_ARG);

  return mTaskHost->FetchChannelModerators(
    channelId, cursor, [callback](TTV_ErrorCode ec, std::vector<std::string>&& modNames, std::string&& nextCursor) {
      if (callback != nullptr) {
        callback(ec, std::move(modNames), std::move(nextCursor));
      }
    });
}

void ttv::chat::ChatAPI::CoreUserLoggedIn(std::shared_ptr<User> user) {
  if (mState == State::Uninitialized || mState == State::ShuttingDown) {
    return;
  }

  // Update ChatUserBlockList
  std::shared_ptr<ChatUserBlockList> blockList = user->GetComponentContainer()->GetComponent<ChatUserBlockList>();
  if (blockList == nullptr) {
    blockList = CreateUserBlockList(user);
  }

  // Update ChatUserThreads
  if (mEnabledFeatures.conversations) {
    std::shared_ptr<ChatUserThreads> userThreads = user->GetComponentContainer()->GetComponent<ChatUserThreads>();
    if (userThreads == nullptr) {
      userThreads = std::make_shared<ChatUserThreads>(user);
      userThreads->SetUserRepository(mUserRepository);
      userThreads->SetTaskRunner(mTaskRunner);
      userThreads->SetTokenizationOptions(mTokenizationOptions);
      userThreads->SetSettingRepository(mSettingRepository);
      userThreads->SetBitsConfigRepository(mBitsConfigRepository);
      userThreads->Initialize();

      user->GetComponentContainer()->SetComponent(ChatUserThreads::GetComponentName(), userThreads);
    }

    std::shared_ptr<ChatChannelSet> channelSet = user->GetComponentContainer()->GetComponent<ChatChannelSet>();
    if (channelSet == nullptr) {
      std::string language;
      mCoreApi->GetLocalLanguage(language);

      channelSet = std::make_shared<ChatChannelSet>(user);
      channelSet->SetChannelRepository(mChannelRepository);
      channelSet->SetBitsConfigRepository(mBitsConfigRepository);
      channelSet->SetChatObjectFactory(mChatObjectFactory);
      channelSet->SetMainEventScheduler(mMainEventScheduler);
      channelSet->SetTokenizationOptions(mTokenizationOptions);
      channelSet->SetTaskRunner(mTaskRunner);
      channelSet->SetSettingRepository(mSettingRepository);
      channelSet->Initialize();

      user->GetComponentContainer()->SetComponent(ChatChannelSet::GetComponentName(), channelSet);
    }
  }
  // NOTE: There is no need to explicitly update ChatChannel instances since channels will grab the latest oauth token
  // when they try to connect

  // Update UserEmoticonSets
  if (mTokenizationOptions.emoticons) {
    std::shared_ptr<UserEmoticonSets> userEmoticonSets =
      user->GetComponentContainer()->GetComponent<UserEmoticonSets>();

    if (userEmoticonSets == nullptr) {
      if (mInternalData->userEmoticonSetsListenerProxy == nullptr) {
        mInternalData->userEmoticonSetsListenerProxy = std::make_shared<UserEmoticonSetsListenerProxy>();

        mInternalData->userEmoticonSetsListenerProxy->mUserEmoticonSetsChangedFunc =
          [this](UserId userId, const std::vector<EmoticonSet>& emoticonSets) {
            Invoke<IChatAPIListener>([userId, &emoticonSets](const std::shared_ptr<IChatAPIListener>& listener) {
              listener->ChatUserEmoticonSetsChanged(userId, emoticonSets);
            });
          };
      }

      userEmoticonSets = std::make_shared<UserEmoticonSets>(user, mTokenizationOptions);
      userEmoticonSets->SetTaskRunner(mTaskRunner);
      userEmoticonSets->SetListener(mInternalData->userEmoticonSetsListenerProxy);
      userEmoticonSets->Initialize();

      user->GetComponentContainer()->SetComponent(UserEmoticonSets::GetComponentName(), userEmoticonSets);
    }
  }

  // Update ChatUserBadges
  std::shared_ptr<ChatUserBadges> userBadges = user->GetComponentContainer()->GetComponent<ChatUserBadges>();
  if (userBadges == nullptr) {
    userBadges = std::make_shared<ChatUserBadges>(user);
    userBadges->Initialize();

    user->GetComponentContainer()->SetComponent(ChatUserBadges::GetComponentName(), userBadges);
  }
}

void ttv::chat::ChatAPI::CoreUserLoggedOut(std::shared_ptr<User> user) {
  // We don't need these components anymore
  auto container = user->GetComponentContainer();
  if (container != nullptr) {
    container->DisposeComponent(ChatUserBlockList::GetComponentName());
    container->DisposeComponent(ChatUserThreads::GetComponentName());
    container->DisposeComponent(ChatChannelSet::GetComponentName());
    container->DisposeComponent(UserEmoticonSets::GetComponentName());
    container->DisposeComponent(ChatUserBadges::GetComponentName());
  }
}

ttv::chat::ChatAPI::CoreApiClient::CoreApiClient(ChatAPI* api) : mApi(api) {}

void ttv::chat::ChatAPI::CoreApiClient::CoreUserLoggedIn(std::shared_ptr<User> user) {
  mApi->CoreUserLoggedIn(user);
}

void ttv::chat::ChatAPI::CoreApiClient::CoreUserLoggedOut(std::shared_ptr<User> user) {
  mApi->CoreUserLoggedOut(user);
}
