/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatapitaskhost.h"

#include "twitchsdk/chat/internal/graphql/generated/fetchchannelbadges.h"
#include "twitchsdk/chat/internal/graphql/generated/fetchchannelvipsqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/generated/fetchglobalbadges.h"
#include "twitchsdk/chat/internal/graphql/grantvipqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/revokevipqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/utilities/badges.h"
#include "twitchsdk/chat/internal/task/chatbanusertask.h"
#include "twitchsdk/chat/internal/task/chatfetchchannelmoderatorstask.h"
#include "twitchsdk/chat/internal/task/chatgetchannelvodcommentsettingstask.h"
#include "twitchsdk/chat/internal/task/chatmodusertask.h"
#include "twitchsdk/chat/internal/task/chatsetchannelvodcommentsettingstask.h"
#include "twitchsdk/chat/internal/task/chatunbanusertask.h"
#include "twitchsdk/chat/internal/task/chatupdatecolortask.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/graphqltask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

ttv::chat::ChatAPITaskHost::ChatAPITaskHost() : Component() {}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::BanUser(UserId userId, ChannelId channelId, const std::string& bannedUserName,
  uint32_t duration, BanUserCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatBanUserTask>(channelId, bannedUserName, oauthToken->GetToken(),
    [this, user, oauthToken, callback = std::move(callback)](
      ChatBanUserTask* source, TTV_ErrorCode ec, BanUserError&& error) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error));
      }
    });

  if (duration != 0) {
    task->SetTimeout(duration);
  }

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::UnbanUser(
  UserId userId, ChannelId channelId, const std::string& unbannedUserName, UnbanUserCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatUnbanUserTask>(channelId, unbannedUserName, oauthToken->GetToken(),
    [this, user, oauthToken, callback = std::move(callback)](
      ChatUnbanUserTask* source, TTV_ErrorCode ec, UnbanUserError&& error) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::ModUser(
  UserId userId, ChannelId channelId, const std::string& modUserName, ModUserCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatModUserTask>(channelId, modUserName, oauthToken->GetToken(),
    [this, user, oauthToken, callback = std::move(callback)](
      ChatModUserTask* source, TTV_ErrorCode ec, ModUserError&& error) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::UnmodUser(
  UserId userId, ChannelId channelId, const std::string& unmodUserName, UnmodUserCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatModUserTask>(channelId, unmodUserName, oauthToken->GetToken(),
    [this, user, oauthToken, callback = std::move(callback)](
      ChatModUserTask* source, TTV_ErrorCode ec, UnmodUserError&& error) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::GrantVIP(
  UserId userId, ChannelId channelId, const std::string& vipUserName, GrantVIPCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  graphql::GrantVIPQueryInfo::InputParams inputs;
  inputs.authToken = oauthToken->GetToken();
  inputs.input.channelID = std::to_string(channelId);
  inputs.input.granteeLogin = vipUserName;

  auto task = std::make_shared<GraphQLTask<graphql::GrantVIPQueryInfo>>(
    std::move(inputs), [this, taskCallback = std::move(callback)](GraphQLTask<graphql::GrantVIPQueryInfo>* source,
                         Result<graphql::GrantVIPQueryInfo::PayloadType>&& result) {
      CompleteTask(source);

      TTV_ErrorCode ec = TTV_EC_SUCCESS;
      GrantVIPErrorCode gqlErrorCode = GrantVIPErrorCode::SUCCESS;

      if (result.IsError()) {
        ec = result.GetErrorCode();
      } else {
        if (result.GetResult().error) {
          ec = TTV_EC_GRAPHQL_ERROR;
          gqlErrorCode = (*result.GetResult().error).code;
        }
      }

      if (taskCallback != nullptr) {
        taskCallback(ec, gqlErrorCode);
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::RevokeVIP(
  UserId userId, ChannelId channelId, const std::string& unvipUserName, RevokeVIPCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  graphql::RevokeVIPQueryInfo::InputParams inputs;
  inputs.authToken = oauthToken->GetToken();
  inputs.input.channelID = std::to_string(channelId);
  inputs.input.revokeeLogin = unvipUserName;

  auto task = std::make_shared<GraphQLTask<graphql::RevokeVIPQueryInfo>>(
    std::move(inputs), [this, taskCallback = std::move(callback)](GraphQLTask<graphql::RevokeVIPQueryInfo>* source,
                         Result<graphql::RevokeVIPQueryInfo::PayloadType>&& result) {
      CompleteTask(source);

      TTV_ErrorCode ec = TTV_EC_SUCCESS;
      RevokeVIPErrorCode gqlErrorCode = RevokeVIPErrorCode::SUCCESS;

      if (result.IsError()) {
        ec = result.GetErrorCode();
      } else {
        if (result.GetResult().error) {
          ec = TTV_EC_GRAPHQL_ERROR;
          gqlErrorCode = (*result.GetResult().error).code;
        }
      }

      if (taskCallback != nullptr) {
        taskCallback(ec, gqlErrorCode);
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::FetchChannelVIPs(ChannelId channelId, FetchChannelVIPsCallback&& callback) {
  graphql::FetchChannelVIPsQueryInfo::InputParams inputs;
  inputs.channelId = std::to_string(channelId);

  auto task = std::make_shared<GraphQLTask<graphql::FetchChannelVIPsQueryInfo>>(std::move(inputs),
    [this, taskCallback = std::move(callback)](GraphQLTask<graphql::FetchChannelVIPsQueryInfo>* source,
      Result<graphql::FetchChannelVIPsQueryInfo::PayloadType>&& result) {
      CompleteTask(source);

      TTV_ErrorCode ec = TTV_EC_SUCCESS;
      std::vector<std::string> vips;

      if (result.IsError()) {
        ec = result.GetErrorCode();
      } else {
        if (result.GetResult().vips.HasValue()) {
          for (auto& vipEdge : result.GetResult().vips.Value().edges) {
            if (vipEdge.node) {
              auto& user = vipEdge.node.Value();
              if (user.login) {
                vips.push_back(std::move(user.login.Value()));
              }
            }
          }
        }
      }

      if (taskCallback != nullptr) {
        taskCallback(ec, std::move(vips));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::UpdateUserColor(
  UserId userId, const std::string& color, UpdateUserColorCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatUpdateColorTask>(color, oauthToken->GetToken(),
    [this, user, oauthToken, callback = std::move(callback)](ChatUpdateColorTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::FetchChannelModerators(
  ChannelId channelId, const std::string& cursor, FetchChannelModeratorsCallback&& callback) {
  auto task = std::make_shared<ChatFetchChannelModeratorsTask>(channelId, cursor,
    [this, callback = std::move(callback)](ChatFetchChannelModeratorsTask* source, TTV_ErrorCode ec,
      std::vector<std::string>&& modNames, std::string&& nextCursor) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec, std::move(modNames), std::move(nextCursor));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::FetchGlobalBadges(FetchBadgesCallback&& callback) {
  graphql::FetchGlobalBadgesQueryInfo::InputParams inputs;

  auto task = std::make_shared<GraphQLTask<graphql::FetchGlobalBadgesQueryInfo>>(std::move(inputs),
    [this, taskCallback = std::move(callback)](GraphQLTask<graphql::FetchGlobalBadgesQueryInfo>* source,
      Result<graphql::FetchGlobalBadgesQueryInfo::PayloadType>&& result) {
      CompleteTask(source);

      TTV_ErrorCode ec = TTV_EC_SUCCESS;

      BadgeSet badgeSet;
      if (result.IsError()) {
        ec = result.GetErrorCode();
      } else {
        auto payload = result.GetResult();

        if (payload.badges.HasValue()) {
          for (const auto& badge : payload.badges.Value()) {
            if (!badge.HasValue()) {
              continue;
            }
            auto setId = badge.Value().setID;
            auto versionBadge = graphql::ToBadgeVersion(badge.Value());
            auto& namedBadges = badgeSet.badges[setId];
            namedBadges.name = setId;
            namedBadges.versions[versionBadge.name] = versionBadge;
          }
        }
      }

      if (taskCallback != nullptr) {
        taskCallback(ec, std::move(badgeSet));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::FetchChannelBadges(ChannelId channelId, FetchBadgesCallback&& callback) {
  graphql::FetchChannelBadgesQueryInfo::InputParams inputs;

  inputs.channelId = std::to_string(channelId);

  auto task = std::make_shared<GraphQLTask<graphql::FetchChannelBadgesQueryInfo>>(std::move(inputs),
    [this, taskCallback = std::move(callback)](GraphQLTask<graphql::FetchChannelBadgesQueryInfo>* source,
      Result<graphql::FetchChannelBadgesQueryInfo::PayloadType>&& result) {
      CompleteTask(source);

      TTV_ErrorCode ec = TTV_EC_SUCCESS;

      BadgeSet badgeSet;
      if (result.IsError()) {
        ec = result.GetErrorCode();
      } else {
        auto payload = result.GetResult();

        if (payload.broadcastBadges.HasValue()) {
          for (const auto& badge : payload.broadcastBadges.Value()) {
            if (!badge.HasValue()) {
              continue;
            }
            auto setId = badge.Value().setID;
            auto versionBadge = graphql::ToBadgeVersion(badge.Value());
            auto& namedBadges = badgeSet.badges[setId];
            namedBadges.name = setId;
            namedBadges.versions[versionBadge.name] = versionBadge;
          }
        }
      }

      if (taskCallback != nullptr) {
        taskCallback(ec, std::move(badgeSet));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::FetchVodCommentSettings(
  UserId userId, ChannelId channelId, FetchVodCommentSettingsCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatGetChannelVodCommentSettingsTask>(channelId, oauthToken->GetToken(),
    [this, callback](
      ChatGetChannelVodCommentSettingsTask* source, TTV_ErrorCode ec, ChannelVodCommentSettings&& result) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec, std::move(result));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::SetVodCommentFollowersOnlyDurationSeconds(
  UserId userId, ChannelId channelId, uint32_t durationSeconds, SetVodCommentSettingsCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatSetChannelVodCommentSettingsTask>(channelId, durationSeconds, oauthToken->GetToken(),
    [this, callback](ChatSetChannelVodCommentSettingsTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec);
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatAPITaskHost::SetVodCommentPublishingMode(
  UserId userId, ChannelId channelId, CommentPublishingMode mode, SetVodCommentSettingsCallback&& callback) {
  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatSetChannelVodCommentSettingsTask>(channelId, mode, oauthToken->GetToken(),
    [this, callback](ChatSetChannelVodCommentSettingsTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec);
      }
    });

  return StartTask(task);
}
