/********************************************************************************************
 * Twitch Broadcasting SDK
 *
 * This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
 * may not be copied or used except in accordance with the terms of that agreement
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 *********************************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatchannel.h"

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/internal/chatuserbadges.h"
#include "twitchsdk/chat/internal/chatuserblocklist.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"
#include "twitchsdk/chat/internal/task/chatchanneluserstask.h"
#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/eventtracker.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/settingrepository.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/getchanneltask.h"
#include "twitchsdk/core/task/simplejsonhttptask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/userrepository.h"

// TODO: Convert ChatChannel to a Component

#define CHAT_THREAD_ONLY
#define MAIN_THREAD_ONLY

namespace {
using namespace ttv;
using namespace ttv::chat;

const char* kLoggerName = "ChatChannel";
const char* kModeratorActionsTopicPrefix = "chat_moderator_actions.";
const char* kChatroomsUserV1TopicPrefix = "chatrooms-user-v1.";

const char* kChatChannelTmiHostNameSettingKey =
  "CHAT_CHANNEL_TMI_HOST_NAME";  // Use this specific host to connect to chat channels.  This should contain the
                                 // protocol as well.
const char* kChatAllowWssConnectionsSettingKey =
  "CHAT_ALLOW_WSS_CONNECTIONS";  // Use this to specify if wss connections are permitted
const char* kChatAllowWsConnectionsSettingKey =
  "CHAT_ALLOW_WS_CONNECTIONS";  // Use this to specify if ws connections are permitted
const char* kChatAllowRawTcpConnectionsSettingKey =
  "CHAT_ALLOW_RAW_TCP_CONNECTIONS";  // Use this to specifiy if raw tcp connections are permitted

// wss chat host
const char* kChatWssHost = "wss://irc-ws.chat.twitch.tv";

// ws chat host
const char* kChatWsHost = "ws://irc-ws.chat.twitch.tv";

// tcp chat host
const char* kChatTcpHost = "irc.chat.twitch.tv:6667";

const uint64_t kThreadShutdownTimeoutMilliseconds = 30000;
const uint64_t kUserMessageFlushIntervalMilliseconds = 500;
const size_t kUserMessageBatchSize = 64;
const size_t kMaxMainQueueFlushCount = 200;

const uint64_t kFetchChatPropertiesErrorIntervalMilliseconds =
  1 * 1000 * 30;  //!< When to retry fetching chat properties if it fails to be fetched
const uint64_t kFetchChatPropertiesRefetchIntervalMilliseconds = 1 * 1000 * 60 * 60 * 6;  //!< Refresh chat properties
const uint64_t kFetchChatPropertiesJitterMilliseconds =
  1 * 1000 * 60;  //!< The jitter to apply to chat property fetch intervals

const uint64_t kGlobalConnectionRetryJitterMilliseconds = 1000;
const uint64_t kGlobalConnectionCounterResetMilliseconds = 1 * 1000 * 60;
const uint64_t kConnectionRetryMaxBackOff = 120 * 1000;  // 2 minutes

const uint32_t kChatDelayMaximumSeconds = 10;

std::string FindOrDefault(
  const std::map<std::string, std::string>& kvp, const std::string& key, const std::string& defaultValue = "") {
  const auto findItr = kvp.find(key);

  if (findItr == kvp.cend()) {
    return defaultValue;
  }

  return findItr->second;
}

bool FindAndSet(const std::map<std::string, std::string>& kvp, const std::string& key, std::string& out) {
  const auto findItr = kvp.find(key);

  if (findItr == kvp.cend()) {
    return false;
  }

  out = findItr->second;
  return true;
}

bool IsSystemUser(const std::string& username) {
  return (username == "jtv") || (username == "twitchnotify");
}

bool IsPrivilegedUser(const UserMode& mode) {
  return (mode.globalModerator || mode.moderator || mode.broadcaster || mode.administrator || mode.staff || mode.vip);
}
}  // namespace

ttv::chat::ChatChannel::ChatChannel(const std::shared_ptr<User>& user, ChannelId channelId,
  std::shared_ptr<IChatChannelListener> chatCallbacks, std::shared_ptr<TaskRunner> taskRunner)
    : mChatCallbacks(chatCallbacks),
      mTaskRunner(taskRunner),
      mBackgroundTaskRunner(std::make_shared<TaskRunner>(
        "ChatChannel-" + std::to_string(channelId) + "(" + ttv::PointerToString(this) + ")")),
      mConnectionRetryTimer(kConnectionRetryMaxBackOff, kGlobalConnectionRetryJitterMilliseconds),
      mNextChatHostIndex(0),
      mConnectTrackingStartTime(0),
      mUserMessageFlushInterval(kUserMessageFlushIntervalMilliseconds),
      mConnectionError(TTV_EC_SUCCESS),
      mChannelState(State::Initialized),
      mReportedClientState(ChatChannelState::Disconnected),
      mUser(user),
      mBitsConfigFetchToken(0),
      mChannelId(channelId),
      mNumOutstandingTasks(0),
      mApplySlowMode(true),
      mDisconnectionRequested(false) {
  TTV_ASSERT(chatCallbacks != nullptr);
  TTV_ASSERT(taskRunner != nullptr);
  TTV_ASSERT(user != nullptr);

  mAnonymous = (user->GetUserId() == 0);
  if (!mAnonymous) {
    mModeratorActionsPubSubTopic =
      kModeratorActionsTopicPrefix + std::to_string(user->GetUserId()) + "." + std::to_string(channelId);
    mChatroomUserV1PubSubTopic = kChatroomsUserV1TopicPrefix + std::to_string(user->GetUserId());
  }

  mUserMessageBatch.reserve(kUserMessageBatchSize);

  if (user != nullptr) {
    mClientLocalUserInfo.userId = user->GetUserId();
    mClientLocalUserInfo.userName = user->GetUserName();
    mClientLocalUserInfo.displayName = user->GetDisplayName();
  }

  mServerLocalUserInfo = mClientLocalUserInfo;

  mSystemUserInfo.userName = "jtv";
  mSystemUserInfo.userMode.system = true;
  mSystemUserInfo.nameColorARGB = 0xFF000000;

  mPubSub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (mPubSub != nullptr) {
    mPubSubTopicListener = std::make_shared<PubSubTopicListener>(this);
    mPubSubTopicListenerHelper = std::make_shared<PubSubTopicListenerHelper>(mPubSub, mPubSubTopicListener);

    SubscribeTopics();
  }
}

ttv::chat::ChatChannel::~ChatChannel() {
  // make sure the app has released all lists by this point
  ForceShutdown();

  TTV_ASSERT(mChannelState == State::Initialized || mChannelState == State::ShutDown);
  TTV_ASSERT(mChatThread == nullptr);
}

void ttv::chat::ChatChannel::SetChatObjectFactory(std::shared_ptr<IChatObjectFactory> factory) {
  mChatObjectFactory = factory;
}

void ttv::chat::ChatChannel::SetSettingRepository(std::shared_ptr<SettingRepository> settings) {
  mSettingRepository = settings;
}

void ttv::chat::ChatChannel::SetState(State::Enum state) {
  CHAT_THREAD_ONLY

  // Avoid redundant changes
  if (state == mChannelState) {
    return;
  }

  ttv::trace::Message("Chat", MessageLevel::Debug, "ChatChannel changing state: %d -> %d", mChannelState, state);

  mChannelState = state;

  ChatChannelState clientState = mReportedClientState;
  TTV_ErrorCode clientError = mConnectionError;

  switch (state) {
    case State::Connecting: {
      clientState = ChatChannelState::Connecting;

      // Discard any previous connection
      CloseConnection();

      if (mServerChannelInfo.name.empty()) {
        FetchChannelInfo();
      }

      if (mChatProperties == nullptr) {
        FetchChatProperties();
      }

      if (mBitsConfiguration == nullptr) {
        FetchBitsConfiguration();
      }

      AttemptConnection();

      mConnectionRetryTimer.ClearGlobalReset();

      // Start the timer for connecting to the server
      mConnectionRetryTimer.ScheduleNextRetry();

      break;
    }
    case State::Connected: {
      mSlowModeTimer.Clear();
      mTimeoutTimer.Clear();

      clientState = ChatChannelState::Connected;

      // Start the timer which will reset the retry backoff
      mConnectionRetryTimer.StartGlobalReset(kGlobalConnectionCounterResetMilliseconds);

      mNextUserMessageFlush.Set(mUserMessageFlushInterval);

      // Flush all the messages that were supposed to be sent while we were trying to reconnect
      for (size_t i = 0; i < mQueuedChatMessages.size(); ++i) {
        (void)ProcessClientChatMessage(mQueuedChatMessages[i]);
      }
      mQueuedChatMessages.clear();

      if (mConnectTrackingStartTime > 0) {
        uint32_t loadTime = static_cast<uint32_t>(GetSystemTimeMilliseconds() - mConnectTrackingStartTime);
        mConnectTrackingStartTime = 0;

        ttv::TrackEvent("mobile_latency_event",
          {{"content_type", "channel_chat"}, {"latency_event", "sdk_chat_connected"}, {"load_time", loadTime}});
      }
      break;
    }
    case State::ShuttingDown: {
      clientState = ChatChannelState::Disconnecting;
      CloseConnection();
      break;
    }
    case State::ShutDown: {
      clientState = ChatChannelState::Disconnected;
      break;
    }
    default: { break; }
  }

  // Send the client notification
  if (clientState != mReportedClientState) {
    mReportedClientState = clientState;

    mToMainQ.push([this, clientState, clientError]() {
      ttv::trace::Message(
        "Chat", MessageLevel::Debug, "ChannelStateChangedClientMessage: %d - %d", clientState, clientError);

      auto user = mUser.lock();
      UserId userId = (user != nullptr) ? user->GetUserId() : 0;

      if (clientState == ChatChannelState::Disconnected) {
        CompleteShutdown();
      }

      // fire the notification
      mChatCallbacks->ChatChannelStateChanged(userId, mChannelId, clientState, clientError);
    });
  }
}

TTV_ErrorCode ttv::chat::ChatChannel::Connect() {
  MAIN_THREAD_ONLY

  TTV_ASSERT(mChatObjectFactory != nullptr);

  // Already leaving the channel so reject the request
  if (mDisconnectionRequested || mChannelState >= State::ShuttingDown) {
    return TTV_EC_CHAT_LEAVING_CHANNEL;
  }

  // Capture hosts to connect to
  if (mHosts.empty()) {
    if (mSettingRepository != nullptr) {
      std::string value;

      // Custom host
      bool customFound = mSettingRepository->GetSetting(kChatChannelTmiHostNameSettingKey, value);
      if (customFound) {
        mHosts.push_back(value);
      }

      // wss
      bool wssSettingFound = mSettingRepository->GetSetting(kChatAllowWssConnectionsSettingKey, value);
      std::transform(value.begin(), value.end(), value.begin(), ::tolower);
      if (!wssSettingFound || value == "true") {
        mHosts.push_back(kChatWssHost);
      }

      // ws
      mSettingRepository->GetSetting(kChatAllowWsConnectionsSettingKey, value);
      std::transform(value.begin(), value.end(), value.begin(), ::tolower);
      if (value == "true") {
        mHosts.push_back(kChatWsHost);
      }

      // Raw TCP
      mSettingRepository->GetSetting(kChatAllowRawTcpConnectionsSettingKey, value);
      std::transform(value.begin(), value.end(), value.begin(), ::tolower);
      if (value == "true") {
        mHosts.push_back(kChatTcpHost);
      }
    } else {
      mHosts.push_back(kChatWssHost);
    }
  }

  // Make sure we are able to attempt a connection to some host otherwise fast fail
  TTV_ErrorCode ec = TTV_EC_CHAT_NO_HOSTS;

  for (const auto& host : mHosts) {
    ec = ChatConnection::CheckFactoryAvailability(host);

    if (TTV_SUCCEEDED(ec)) {
      break;
    }
  }

  if (TTV_FAILED(ec)) {
    if (ec == TTV_EC_CHAT_NO_HOSTS) {
      ttv::trace::Message("Chat", MessageLevel::Error,
        "ChatChannel::Connect(): No chat hosts were configured, channel chat cannot connect - %s", ErrorToString(ec));
    } else if (ec == TTV_EC_NO_FACTORIES_REGISTERED) {
      ttv::trace::Message(
        "Chat", MessageLevel::Error, "ChatChannel::Connect(): No factories registered - %s", ErrorToString(ec));
    } else if (ec == TTV_EC_UNIMPLEMENTED) {
      ttv::trace::Message("Chat", MessageLevel::Error,
        "ChatChannel::Connect(): No registered factory is able to create socket for allowed chat protocols - %s",
        ErrorToString(ec));
    } else {
      ttv::trace::Message("Chat", MessageLevel::Error,
        "ChatChannel::Connect(): Unknown error when checking registered factories - %s", ErrorToString(ec));
    }

    return ec;
  }

  if (mChatThread == nullptr) {
    // Kick off the thread
    ec = ttv::CreateThread(std::bind(&ChatChannel::ThreadProc, this), "ttv::chat::ChatChannel", mChatThread);
    TTV_ASSERT(TTV_SUCCEEDED(ec) && mChatThread != nullptr);

    mChatThread->Run();
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::Disconnect() {
  MAIN_THREAD_ONLY

  if (mChatThread == nullptr || mDisconnectionRequested || mChannelState >= State::ShuttingDown) {
    return TTV_EC_CHAT_LEAVING_CHANNEL;
  }

  mDisconnectionRequested = true;

  // Cancel outstanding requests
  if (mBitsConfigFetchToken != 0 && mBitsConfigRepository != nullptr) {
    mBitsConfigRepository->CancelFetch(mBitsConfigFetchToken);
  }

  // create the message to send to the chat thread
  mToChatQ.push([this]() { ProcessDisconnectRequest(); });

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatChannel::CompleteShutdown() {
  MAIN_THREAD_ONLY

  TTV_ASSERT(mChannelState == State::ShutDown);

  mChatProperties.reset();
  mChatThread.reset();
  mChatObjectFactory.reset();
  mChatConnection.reset();
  mChannelRepository.reset();
  mBitsConfigRepository.reset();
  mTaskRunner.reset();
  mBackgroundTaskRunner.reset();
  mSettingRepository.reset();
  mBitsConfiguration.reset();
  mUser.reset();
  mPubSub.reset();
  mPubSubTopicListener.reset();
  mPubSubTopicListenerHelper.reset();
}

TTV_ErrorCode ttv::chat::ChatChannel::SendChatMessage(const std::string& message) {
  MAIN_THREAD_ONLY

  if (mChatThread == nullptr) {
    return TTV_EC_CHAT_NOT_IN_CHANNEL;
  }

  // not allowed when anonymous
  if (mAnonymous) {
    return TTV_EC_CHAT_ANON_DENIED;
  }

  // make sure the queue isn't growing too large
  if (mToChatQ.unsafe_size() > 8) {
    return TTV_EC_CHAT_TOO_MANY_REQUESTS;
  }

  // check to see if messages are being submitted too quickly
  if (!mMessagePacer.TrackMessage()) {
    return TTV_EC_CHAT_MESSAGE_SPAM_DISCARDED;
  }

  // pass the request to the chat thread
  mToChatQ.push([this, message]() { ProcessClientChatMessage(message); });

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::FetchUserList(const FetchUserListCallback& callback) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto user = mUser.lock();
  if (user == nullptr) {
    ec = TTV_EC_AUTHENTICATION;
  } else if (mChannelState >= State::ShuttingDown) {
    ec = TTV_EC_CHAT_LEAVING_CHANNEL;
  } else {
    auto fetchUsersTask = std::make_shared<ChatChannelUsersTask>(
      mClientChannelInfo.name, [this, callback](ChatChannelUsersTask* /*source*/, TTV_ErrorCode callbackEc,
                                 std::shared_ptr<ChatChannelUsersTask::Result> result) {
        mNumOutstandingTasks--;

        if (callback != nullptr) {
          callback(callbackEc, std::move(result->users));
        }
      });
    mNumOutstandingTasks++;
    bool added = mTaskRunner->AddTask(fetchUsersTask);
    if (!added) {
      ec = TTV_EC_API_REQUEST_FAILED;
      mNumOutstandingTasks--;
    }
  }
  return ec;
}

uint64_t ttv::chat::ChatChannel::GetRemainingSlowModeTime() {
  return mSlowModeTimer.GetRemainingTime();
}

TTV_ErrorCode ttv::chat::ChatChannel::FetchChannelInfo() {
  CHAT_THREAD_ONLY

  if (mChannelState >= State::ShuttingDown) {
    return TTV_EC_CHAT_LEAVING_CHANNEL;
  }

  ChannelInfo info;
  if (TTV_SUCCEEDED(mChannelRepository->GetChannelInfo(mChannelId, info))) {
    ProcessChannelInfoFetchResult(info);
  } else {
    mNumOutstandingTasks++;
    bool added = mBackgroundTaskRunner->AddTask(std::make_shared<GetChannelTask>(
      mChannelId, [this](GetChannelTask* /*task*/, TTV_ErrorCode ec, std::shared_ptr<GetChannelTask::Result> result) {
        mNumOutstandingTasks--;

        if (mChannelState >= State::ShuttingDown) {
          return;
        }

        if (TTV_SUCCEEDED(ec)) {
          TTV_ASSERT(result != nullptr);
          ProcessChannelInfoFetchResult(result->channelInfo);
        } else {
          mFetchChatPropertiesTimer.SetWithJitter(
            kFetchChatPropertiesErrorIntervalMilliseconds, kFetchChatPropertiesJitterMilliseconds);
        }
      }));

    if (!added) {
      mNumOutstandingTasks--;
      return TTV_EC_SHUTTING_DOWN;
    }
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::FetchChatProperties() {
  CHAT_THREAD_ONLY

  if (mChannelState >= State::ShuttingDown) {
    return TTV_EC_CHAT_LEAVING_CHANNEL;
  }

  mNumOutstandingTasks++;

  auto callback = [this](ChatPropertiesTask* /*source*/, TTV_ErrorCode ec,
                    std::shared_ptr<ChatPropertiesTask::Result> result) {
    TTV_ASSERT(mNumOutstandingTasks > 0);
    mNumOutstandingTasks--;

    if (mChannelState >= State::ShuttingDown) {
      return;
    }

    ProcessChatPropertyFetchResult(ec, result);
  };

  // Kick off the request for chat properties
  std::shared_ptr<ChatPropertiesTask> task = std::make_shared<ChatPropertiesTask>(mChannelId, callback);
  bool added = mBackgroundTaskRunner->AddTask(task);
  if (!added) {
    // Decrement since the task will never run.
    mNumOutstandingTasks--;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::FetchBitsConfiguration() {
  CHAT_THREAD_ONLY

  if (mChannelState >= State::ShuttingDown) {
    return TTV_EC_CHAT_LEAVING_CHANNEL;
  }

  mToMainQ.push([this]() {
    if (mChannelState >= State::ShuttingDown || mBitsConfigFetchToken != 0) {
      return;
    }

    mNumOutstandingTasks++;

    UserId userId = 0;
    auto user = mUser.lock();
    if (user != nullptr) {
      userId = user->GetUserId();
    }

    auto ec = mBitsConfigRepository->FetchChannelBitsConfiguration(userId, mChannelId,
      [this](TTV_ErrorCode callbackEc, const std::shared_ptr<BitsConfiguration>& config) {
        mNumOutstandingTasks--;

        mBitsConfigFetchToken = 0;

        if (mChannelState >= State::ShuttingDown) {
          return;
        }

        mToChatQ.push([this, callbackEc, config] { ProcessBitsConfigFetchResult(callbackEc, config); });
      },
      mBitsConfigFetchToken);

    if (TTV_FAILED(ec)) {
      // Decrement since the task will never run.
      mNumOutstandingTasks--;
    }
  });

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::ProcessChannelInfoFetchResult(const ChannelInfo& channelInfo) {
  CHAT_THREAD_ONLY

  mServerChannelInfo.name = channelInfo.name;
  if (mChatConnection != nullptr) {
    mChatConnection->SetChannelName(channelInfo.name);
  }

  // As of right now, at least, the broadcaster name and the channel name are one and the same.
  mBroadcasterName = channelInfo.name;

  // Update the client channel info
  mToMainQ.push([this, channelInfo = mServerChannelInfo] { SetClientChannelInfo(channelInfo); });

  FetchChatProperties();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::ProcessChatPropertyFetchResult(
  TTV_ErrorCode ec, std::shared_ptr<ChatPropertiesTask::Result> result) {
  CHAT_THREAD_ONLY

  if (result != nullptr) {
    TTV_ASSERT(TTV_SUCCEEDED(ec));

    mChatProperties = result;
  }

  // Handle an API failure or invalid response
  if (TTV_FAILED(ec)) {
    // We don't have it yet
    if (mChatProperties == nullptr) {
      ttv::trace::Message("Chat", MessageLevel::Error, "Failed to get channel properties, can't connect until fetched");

      // Schedule another refresh for sometime soon
      mFetchChatPropertiesTimer.SetWithJitter(
        kFetchChatPropertiesErrorIntervalMilliseconds, kFetchChatPropertiesJitterMilliseconds);
    }
    // We already have from a previous fetch
    else {
      ttv::trace::Message("Chat", MessageLevel::Error, "Failed to get channel properties, using previously cached");
    }
  }

  // Schedule a refresh for sometime in the future
  if (!mFetchChatPropertiesTimer.IsSet()) {
    mFetchChatPropertiesTimer.SetWithJitter(
      kFetchChatPropertiesRefetchIntervalMilliseconds, kFetchChatPropertiesJitterMilliseconds);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::ProcessBitsConfigFetchResult(
  TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
  CHAT_THREAD_ONLY

  // Handle an API failure or invalid response
  if (TTV_SUCCEEDED(ec) && config != nullptr) {
    mBitsConfiguration = config;
  } else {
    // We don't have it yet
    if (mBitsConfiguration == nullptr) {
      ttv::trace::Message("Chat", MessageLevel::Error, "Failed to get bits configuration, can't connect until fetched");

      // Schedule another refresh for sometime soon
      mFetchBitsConfigTimer.SetWithJitter(
        kFetchChatPropertiesErrorIntervalMilliseconds, kFetchChatPropertiesJitterMilliseconds);
    }
    // We already have from a previous fetch
    else {
      ttv::trace::Message("Chat", MessageLevel::Error, "Failed to get bits configuration, using previously cached");
    }
  }

  // Schedule a refresh for sometime in the future
  if (!mFetchBitsConfigTimer.IsSet()) {
    mFetchBitsConfigTimer.SetWithJitter(
      kFetchChatPropertiesRefetchIntervalMilliseconds, kFetchChatPropertiesJitterMilliseconds);
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatChannel::SetClientChannelInfo(const ChatChannelInfo& channelInfo) {
  ttv::trace::Message("Chat", MessageLevel::Debug, "ChannelInfoChangedClientMessage");

  mClientChannelInfo = channelInfo;

  auto user = mUser.lock();
  UserId userId = (user != nullptr) ? user->GetUserId() : 0;

  // fire the notification
  mChatCallbacks->ChatChannelInfoChanged(userId, mChannelId, mClientChannelInfo);
}

void ttv::chat::ChatChannel::SetClientChatRestrictions(const ChatChannelRestrictions& restrictions) {
  ttv::trace::Message("Chat", MessageLevel::Debug, "ChatRestrictionsChangedClientMessage");

  mClientChatRestrictions = restrictions;

  auto user = mUser.lock();
  UserId userId = (user != nullptr) ? user->GetUserId() : 0;

  mChatCallbacks->ChatChannelRestrictionsChanged(userId, mChannelId, mClientChatRestrictions);
}

TTV_ErrorCode ttv::chat::ChatChannel::AttemptConnection() {
  CHAT_THREAD_ONLY

  TTV_ASSERT(mHosts.size() > 0);
  TTV_ASSERT(mChannelState == State::Connecting);
  TTV_ASSERT(mChatConnection == nullptr);

  TTV_ErrorCode ec = TTV_EC_CHAT_COULD_NOT_CONNECT;

  auto user = mUser.lock();
  if (user != nullptr) {
    mChatConnection = std::make_shared<ChatConnection>(mChannelId, user);
    if (!mServerChannelInfo.name.empty()) {
      mChatConnection->SetChannelName(mServerChannelInfo.name);
    }
  } else {
    return TTV_EC_NEED_TO_LOGIN;
  }

  mChatConnection->SetListener(this);
  mChatConnection->SetChatObjectFactory(mChatObjectFactory);

  mConnectionError = TTV_EC_SUCCESS;

  auto connect = [this](const std::string& uri) -> TTV_ErrorCode {
    // Abort connection attempts
    if (mDisconnectionRequested) {
      return TTV_EC_CHAT_LEAVING_CHANNEL;
    }

    TTV_ErrorCode connectEc = mChatConnection->Connect(uri);

    // Socket connected
    if (TTV_SUCCEEDED(connectEc)) {
      return connectEc;
    } else {
      return TTV_EC_CHAT_COULD_NOT_CONNECT;
    }
  };

  std::string uri;

  // Check for an explicitly configured host
  if (mSettingRepository != nullptr) {
    mSettingRepository->GetSetting(kChatChannelTmiHostNameSettingKey, uri);
  }

  // Process the regular host list
  if (uri == "") {
    // Try each host in the list until one connects
    // We don't always start at the first one in case a host was unresponsive during the handshake
    while (mNextChatHostIndex < mHosts.size()) {
      // Try the next host
      uri = mHosts[mNextChatHostIndex];
      mNextChatHostIndex++;

      // NOTE: Successful connection does not mean that it's ready to send messages
      ec = connect(uri);

      // Found one that we could connect to or leaving the channel
      if (TTV_SUCCEEDED(ec) || ec == TTV_EC_CHAT_LEAVING_CHANNEL) {
        break;
      }
    }

    mNextChatHostIndex = mNextChatHostIndex % mHosts.size();
  }
  // Use the setting
  else {
    ec = connect(uri);
  }

  // Couldn't connect after trying several hosts
  if (TTV_FAILED(ec)) {
    if (mDisconnectionRequested) {
      ttv::trace::Message("Chat", MessageLevel::Info, "Connection attempt aborted by client request");
    } else {
      ttv::trace::Message(
        "Chat", MessageLevel::Info, "Connection to all attempted hosts failed, waiting for next round of attempts");
    }

    mChatConnection.reset();
    mConnectionRetryTimer.ScheduleNextRetry();
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatChannel::ProcessDisconnectRequest() {
  CHAT_THREAD_ONLY

  mConnectionError = TTV_EC_SUCCESS;
  SetState(State::ShuttingDown);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::ProcessClientChatMessage(const std::string& msg) {
  CHAT_THREAD_ONLY

  // save the message for when we reconnect
  if (mChannelState != State::Connected) {
    mQueuedChatMessages.push_back(msg);
    return TTV_EC_SUCCESS;
  }

  // if slow mode is enabled and the slowModeTimer is not currently running then enable the timeout
  if (mApplySlowMode && !mServerChannelInfo.localUserRestriction.slowMode) {
    mSlowModeTimer.Set(mServerChatRestrictions.slowModeDuration * 1000);
  }

  return mChatConnection->SendChatMessage(msg, mServerLocalUserInfo);
}

TTV_ErrorCode ttv::chat::ChatChannel::ProcessClientRequestQueue() {
  CHAT_THREAD_ONLY

  // go through the queue of requests from the client and handle them internally
  ThreadEvent event;
  while (mToChatQ.try_pop(event)) {
    event();
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannel::FlushClientEvents() {
  MAIN_THREAD_ONLY

  // go through the list of events from the chat thread and flush them to the client thread
  ThreadEvent event;

  size_t numProcessed = 0;

  while (numProcessed < kMaxMainQueueFlushCount && mToMainQ.try_pop(event)) {
    numProcessed++;
    event();
  }

  return TTV_EC_SUCCESS;
}

bool ttv::chat::ChatChannel::UpdateRestriction() {
  CHAT_THREAD_ONLY

  bool privileged = IsPrivilegedUser(mServerLocalUserInfo.userMode);

  RestrictionReason restriction;

  restriction.banned = mServerLocalUserInfo.userMode.banned;
  restriction.anonymous = mAnonymous;
  restriction.subscribersOnly =
    (mServerChatRestrictions.subscribersOnly && !mServerLocalUserInfo.userMode.subscriber && !privileged);

  // TODO: We currently don't have access to the setting that specifies whether slow mode applies to subscribers but we
  // really should respect that flag sometime
  mApplySlowMode = mServerChatRestrictions.slowMode && !privileged;

  if (mApplySlowMode) {
    restriction.slowMode = (mSlowModeTimer.IsSet() && !mSlowModeTimer.Check(true));
  } else {
    mSlowModeTimer.Clear();
  }

  restriction.timeout = (mTimeoutTimer.IsSet() && !mTimeoutTimer.Check(true));

  // notify if changed
  if (restriction != mServerChannelInfo.localUserRestriction) {
    mServerChannelInfo.localUserRestriction = restriction;

    mToMainQ.push([this, channelInfo = mServerChannelInfo] { SetClientChannelInfo(channelInfo); });

    return true;
  } else {
    return false;
  }
}

void ttv::chat::ChatChannel::CloseConnection() {
  if (mChatConnection != nullptr) {
    mChatConnection->Disconnect();
    mChatConnection.reset();
  }
}

void ttv::chat::ChatChannel::ThreadProc() {
  CHAT_THREAD_ONLY

  ttv::AutoTracer entryAndExitTrace("Chat", MessageLevel::Debug, "ChatChannel::ThreadProc");

  SetState(State::Connecting);

  while (mChannelState < State::ShuttingDown) {
    // Fetch chat properties if needed
    if (mFetchChatPropertiesTimer.Check(true)) {
      if (mServerChannelInfo.name.empty()) {
        FetchChannelInfo();
      } else {
        FetchChatProperties();
      }
    }

    // Fetch channel's bit configuration if needed
    if (mFetchBitsConfigTimer.Check(true)) {
      FetchBitsConfiguration();
    }

    mBackgroundTaskRunner->PollTasks();

    mConnectionRetryTimer.CheckGlobalReset();

    if (mChannelState < State::ShuttingDown) {
      // check if waiting to try a round of reconnects
      switch (mChannelState) {
        case State::Connecting: {
          if (mChatConnection == nullptr) {
            // Check if we've waited long enough to retry again
            if (mConnectionRetryTimer.CheckNextRetry()) {
              AttemptConnection();
            }
          }

          break;
        }
        default: { break; }
      }

      if (mChatConnection != nullptr) {
        // Keep a reference to the connection around in case it is implicitly destroyed in an event
        auto connection = mChatConnection;

        // Events will be fired on the ChatConnection::Listener interface here
        connection->Update();
      }

      (void)UpdateRestriction();

      ProcessClientRequestQueue();
    }

    if (mChannelState < State::ShuttingDown) {
      FlushUserMessages(false);

      if (mChannelState == State::Connecting) {
        // Poll at a faster cadence while connecting.
        Sleep(10);
      } else {
        Sleep(250);
      }
    }
  }

  mConnectionRetryTimer.Clear();

  // Need to ensure that all outstanding tasks are complete
  WaitForExpiry tasksWaitTimer;
  tasksWaitTimer.Set(kThreadShutdownTimeoutMilliseconds);

  while (mNumOutstandingTasks > 0) {
    mBackgroundTaskRunner->PollTasks();

    // Make sure we don't hang forever - this should never occur
    if (tasksWaitTimer.Check(true)) {
      ttv::trace::Message("Chat", MessageLevel::Error, "ThreadProc is waiting too long for tasks to complete");
      TTV_ASSERT(false);
    }

    Sleep(10);
  }

  // TODO: https://jira.twitch.com/browse/SDK-381
  // Remove this once we do ChatChannel refactor with EventScheduler
  mBackgroundTaskRunner->Shutdown();
  mPubSubTopicListenerHelper->Shutdown();

  while (!mBackgroundTaskRunner->IsShutdown() ||
         mPubSubTopicListenerHelper->GetState() != PubSubTopicListenerHelper::State::Shutdown) {
    Sleep(10);
  }

  SetState(State::ShutDown);
}

/**
 * Creates a message from the chat threat to the main thread containing the queued up chat messages.
 */
void ttv::chat::ChatChannel::FlushUserMessages(bool force) {
  CHAT_THREAD_ONLY

  // fire the user join and leave notification if needed
  if (!force && !mNextUserMessageFlush.Check(true)) {
    return;
  }

  // schedule the next flush check
  mNextUserMessageFlush.Set(mUserMessageFlushInterval);

  // send the messages to the client
  // nothing to do
  if (mUserMessageBatch.empty() && mDelayedMessageBatch.empty()) {
    return;
  }

  // Only add user messages if the queue is draining fast enough
  if (mToMainQ.unsafe_size() < kMaxMainQueueFlushCount) {
    std::vector<LiveChatMessage> messageBatchToSend;

    UserId userId = 0;
    auto user = mUser.lock();
    if (user != nullptr) {
      userId = user->GetUserId();
    }

    uint32_t chatDelay = 0;
    if (mChatProperties != nullptr && mChatProperties->chat_delay_duration < kChatDelayMaximumSeconds) {
      chatDelay = mChatProperties->chat_delay_duration;
    }

    // Send all the messages if user is a moderator or if there's no delay
    if (mServerLocalUserInfo.userMode.moderator || mServerLocalUserInfo.userMode.globalModerator ||
        mServerLocalUserInfo.userMode.broadcaster || chatDelay == 0) {
      if (mDelayedMessageBatch.empty()) {
        messageBatchToSend = std::move(mUserMessageBatch);
      } else {
        // In the case where the user becomes a moderator while in chat
        messageBatchToSend = std::move(mDelayedMessageBatch);
        mDelayedMessageBatch.clear();
        std::move(mUserMessageBatch.begin(), mUserMessageBatch.end(), std::back_inserter(messageBatchToSend));
      }
    } else {
      std::move(mUserMessageBatch.begin(), mUserMessageBatch.end(), std::back_inserter(mDelayedMessageBatch));

      auto iter = mDelayedMessageBatch.begin();
      while (iter != mDelayedMessageBatch.end()) {
        if (iter->messageInfo.userId == userId) {
          // A user sees their own messages without a delay
          std::move(iter, iter + 1, std::back_inserter(messageBatchToSend));
          iter = mDelayedMessageBatch.erase(iter);
          continue;
        } else if (mTimedOutUsersInDelayWindow.find(iter->messageInfo.userId) != mTimedOutUsersInDelayWindow.end()) {
          iter->messageInfo.flags.ignored = true;
        }

        ++iter;
      }
      mTimedOutUsersInDelayWindow.clear();

      Timestamp currentTime = GetCurrentTimeAsUnixTimestamp();
      iter = std::upper_bound(mDelayedMessageBatch.begin(), mDelayedMessageBatch.end(), currentTime - chatDelay,
        [](auto timestampCutoff, const LiveChatMessage& message) {
          return timestampCutoff < message.messageInfo.timestamp;
        });

      if (iter != mDelayedMessageBatch.begin()) {
        std::move(mDelayedMessageBatch.begin(), iter, std::back_inserter(messageBatchToSend));
        mDelayedMessageBatch.erase(mDelayedMessageBatch.begin(), iter);
      }
    }

    mUserMessageBatch.reserve(kUserMessageBatchSize);

    if (!messageBatchToSend.empty()) {
      mToMainQ.push([this, batch = std::move(messageBatchToSend)]() mutable {
        ttv::trace::Message("Chat", MessageLevel::Debug, "UserMessageListClientMessage");

        UserId mainThreadUserId = 0;

        // update the ignore flag on messages
        auto mainThreadUser = mUser.lock();
        if (mainThreadUser != nullptr) {
          mainThreadUserId = mainThreadUser->GetUserId();

          std::shared_ptr<ChatUserBlockList> blockList =
            mainThreadUser->GetComponentContainer()->GetComponent<ChatUserBlockList>();
          if (blockList != nullptr) {
            for (auto& message : batch) {
              if (blockList->IsUserBlocked(message.messageInfo.userId)) {
                message.messageInfo.flags.ignored = true;
              }
            }
          }
        }

        mChatCallbacks->ChatChannelMessagesReceived(mainThreadUserId, mChannelId, batch);
      });
    }
  } else {
    ttv::trace::Message("Chat", MessageLevel::Warning, "Dropping message batch due to message queue backup");
  }

  // Guarantee the batch is always cleared after flushing
  mUserMessageBatch.clear();
}

void ttv::chat::ChatChannel::ForceShutdown() {
  MAIN_THREAD_ONLY

  if (IsShutdown()) {
    return;
  }

  // force a disconnect
  (void)Disconnect();

  // block until the chat thread is done
  mChatThread->Join();

  // cleanup the thread data
  CompleteShutdown();
}

void ttv::chat::ChatChannel::HandleConnectionIssue(bool recoverableError) {
  // Schedule a connection retry attempt
  if (recoverableError && !mDisconnectionRequested) {
    CloseConnection();
    mConnectionRetryTimer.ScheduleNextRetry();

    SetState(State::Connecting);
  } else {
    SetState(State::ShuttingDown);
  }
}

// ChatConnection::Listener implementation /////////////////////////////////////////////////////////////////////////

void ttv::chat::ChatChannel::OnConnected(ChatConnection* /*source*/) {
  CHAT_THREAD_ONLY
  TTV_ASSERT(mChannelState == State::Connecting);

  SetState(State::Connected);
}

void ttv::chat::ChatChannel::OnConnectionFailed(ChatConnection* /*source*/, TTV_ErrorCode ec) {
  CHAT_THREAD_ONLY
  TTV_ASSERT(mChannelState == State::Connecting);

  mConnectionError = ec;

  // Schedule a connection retry attempt
  bool recoverableError = (ec == TTV_EC_CHAT_COULD_NOT_CONNECT);
  HandleConnectionIssue(recoverableError);
}

void ttv::chat::ChatChannel::OnConnectionLost(ChatConnection* /*source*/, TTV_ErrorCode ec) {
  CHAT_THREAD_ONLY
  TTV_ASSERT(mChannelState == State::Connecting || mChannelState == State::Connected);

  mConnectionError = ec;

  // Schedule a connection retry attempt
  bool recoverableError = (ec == TTV_EC_CHAT_LOST_CONNECTION);
  HandleConnectionIssue(recoverableError);
}

void ttv::chat::ChatChannel::OnCleared(
  ChatConnection* /*source*/, const std::string& /*userName*/, const std::map<std::string, std::string>& messageTags) {
  CHAT_THREAD_ONLY

  UserId targetUserId = 0;
  auto iter = messageTags.find("target-user-id");
  if (iter != messageTags.end()) {
    UserId parsedUserId;
    if (ParseNum(iter->second, parsedUserId)) {
      targetUserId = parsedUserId;
    }
  }

  if (targetUserId != 0) {
    mTimedOutUsersInDelayWindow.insert(targetUserId);
  }

  mToMainQ.push([this, targetUserId] {
    auto user = mUser.lock();
    UserId userId = (user != nullptr) ? user->GetUserId() : 0;

    // fire the notification
    if (targetUserId != 0) {
      mChatCallbacks->ChatChannelUserMessagesCleared(userId, mChannelId, targetUserId);
    } else {
      mChatCallbacks->ChatChannelMessagesCleared(userId, mChannelId);
    }
  });
}

void ttv::chat::ChatChannel::HandleMessageReceived(const std::string& username, const std::string& message,
  const std::map<std::string, std::string>& messageTags, const MessageInfo::Flags& flags) {
  CHAT_THREAD_ONLY

  // Don't process any more messages if exiting the channel
  if (mChannelState >= State::ShuttingDown) {
    return;
  }

  ChatUserInfo userInfo;

  // From a regular user
  if (IsSystemUser(username)) {
    userInfo = mSystemUserInfo;
    userInfo.userName = username;
  } else {
    GenerateUserInfo(username, messageTags, userInfo);
  }

  // Flush the batch if it's full
  if (mUserMessageBatch.size() >= kUserMessageBatchSize) {
    FlushUserMessages(true);
  }

  // Add to the batch
  LiveChatMessage tokenizedMessage;
  GenerateLiveMessage(tokenizedMessage, mTokenizationOptions, userInfo, message, messageTags, flags);
  mUserMessageBatch.emplace_back(std::move(tokenizedMessage));
}

void ttv::chat::ChatChannel::HandleUserNotice(
  const std::string& message, const std::map<std::string, std::string>& messageTags) {
  CHAT_THREAD_ONLY

  // Don't process any more messages if exiting the channel
  if (mChannelState >= State::ShuttingDown) {
    return;
  }

  // Flush this batch of messages before we end up calling back with the notice.
  FlushUserMessages(true);

  // For whatever reason, the type field is keyed with "msg-id"

  std::string msgId;

  // Do not handle messages that have msg-id missing
  if (!FindAndSet(messageTags, "msg-id", msgId)) {
    return;
  }

  if (msgId == "sub") {
    return HandleSubscriptionNotice(SubscriptionNotice::Type::Sub, message, messageTags);
  }

  if (msgId == "resub") {
    return HandleSubscriptionNotice(SubscriptionNotice::Type::Resub, message, messageTags);
  }

  if (msgId == "subgift") {
    return HandleSubscriptionNotice(SubscriptionNotice::Type::SubGift, message, messageTags);
  }

  if (msgId == "extendsub") {
    return HandleSubscriptionNotice(SubscriptionNotice::Type::ExtendSub, message, messageTags);
  }

  if (msgId == "submysterygift") {
    return HandleSubscriptionNotice(SubscriptionNotice::Type::SubMassGift, message, messageTags);
  }

  if (msgId == "raid") {
    return HandleRaidNotice(message, messageTags);
  }

  if (msgId == "unraid") {
    return HandleUnraidNotice(message, messageTags);
  }

  if (msgId == "ritual") {
    auto ritualType = FindOrDefault(messageTags, "msg-param-ritual-name");
    if (ritualType == "new_chatter") {
      return HandleFirstTimeChatterNotice(message, messageTags);
    }
  }

  HandleGenericNotice(message, messageTags);
}

void ttv::chat::ChatChannel::HandleSubscriptionNotice(
  SubscriptionNotice::Type type, const std::string& message, const std::map<std::string, std::string>& messageTags) {
  SubscriptionNotice notice;

  notice.type = type;

  auto tagEnd = messageTags.end();
  auto tagIter = messageTags.find("msg-param-should-share-streak");
  if (tagIter != tagEnd && tagIter->second == "1") {
    notice.shouldShowSubStreak = true;
    tagIter = messageTags.find("msg-param-streak-months");

    if (tagIter != tagEnd) {
      ParseNum(tagIter->second, notice.subStreakMonthCount);
    }
  }

  tagIter = messageTags.find("msg-param-cumulative-months");
  if (tagIter != tagEnd) {
    ParseNum(tagIter->second, notice.subCumulativeMonthCount);
  }

  tagIter = messageTags.find("id");
  if (tagIter != tagEnd) {
    notice.messageId = tagIter->second;
  }

  if (type == SubscriptionNotice::Type::SubGift) {
    tagIter = messageTags.find("msg-param-recipient-display-name");
    if (tagIter != tagEnd) {
      notice.recipient.displayName = tagIter->second;
    }
    tagIter = messageTags.find("msg-param-recipient-user-name");
    if (tagIter != tagEnd) {
      notice.recipient.userName = tagIter->second;
    }
    tagIter = messageTags.find("msg-param-recipient-id");
    if (tagIter != tagEnd) {
      UserId userId = 0;
      if (ParseNum(tagIter->second, userId)) {
        notice.recipient.userId = static_cast<uint32_t>(userId);
      }
    }
  } else if (type == SubscriptionNotice::Type::SubMassGift) {
    tagIter = messageTags.find("msg-param-mass-gift-count");
    if (tagIter != tagEnd) {
      ParseNum(tagIter->second, notice.massGiftCount);
    }
  }

  if (type == SubscriptionNotice::Type::SubGift || type == SubscriptionNotice::Type::SubMassGift) {
    tagIter = messageTags.find("msg-param-sender-count");
    if (tagIter != tagEnd) {
      ParseNum(tagIter->second, notice.senderCount);
    }
  }

  if (type == SubscriptionNotice::Type::ExtendSub) {
    tagIter = messageTags.find("msg-param-sub-benefit-end-month");
    if (tagIter != tagEnd) {
      ParseNum(tagIter->second, notice.benefitEndMonth);
    }
  }

  tagIter = messageTags.find("msg-param-sub-plan");
  if (tagIter != tagEnd) {
    if (tagIter->second == "Prime") {
      notice.plan = SubscriptionNotice::Plan::Prime;
    } else if (tagIter->second == "1000") {
      notice.plan = SubscriptionNotice::Plan::Sub1000;
    } else if (tagIter->second == "2000") {
      notice.plan = SubscriptionNotice::Plan::Sub2000;
    } else if (tagIter->second == "3000") {
      notice.plan = SubscriptionNotice::Plan::Sub3000;
    }
  }

  tagIter = messageTags.find("msg-param-sub-plan-name");
  if (tagIter != tagEnd) {
    notice.planDisplayName = tagIter->second;
  }

  tagIter = messageTags.find("system-msg");
  if (tagIter != tagEnd) {
    notice.systemMessage = tagIter->second;
  }

  tagIter = messageTags.find("login");
  const std::string& username = (tagIter != tagEnd) ? tagIter->second : "";

  ChatUserInfo userInfo;
  GenerateUserInfo(username, messageTags, userInfo);

  auto tokenizedMessage = std::make_unique<MessageInfo>();
  GenerateMessage(*tokenizedMessage, mTokenizationOptions, userInfo, message, messageTags, {});
  notice.userMessage = std::move(tokenizedMessage);

  mToMainQ.push([this, notice = std::move(notice)]() {
    auto user = mUser.lock();
    UserId userId = (user != nullptr) ? user->GetUserId() : 0;

    mChatCallbacks->ChatChannelSubscriptionNoticeReceived(userId, mChannelId, notice);
  });
}

void ttv::chat::ChatChannel::HandleFirstTimeChatterNotice(
  const std::string& message, const std::map<std::string, std::string>& messageTags) {
  FirstTimeChatterNotice notice;

  auto tagEnd = messageTags.end();

  auto tagIter = messageTags.find("id");
  if (tagIter != tagEnd) {
    notice.messageId = tagIter->second;
  }

  tagIter = messageTags.find("system-msg");
  if (tagIter != tagEnd) {
    notice.systemMessage = tagIter->second;
  }

  tagIter = messageTags.find("login");
  const std::string& username = (tagIter != tagEnd) ? tagIter->second : "";

  ChatUserInfo userInfo;
  GenerateUserInfo(username, messageTags, userInfo);

  MessageInfo tokenizedMessage;
  GenerateMessage(tokenizedMessage, mTokenizationOptions, userInfo, message, messageTags, {});
  notice.userMessage = std::move(tokenizedMessage);

  mToMainQ.push([this, notice = std::move(notice)]() {
    auto user = mUser.lock();
    UserId userId = (user != nullptr) ? user->GetUserId() : 0;

    mChatCallbacks->ChatChannelFirstTimeChatterNoticeReceived(userId, mChannelId, notice);
  });
}

void ttv::chat::ChatChannel::HandleRaidNotice(
  const std::string& /*message*/, const std::map<std::string, std::string>& messageTags) {
  RaidNotice notice;

  auto tagEnd = messageTags.end();
  auto tagIter = messageTags.find("msg-param-login");
  if (tagIter != tagEnd) {
    notice.raidingUserInfo.userName = tagIter->second;
  }

  tagIter = messageTags.find("msg-param-displayName");
  if (tagIter != tagEnd) {
    notice.raidingUserInfo.displayName = tagIter->second;
  }

  tagIter = messageTags.find("user-id");
  if (tagIter != tagEnd) {
    UserId userId = 0;
    if (ParseNum(tagIter->second, userId)) {
      notice.raidingUserInfo.userId = userId;
    }
  }

  tagIter = messageTags.find("system-msg");
  if (tagIter != tagEnd) {
    notice.systemMessage = tagIter->second;
  }

  tagIter = messageTags.find("msg-param-viewerCount");
  if (tagIter != tagEnd) {
    uint32_t viewerCount = 0;
    if (ParseNum(tagIter->second, viewerCount)) {
      notice.viewerCount = viewerCount;
    }
  }

  tagIter = messageTags.find("msg-param-profileImageURL");
  if (tagIter != tagEnd) {
    notice.profileImageUrl = tagIter->second;
  }

  mToMainQ.push([this, notice = std::move(notice)] {
    auto user = mUser.lock();
    UserId userId = (user != nullptr) ? user->GetUserId() : 0;

    mChatCallbacks->ChatChannelRaidNoticeReceived(userId, mChannelId, notice);
  });
}

void ttv::chat::ChatChannel::HandleUnraidNotice(
  const std::string& /*message*/, const std::map<std::string, std::string>& messageTags) {
  UnraidNotice notice;

  auto tagIter = messageTags.find("system-msg");
  if (tagIter != messageTags.end()) {
    notice.systemMessage = tagIter->second;
  }

  mToMainQ.push([this, notice = std::move(notice)] {
    auto user = mUser.lock();
    UserId userId = (user != nullptr) ? user->GetUserId() : 0;

    mChatCallbacks->ChatChannelUnraidNoticeReceived(userId, mChannelId, notice);
  });
}

void ttv::chat::ChatChannel::HandleGenericNotice(
  const std::string& message, const std::map<std::string, std::string>& messageTags) {
  GenericMessageNotice notice;

  std::string username = FindOrDefault(messageTags, "login");
  ChatUserInfo userInfo;
  GenerateUserInfo(username, messageTags, userInfo);

  FindAndSet(messageTags, "id", notice.messageId);

  GenerateMessage(notice.messageInfo, mTokenizationOptions, userInfo, message, messageTags, {});

  mToMainQ.push([this, notice = std::move(notice)] {
    auto user = mUser.lock();
    UserId userId = (user != nullptr) ? user->GetUserId() : 0;

    mChatCallbacks->ChatChannelGenericNoticeReceived(userId, mChannelId, notice);
  });
}

void ttv::chat::ChatChannel::GenerateUserInfo(
  const std::string& userName, const std::map<std::string, std::string>& messageTags, ChatUserInfo& userInfo) {
  auto tagIter = messageTags.find("user-id");
  if (tagIter != messageTags.end()) {
    UserId userId = 0;
    if (ParseNum(tagIter->second, userId)) {
      userInfo.userId = userId;
    }
  }

  tagIter = messageTags.find("user-type");
  if (tagIter != messageTags.end()) {
    userInfo.userMode = ParseUserType(tagIter->second);
  }

  if (userName == mServerChannelInfo.name) {
    userInfo.userMode.broadcaster = true;
  }

  tagIter = messageTags.find("mod");
  if (tagIter != messageTags.end()) {
    if (tagIter->second == "1") {
      userInfo.userMode.moderator = true;
    }
  }

  tagIter = messageTags.find("subscriber");
  if (tagIter != messageTags.end()) {
    if (tagIter->second == "1") {
      userInfo.userMode.subscriber = true;
    }
  }

  tagIter = messageTags.find("color");
  if (tagIter != messageTags.end()) {
    Color userColor;
    if (ParseColor(tagIter->second, userColor)) {
      userInfo.nameColorARGB = userColor;
    } else {
      userInfo.nameColorARGB = GetRandomUserColor(userName);
    }
  } else {
    userInfo.nameColorARGB = GetRandomUserColor(userName);
  }

  tagIter = messageTags.find("display-name");
  if (tagIter != messageTags.end()) {
    userInfo.displayName = tagIter->second;
  }

  // Currently the only way to find if the user is a VIP is through badges
  // Should switch over to querying the ChatUserInfo from GQL on connecting to channel
  tagIter = messageTags.find("badges");
  if (tagIter != messageTags.end()) {
    if (tagIter->second.find("vip") != std::string::npos) {
      userInfo.userMode.vip = true;
    }
  }

  userInfo.userName = userName;
}

void ttv::chat::ChatChannel::GenerateMessage(MessageInfo& msg, const TokenizationOptions& tokenizationOptions,
  const ChatUserInfo& userInfo, const std::string& message, const std::map<std::string, std::string>& messageTags,
  const MessageInfo::Flags& flags) {
  msg.userId = userInfo.userId;
  msg.nameColorARGB = userInfo.nameColorARGB;
  msg.userMode = userInfo.userMode;
  msg.flags = flags;
  msg.timestamp = GetCurrentTimeAsUnixTimestamp();
  msg.numBitsSent = 0;
  msg.messageTags = messageTags;

  msg.userName = userInfo.userName;

  msg.displayName = userInfo.displayName;

  // Handle msg-id
  FindAndSet(messageTags, "msg-id", msg.messageType);

  // Handle automod flags
  std::string autoModFlagsMessageTag;
  auto iter = messageTags.find("flags");
  if (iter != messageTags.end()) {
    autoModFlagsMessageTag = iter->second;
  }

  // Handle bits
  bool isBitsMessage = false;
  iter = messageTags.find("bits");
  if (iter != messageTags.end()) {
    const std::string& bits = iter->second;

    if (ParseNum(bits, msg.numBitsSent) && msg.numBitsSent > 0) {
      isBitsMessage = true;
    }
  }

  // Handle emotes
  std::string emotesMessageTag;
  iter = messageTags.find("emotes");
  if (iter != messageTags.end()) {
    emotesMessageTag = iter->second;
  }

  // Remove bits tokenization if not marked by the backend as a message containing bits
  TokenizationOptions modifiedTokenizationOptions = tokenizationOptions;
  if (!isBitsMessage) {
    modifiedTokenizationOptions.bits = false;
  }

  std::vector<std::string> localUserNames;
  auto user = GetUser();
  if (user != nullptr) {
    localUserNames.emplace_back(user->GetUserName());
    localUserNames.emplace_back(user->GetDisplayName());
  }

  TokenizeServerMessage(message, modifiedTokenizationOptions, emotesMessageTag, autoModFlagsMessageTag,
    mBitsConfiguration, localUserNames, msg);

  // Handle badges
  iter = messageTags.find("badges");
  if (iter != messageTags.end()) {
    std::vector<std::pair<std::string, std::string>> pairs;
    ParseBadgesMessageTag(iter->second, pairs);

    std::transform(
      pairs.begin(), pairs.end(), std::back_inserter(msg.badges), [](const std::pair<std::string, std::string>& pair) {
        MessageBadge badge;
        badge.name = pair.first;
        badge.version = pair.second;

        return badge;
      });
  }

  // Respect the url hiding setting on the channel.
  if ((mChatProperties != nullptr) && mChatProperties->hide_chat_links && !IsPrivilegedUser(userInfo.userMode)) {
    for (auto& token : msg.tokens) {
      if (token->GetType() == MessageToken::Type::Url) {
        static_cast<UrlToken*>(token.get())->hidden = true;
      }
    }
  }
}

void ttv::chat::ChatChannel::GenerateLiveMessage(LiveChatMessage& msg, const TokenizationOptions& tokenizationOptions,
  const ChatUserInfo& userInfo, const std::string& message, const std::map<std::string, std::string>& messageTags,
  const MessageInfo::Flags& flags) {
  // Handle message id
  auto iter = messageTags.find("id");
  if (iter != messageTags.end()) {
    msg.messageId = iter->second;
  }

  GenerateMessage(msg.messageInfo, tokenizationOptions, userInfo, message, messageTags, flags);
}

void ttv::chat::ChatChannel::OnPrivateMessageReceived(ChatConnection* /*source*/, const std::string& username,
  const std::string& message, const std::map<std::string, std::string>& messageTags, bool action) {
  MessageInfo::Flags flags;
  flags.action = action;

  HandleMessageReceived(username, message, messageTags, flags);
}

void ttv::chat::ChatChannel::OnUserNoticeReceived(ttv::chat::ChatConnection* /*source*/, const std::string& message,
  const std::map<std::string, std::string>& messageTags) {
  HandleUserNotice(message, messageTags);
}

void ttv::chat::ChatChannel::OnUserStateChanged(
  ttv::chat::ChatConnection* /*source*/, const std::map<std::string, std::string>& messageTags) {
  CHAT_THREAD_ONLY

  auto user = mUser.lock();
  if (user == nullptr) {
    return;
  }

  ChatUserInfo newLocalUserInfo;
  GenerateUserInfo(user->GetUserName(), messageTags, newLocalUserInfo);

  // The server doesn't provide all the info on userstate for some reason, so fill it out here.
  newLocalUserInfo.userId = user->GetUserId();
  newLocalUserInfo.displayName = user->GetDisplayName();

  UpdateLocalUserInfo(newLocalUserInfo);
}

void ttv::chat::ChatChannel::UpdateLocalUserInfo(const ChatUserInfo& newUserInfo) {
  CHAT_THREAD_ONLY

  if (mServerLocalUserInfo != newUserInfo) {
    mServerLocalUserInfo = newUserInfo;

    mToMainQ.push([this, newUserInfo]() {
      ttv::trace::Message("Chat", MessageLevel::Debug, "LocalUserChangedClientMessage");

      mClientLocalUserInfo = newUserInfo;

      auto userLambda = mUser.lock();
      UserId userId = (userLambda != nullptr) ? userLambda->GetUserId() : 0;

      // fire the notification
      mChatCallbacks->ChatChannelLocalUserChanged(userId, mChannelId, mClientLocalUserInfo);
    });
  }
}

void ttv::chat::ChatChannel::OnChatRestrictionsChanged(
  ChatConnection* /*source*/, const ChatChannelRestrictions& restrictions) {
  CHAT_THREAD_ONLY

  mServerChatRestrictions = restrictions;
  UpdateRestriction();

  if (mApplySlowMode) {
    mSlowModeTimer.Set(mServerChatRestrictions.slowModeDuration * 1000);
  }

  // notify the main thread
  mToMainQ.push([this, restrictions = mServerChatRestrictions] { SetClientChatRestrictions(restrictions); });
}

void ttv::chat::ChatChannel::OnPermanentBanChanged(ChatConnection* /*source*/, bool banned) {
  CHAT_THREAD_ONLY

  mServerLocalUserInfo.userMode.banned = banned;
}

void ttv::chat::ChatChannel::OnTemporaryBanChanged(
  ChatConnection* /*source*/, bool temporarilyBanned, uint32_t timeout) {
  CHAT_THREAD_ONLY

  if (temporarilyBanned) {
    mTimeoutTimer.Set(timeout * 1000);
  } else {
    mTimeoutTimer.Clear();
  }
}

void ttv::chat::ChatChannel::OnBadgesChanged(
  ChatConnection* /*source*/, const std::string& username, const std::string& badgesMessageTag) {
  CHAT_THREAD_ONLY

  auto user = mUser.lock();
  if ((user != nullptr) && (user->GetUserName() == username)) {
    // notify the main thread
    mToMainQ.push([this, badges = user->GetComponentContainer()->GetComponent<ChatUserBadges>(), badgesMessageTag]() {
      ttv::trace::Message("Chat", MessageLevel::Debug, "LocalUserBadgesChangedClientMessage");

      if (badges != nullptr) {
        badges->SetBadgesMessageTag(mChannelId, badgesMessageTag);
      }
    });
  }
}

void ttv::chat::ChatChannel::OnHostTargetChanged(
  ChatConnection* /*source*/, const std::string& targetChannel, uint32_t numViewers) {
  CHAT_THREAD_ONLY

  // notify the main thread
  mToMainQ.push([this, targetChannel, numViewers]() {
    ttv::trace::Message("Chat", MessageLevel::Debug, "HostTargetClientMessage");

    auto user = mUser.lock();
    UserId userId = (user != nullptr) ? user->GetUserId() : 0;

    // fire the notification
    mChatCallbacks->ChatChannelHostTargetChanged(userId, mChannelId, targetChannel, numViewers);
  });
}

void ttv::chat::ChatChannel::OnNoticeReceived(
  ChatConnection* /*source*/, const std::string& id, const std::map<std::string, std::string>& params) {
  CHAT_THREAD_ONLY

  // notify the main thread
  mToMainQ.push([this, id, params]() {
    ttv::trace::Message("Chat", MessageLevel::Debug, "NoticeClientMessage");

    auto user = mUser.lock();
    UserId userId = (user != nullptr) ? user->GetUserId() : 0;

    // fire the notification
    mChatCallbacks->ChatChannelNoticeReceived(userId, mChannelId, id, params);
  });
}

void ttv::chat::ChatChannel::OnIgnoreChanged(
  ChatConnection* /*source*/, const std::string& blockUserName, bool ignore) {
  CHAT_THREAD_ONLY

  // notify the main thread
  mToMainQ.push([this, localUserName = mChatConnection->GetLocalUserName(), blockUserName, ignore]() {
    ttv::trace::Message("Chat", MessageLevel::Debug, "IgnoreClientMessage");

    // Update the user block list
    auto user = mUser.lock();
    if ((user != nullptr) && (user->GetUserName() == localUserName)) {
      std::shared_ptr<ChatUserBlockList> blockList = user->GetComponentContainer()->GetComponent<ChatUserBlockList>();
      if (blockList != nullptr) {
        if (ignore) {
          blockList->BlockUser(blockUserName, "", false);
        } else {
          blockList->UnblockUser(blockUserName);
        }
      }
    }
  });
}

void ttv::chat::ChatChannel::OnMessageDeleted(ChatConnection* /*source*/, std::string&& deletedMessageId,
  std::string&& senderLoginName, std::string&& deletedMessageContent) {
  CHAT_THREAD_ONLY

  for (auto& message : mUserMessageBatch) {
    if (message.messageId == deletedMessageId) {
      message.messageInfo.flags.ignored = true;
    }
  }

  for (auto& message : mDelayedMessageBatch) {
    if (message.messageId == deletedMessageId) {
      message.messageInfo.flags.ignored = true;
    }
  }

  // notify the main thread
  mToMainQ.push([this, messageId = std::move(deletedMessageId), loginName = std::move(senderLoginName),
                  messageContent = std::move(deletedMessageContent)]() mutable {
    ttv::trace::Message("Chat", MessageLevel::Debug, "OnMessageDeleted");

    UserId userId = 0;
    if (auto user = mUser.lock()) {
      userId = user->GetUserId();
    }

    mChatCallbacks->ChatChannelMessageDeleted(
      userId, mChannelId, std::move(messageId), std::move(loginName), std::move(messageContent));
  });
}

TTV_ErrorCode ttv::chat::ChatChannel::SubscribeTopics() {
  if (mPubSubTopicListenerHelper == nullptr || mAnonymous) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  for (const auto& pubsubTopic : {mModeratorActionsPubSubTopic, mChatroomUserV1PubSubTopic}) {
    if (!mPubSubTopicListenerHelper->ContainsTopic(pubsubTopic)) {
      auto subscribedEc = mPubSubTopicListenerHelper->Subscribe(pubsubTopic);
      if (TTV_FAILED(subscribedEc)) {
        ttv::trace::Message(kLoggerName, MessageLevel::Error, "Failed to subscribe to topic: %s", pubsubTopic.c_str());
        ec = subscribedEc;
      }
    }
  }

  return ec;
}

void ttv::chat::ChatChannel::OnTopicSubscribeStateChanged(
  const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "ChannelListener SubscribeStateChanged: %s %s", topic.c_str(),
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::ChatChannel::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    ttv::trace::Message(kLoggerName, MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mChatroomUserV1PubSubTopic) {
    auto user = mUser.lock();
    if (user == nullptr) {
      return;
    }

    std::string type;
    if (!ParseString(msg, "type", type)) {
      ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message missing type, dropping");
      return;
    }

    const auto& jData = msg["data"];
    if (jData.isNull() || !jData.isObject()) {
      ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message missing data, dropping");
      return;
    }

    if (type == "user_moderation_action") {
      std::string action;
      if (!ParseString(jData, "action", action)) {
        return;
      }

      UserId userId;
      const auto& jUserId = jData["target_id"];
      ParseUserId(jUserId, userId);
      if (userId != user->GetUserId()) {
        return;
      }

      ChannelId channelId;
      const auto& jChannelId = jData["channel_id"];
      if (!ParseChannelId(jChannelId, channelId)) {
        return;
      }

      if (channelId != mChannelId) {
        return;
      }

      if (action == "ban") {
        mToChatQ.push([this]() {
          mTimeoutTimer.Clear();
          mServerLocalUserInfo.userMode.banned = true;
        });
      } else if (action == "unban") {
        // command: /untimeout {{username}} will also emit an unban update
        mToChatQ.push([this]() {
          mTimeoutTimer.Clear();
          mServerLocalUserInfo.userMode.banned = false;
        });
      } else {
        return;
      }
    }
  } else if (topic == mModeratorActionsPubSubTopic) {
    auto user = mUser.lock();
    if (user == nullptr) {
      return;
    }

    const auto& jData = msg["data"];
    if (jData.isNull() || !jData.isObject()) {
      ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message missing data, dropping");
      return;
    }

    const auto& jType = msg["type"];
    if (jType.isString()) {
      ChatUserInfo newUserInfo = mClientLocalUserInfo;
      std::string type = jType.asString();

      if (type != "moderation_action") {
        // User role change events.
        if (type == "moderator_added") {
          newUserInfo.userMode.moderator = true;
          newUserInfo.userMode.vip = false;
        } else if (type == "moderator_removed") {
          newUserInfo.userMode.moderator = false;
        } else if (type == "vip_added") {
          newUserInfo.userMode.vip = true;
          newUserInfo.userMode.moderator = false;
        } else if (type == "vip_removed") {
          newUserInfo.userMode.vip = false;
        } else {
          return;
        }

        mToChatQ.push([this, newUserInfo]() { UpdateLocalUserInfo(newUserInfo); });
        return;
      }

      const auto& jModerationType = jData["type"];
      if (!jModerationType.isString()) {
        ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message missing type, dropping");
        return;
      }

      const auto& jModerationAction = jData["moderation_action"];
      if (!jModerationAction.isString()) {
        ttv::trace::Message(
          kLoggerName, MessageLevel::Error, "Pub sub message missing moderation action type, dropping");
        return;
      }

      std::string moderationAction = jModerationAction.asString();

      if (jModerationType.asString() == "chat_targeted_login_moderation") {
        if (moderationAction == "automod_message_rejected") {
          mChatCallbacks->AutoModCaughtSentMessage(user->GetUserId(), mChannelId);
        } else if (moderationAction == "automod_message_denied") {
          mChatCallbacks->AutoModDeniedSentMessage(user->GetUserId(), mChannelId);
        } else if (moderationAction == "automod_message_approved") {
          mChatCallbacks->AutoModApprovedSentMessage(user->GetUserId(), mChannelId);
        } else if (moderationAction == "automod_cheer_message_denied") {
          mChatCallbacks->AutoModDeniedSentCheer(user->GetUserId(), mChannelId);
        } else if (moderationAction == "automod_cheer_message_timeout") {
          mChatCallbacks->AutoModTimedOutSentCheer(user->GetUserId(), mChannelId);
        } else if (moderationAction == "timeout") {
          const auto& jArgs = jData["args"];
          if (!jArgs.isNonNullArray()) {
            return;
          }
          uint32_t timeoutDurationSeconds = 0;
          if (jArgs.size() != 3 || !jArgs[1].isString() || !jArgs[2].isString() ||
              !ParseNum(jArgs[1].asString(), timeoutDurationSeconds)) {
            return;
          }
          mToChatQ.push([this, timeoutDurationSeconds]() { mTimeoutTimer.Set(timeoutDurationSeconds * 1000); });
        }
      } else if (jModerationType.asString() == "chat_login_moderation") {
        const auto& jArgs = jData["args"];
        if (!jArgs.isNonNullArray()) {
          return;
        }

        UserId targetUserId = 0;
        const auto& jTargetUserId = jData["target_user_id"];
        if (!jTargetUserId.isString() || !ParseNum(jTargetUserId.asString(), targetUserId)) {
          return;
        }

        const auto& jMessageId = jData["msg_id"];
        if (!jMessageId.isString()) {
          return;
        }

        if (moderationAction == "automod_rejected") {
          if (jArgs.size() != 3 || !jArgs[0].isString() || !jArgs[1].isString() || !jArgs[2].isString()) {
            return;
          }

          mChatCallbacks->AutoModCaughtMessageForMods(user->GetUserId(), mChannelId, jMessageId.asString(),
            jArgs[1].asString(), targetUserId, jArgs[0].asString(), jArgs[2].asString());
        } else if (moderationAction == "automod_cheer_rejected") {
          if (jArgs.size() != 3 || !jArgs[0].isString() || !jArgs[1].isString() || !jArgs[2].isString()) {
            return;
          }

          mChatCallbacks->AutoModCaughtCheerForMods(user->GetUserId(), mChannelId, jMessageId.asString(),
            jArgs[1].asString(), targetUserId, jArgs[0].asString(), jArgs[2].asString());
        } else {
          UserId moderatorId = 0;
          const auto& jModeratorId = jData["created_by_user_id"];
          if (!jModeratorId.isString() || !ParseNum(jModeratorId.asString(), moderatorId)) {
            return;
          }

          const auto& jModeratorName = jData["created_by"];
          if (!jModeratorName.isString()) {
            return;
          }

          if (moderationAction == "denied_automod_message") {
            mChatCallbacks->AutoModMessageDeniedByMod(
              user->GetUserId(), mChannelId, jMessageId.asString(), moderatorId, jModeratorName.asString());
          } else if (moderationAction == "approved_automod_message") {
            mChatCallbacks->AutoModMessageApprovedByMod(
              user->GetUserId(), mChannelId, jMessageId.asString(), moderatorId, jModeratorName.asString());
          } else {
            ModerationActionInfo modActionInfo;

            if (jArgs.size() < 1 || !jArgs[0].isString()) {
              return;
            }

            modActionInfo.targetName = jArgs[0].asString();
            modActionInfo.targetId = targetUserId;
            modActionInfo.moderatorName = jModeratorName.asString();
            modActionInfo.moderatorId = moderatorId;

            if (moderationAction == "ban") {
              if (jArgs.size() != 2 || !jArgs[1].isString()) {
                return;
              }

              mChatCallbacks->ChatChannelModNoticeUserBanned(
                user->GetUserId(), mChannelId, std::move(modActionInfo), jArgs[1].asString());
            } else if (moderationAction == "timeout") {
              uint32_t timeoutDurationSeconds = 0;
              if (jArgs.size() != 3 || !jArgs[1].isString() || !jArgs[2].isString() ||
                  !ParseNum(jArgs[1].asString(), timeoutDurationSeconds)) {
                return;
              }

              mChatCallbacks->ChatChannelModNoticeUserTimedOut(
                user->GetUserId(), mChannelId, std::move(modActionInfo), timeoutDurationSeconds, jArgs[2].asString());
            } else if (moderationAction == "unban") {
              mChatCallbacks->ChatChannelModNoticeUserUnbanned(user->GetUserId(), mChannelId, std::move(modActionInfo));
            } else if (moderationAction == "untimeout") {
              mChatCallbacks->ChatChannelModNoticeUserUntimedOut(
                user->GetUserId(), mChannelId, std::move(modActionInfo));
            } else if (moderationAction == "delete") {
              std::string messageId;
              std::string messageContent;

              if (jArgs.size() != 3 || !jArgs[1].isString() || !jArgs[2].isString()) {
                return;
              }

              mChatCallbacks->ChatChannelModNoticeMessageDeleted(
                user->GetUserId(), mChannelId, std::move(modActionInfo), jArgs[2].asString(), jArgs[1].asString());
            }
          }
        }
      } else if (jModerationType.asString() == "chat_channel_moderation") {
        UserId moderatorId = 0;
        const auto& jModeratorId = jData["created_by_user_id"];
        if (!jModeratorId.isString() || !ParseNum(jModeratorId.asString(), moderatorId)) {
          return;
        }

        const auto& jModeratorName = jData["created_by"];
        if (!jModeratorName.isString()) {
          return;
        }

        if (moderationAction == "clear") {
          mChatCallbacks->ChatChannelModNoticeClearChat(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        } else if (moderationAction == "emoteonly") {
          mChatCallbacks->ChatChannelModNoticeEmoteOnly(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        } else if (moderationAction == "emoteonlyoff") {
          mChatCallbacks->ChatChannelModNoticeEmoteOnlyOff(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        } else if (moderationAction == "followers") {
          uint32_t minFollowingDurationMinutes = 0;
          const auto& jArgs = jData["args"];
          if (jArgs.isNonNullArray() && jArgs.size() > 0 && jArgs[0].isString()) {
            ParseNum(jArgs[0].asString(), minFollowingDurationMinutes);
          }

          mChatCallbacks->ChatChannelModNoticeFollowersOnly(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString(), minFollowingDurationMinutes);
        } else if (moderationAction == "followersoff") {
          mChatCallbacks->ChatChannelModNoticeFollowersOnlyOff(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        } else if (moderationAction == "r9kbeta") {
          mChatCallbacks->ChatChannelModNoticeR9K(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        } else if (moderationAction == "r9kbetaoff") {
          mChatCallbacks->ChatChannelModNoticeR9KOff(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        } else if (moderationAction == "slow") {
          uint32_t slowModeDurationSeconds = 0;
          const auto& jArgs = jData["args"];
          if (!jArgs.isNonNullArray() || jArgs.empty() || !jArgs[0].isString() ||
              !ParseNum(jArgs[0].asString(), slowModeDurationSeconds)) {
            return;
          }

          if (static_cast<int32_t>(slowModeDurationSeconds) < 0) {
            slowModeDurationSeconds = 0;
          }

          mChatCallbacks->ChatChannelModNoticeSlow(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString(), slowModeDurationSeconds);
        } else if (moderationAction == "slowoff") {
          mChatCallbacks->ChatChannelModNoticeSlowOff(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        } else if (moderationAction == "subscribers") {
          mChatCallbacks->ChatChannelModNoticeSubsOnly(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        } else if (moderationAction == "subscribersoff") {
          mChatCallbacks->ChatChannelModNoticeSubsOnlyOff(
            user->GetUserId(), mChannelId, moderatorId, jModeratorName.asString());
        }
      } else {
        ttv::trace::Message(kLoggerName, MessageLevel::Error, "Unknown pub sub message type, dropping");
      }
    }
  }
}

ttv::chat::ChatChannel::PubSubTopicListener::PubSubTopicListener(ChatChannel* owner) : mOwner(owner) {}

void ttv::chat::ChatChannel::PubSubTopicListener::OnTopicSubscribeStateChanged(
  PubSubClient* /*source*/, const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) {
  mOwner->OnTopicSubscribeStateChanged(topic, state, ec);
}

void ttv::chat::ChatChannel::PubSubTopicListener::OnTopicMessageReceived(
  PubSubClient* /*source*/, const std::string& topic, const ttv::json::Value& msg) {
  mOwner->OnTopicMessageReceived(topic, msg);
}

void ttv::chat::ChatChannel::PubSubTopicListener::OnTopicListenerRemoved(
  PubSubClient* /*source*/, const std::string& /*topic*/, TTV_ErrorCode /*ec*/) {
  // NOTE: The listener is only removed if we requested it to be removed or pubsub is shutting down
}
