/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatchannelproperties.h"

#include "twitchsdk/chat/internal/graphql/generated/fetchchatchannelsettingsinfo.h"
#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"
#include "twitchsdk/core/task/graphqltask.h"

#include <memory>

namespace {
const char* kLoggerName = "ChatChannelProperties";
const char* kTopicPrefix = "stream-chat-room-v1.";
constexpr uint64_t kRetryTimeMsec = 5000;
constexpr uint64_t kRetryTimeJitterMsec = 500;
}  // namespace

ttv::chat::ChatChannelProperties::HostingInfo::HostingInfo()
    : hostChannelId(0), targetChannelId(0), previousTargetChannelId(0), numViewers(0) {}

ttv::chat::ChatChannelProperties::ChatChannelProperties(const std::shared_ptr<User>& user, ChannelId channelId)
    : PubSubComponent(user),
      mPubSubTopic(kTopicPrefix + std::to_string(channelId)),
      mChannelId(channelId),
      mFetchRetryTimer(kRetryTimeMsec, kRetryTimeJitterMsec),
      areChatRestrictionsInitialized(false) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::ChatChannelProperties::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::ChatChannelProperties::Initialize() {
  auto ec = PubSubComponent::Initialize();

  TTV_RETURN_ON_ERROR(ec);

  HandleInitChatRestrictions();
  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatChannelProperties::Update() {
  PubSubComponent::Update();

  if (mState != State::Initialized) {
    return;
  }

  if (mFetchRetryTimer.CheckNextRetry()) {
    HandleInitChatRestrictions();
  }
}

TTV_ErrorCode ttv::chat::ChatChannelProperties::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatChannelProperties::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "ChatChannelProperties SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::ChatChannelProperties::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) {
  if (jVal.isNull() || !jVal.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    std::string type;
    if (!ParseString(jVal, "type", type)) {
      Log(MessageLevel::Error, "Couldn't find pubsub message type, dropping");
      return;
    }

    const auto& jData = jVal["data"];
    if (jData.isNull() || !jData.isObject()) {
      Log(MessageLevel::Error, "Pub sub message missing data, dropping");
      return;
    }

    if (type == "updated_channel_chat_property") {
      bool ritualsEnabled;
      const auto& jRitualsEnabled = jData["is_rituals_enabled"];
      if (!ttv::json::ToObject(jRitualsEnabled, ritualsEnabled)) {
        return;
      }

      if (mListener != nullptr) {
        mListener->RitualsEnabled(ritualsEnabled);
      }
    } else if (type == "host_target_change_v2") {
      HostingInfo hostingInfo;

      if (!ttv::json::ObjectSchema<json::description::PubSubHostingInfo>::Parse(jData, hostingInfo)) {
        return;
      }

      if (mListener != nullptr) {
        if (hostingInfo.hostChannelId == mChannelId) {
          mListener->OutgoingHostChanged(mChannelId, hostingInfo.previousTargetChannelId, hostingInfo.targetChannelId,
            std::move(hostingInfo.targetChannelName), hostingInfo.numViewers);
        } else if (hostingInfo.targetChannelId == mChannelId) {
          mListener->IncomingHostStarted(
            mChannelId, hostingInfo.hostChannelId, std::move(hostingInfo.hostChannelName), hostingInfo.numViewers);
        } else if (hostingInfo.previousTargetChannelId == mChannelId) {
          mListener->IncomingHostEnded(mChannelId, hostingInfo.hostChannelId, std::move(hostingInfo.hostChannelName));
        }
      }
    } else if (type == "extension_message") {
      ExtensionMessage extensionMessage;

      if (!ttv::json::ToObject(jData, extensionMessage)) {
        return;
      }

      if (mListener != nullptr) {
        mListener->ExtensionMessageReceived(std::move(extensionMessage));
      }
    } else if (type == "updated_room") {
      ChatChannelRestrictions chatChannelRestrictions;

      const auto& modes = jData["room"]["modes"];

      if (modes.isNull() || !modes.isObject()) {
        return;
      }

      if (!ttv::json::ToObject(modes, chatChannelRestrictions)) {
        return;
      }

      // values less than or equal to zero should treated as slow mode disabled
      if (static_cast<int32_t>(chatChannelRestrictions.slowModeDuration) < 0) {
        chatChannelRestrictions.slowModeDuration = 0;
      }
      // slowMode and followersOnly both depend on the duration variables
      chatChannelRestrictions.slowMode = (chatChannelRestrictions.slowModeDuration != 0) ? true : false;
      chatChannelRestrictions.followersOnly = !modes["followers_only_duration_minutes"].isNull();

      if (mListener != nullptr) {
        areChatRestrictionsInitialized = true;
        mListener->ChatChannelRestrictionsReceived(std::move(chatChannelRestrictions));
      }
    } else {
      Log(MessageLevel::Error, "Unrecognized pub-sub message type (%s), dropping", type.c_str());
      return;
    }
  }
}

void ttv::chat::ChatChannelProperties::HandleInitChatRestrictions() {
  if (areChatRestrictionsInitialized) {
    mFetchRetryTimer.Clear();
    return;
  }

  graphql::FetchChatSettingsQueryInfo::InputParams queryInput;
  queryInput.channelId = std::to_string(mChannelId);

  if (auto user = mUser.lock()) {
    queryInput.authToken = user->GetOAuthToken()->GetToken();
  }

  // the task will handle three cases
  // 1. chat restrictions already received, task should use the current chat
  //    restrictions and discard the one received on this task and stop retries
  // 2. request is not successful, schedule a retry
  // 3. request is successful, stop timer and emit chat restrictions
  auto callback = [this](GraphQLTask<graphql::FetchChatSettingsQueryInfo>* source,
                    Result<graphql::FetchChatSettingsQueryInfo::PayloadType>&& result) {
    CompleteTask(source);

    // pubsub event received, initialization complete
    if (areChatRestrictionsInitialized) {
      mFetchRetryTimer.Clear();
      return;
    }

    if (result.IsError()) {
      Log(MessageLevel::Warning, "Could not initialize ChatRestrictions %s", CoreErrorToString(result.GetErrorCode()));
      mFetchRetryTimer.ScheduleNextRetry();
      return;
    }

    auto chatSettings = result.GetResult().chatSettings.ValueOrDefault({});

    ChatChannelRestrictions chatChannelRestrictions;

    chatChannelRestrictions.followersDuration =
      static_cast<uint32_t>(chatSettings.followersOnlyDurationMinutes.ValueOrDefault(0));
    chatChannelRestrictions.followersOnly = chatSettings.followersOnlyDurationMinutes.HasValue();

    // values less than or equal to zero should treated as slow mode disabled
    auto slowModeDurationSeconds = chatSettings.slowModeDurationSeconds.ValueOrDefault(0);
    if (slowModeDurationSeconds < 0) {
      slowModeDurationSeconds = 0;
    }

    chatChannelRestrictions.slowModeDuration = static_cast<uint32_t>(slowModeDurationSeconds);
    chatChannelRestrictions.slowMode = (chatChannelRestrictions.slowModeDuration > 0);

    chatChannelRestrictions.emoteOnly = chatSettings.isEmoteOnlyModeEnabled;
    chatChannelRestrictions.verifiedOnly = chatSettings.requireVerifiedAccount;
    chatChannelRestrictions.subscribersOnly = chatSettings.isSubscribersOnlyModeEnabled;
    chatChannelRestrictions.r9k = chatSettings.isUniqueChatModeEnabled;

    if (mListener != nullptr) {
      mListener->ChatChannelRestrictionsReceived(std::move(chatChannelRestrictions));
      areChatRestrictionsInitialized = true;
      mFetchRetryTimer.Clear();
    }
  };

  auto initChatRestrictionsTask =
    std::make_shared<GraphQLTask<graphql::FetchChatSettingsQueryInfo>>(std::move(queryInput), std::move(callback));

  StartTask(initChatRestrictionsTask);
}
