/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatchannelset.h"

#include "twitchsdk/chat/internal/chatchannel.h"
#include "twitchsdk/chat/internal/chatuserblocklist.h"
#include "twitchsdk/chat/internal/defaultchatobjectfactory.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/types/memorytypes.h"
#include "twitchsdk/core/user/user.h"

namespace {
const char* kLoggerName = "ChatChannelSet";
}

ttv::chat::ChatChannelSet::ChatChannelSet(const std::shared_ptr<User>& user)
    : ttv::UserComponent(user), mUserMessageFlushInterval(250) {
  TTV_ASSERT(user != nullptr);

  SetChatObjectFactory(nullptr);

  mInternalChannelListener =
    std::static_pointer_cast<IChatChannelListener>(std::make_shared<InternalChannelListener>(this));
}

ttv::chat::ChatChannelSet::~ChatChannelSet() {}

TTV_ErrorCode ttv::chat::ChatChannelSet::Shutdown() {
  TTV_ErrorCode ec = UserComponent::Shutdown();
  if (TTV_SUCCEEDED(ec)) {
    for (auto iter = mChannels.begin(); iter != mChannels.end(); ++iter) {
      auto entry = iter->second;

      auto disconnectError = entry->channel->Disconnect();

      if (TTV_FAILED(disconnectError) && (disconnectError != TTV_EC_CHAT_LEAVING_CHANNEL) && TTV_SUCCEEDED(ec)) {
        ec = disconnectError;
      }
    }
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatChannelSet::LookupChannel(ChannelId channelId, std::shared_ptr<ChatChannel>& channel) {
  auto iter = mChannels.find(channelId);
  if (iter == mChannels.end()) {
    return TTV_EC_CHAT_NOT_IN_CHANNEL;
  }

  channel = iter->second->channel;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatChannelSet::Connect(
  ChannelId channelId, const std::shared_ptr<IChatChannelListener>& chatCallbacks) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(chatCallbacks, TTV_EC_INVALID_ARG);

  if (channelId == 0) {
    return TTV_EC_INVALID_CHANNEL_ID;
  }

  // create / find the channel
  auto iter = mChannels.find(channelId);
  std::shared_ptr<ChannelEntry> entry;

  if (iter == mChannels.end()) {
    entry = CreateChannel(channelId);
    entry->channelListener = chatCallbacks;
  } else {
    entry = iter->second;

    // you must use the same callback listener if the channel already exists
    if (chatCallbacks != entry->channelListener) {
      return TTV_EC_INVALID_CALLBACK;
    }
  }

  // join the channel
  TTV_ErrorCode ec = entry->channel->Connect();

  return ec;
}

TTV_ErrorCode ttv::chat::ChatChannelSet::Disconnect(ChannelId channelId) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<ChatChannel> channel;
  TTV_ErrorCode ec = LookupChannel(channelId, channel);

  if (TTV_SUCCEEDED(ec)) {
    // leave the channel
    ec = channel->Disconnect();

    // wait for the status callback to indicate a disconnection before removing the channel from mChannels
  }

  return ec;
}

std::shared_ptr<ttv::chat::ChatChannelSet::ChannelEntry> ttv::chat::ChatChannelSet::CreateChannel(ChannelId channelId) {
  std::shared_ptr<ChannelEntry> entry = std::make_shared<ChannelEntry>();

  std::shared_ptr<IChatChannelListener> channelListener =
    std::static_pointer_cast<IChatChannelListener>(mInternalChannelListener);
  entry->channel = std::make_shared<ChatChannel>(mUser.lock(), channelId, channelListener, mTaskRunner);
  entry->channel->SetChatObjectFactory(mChatObjectFactory);
  entry->channel->SetMessageFlushInterval(mUserMessageFlushInterval);
  entry->channel->SetTokenizationOptions(mTokenizationOptions);
  entry->channel->SetChannelRepository(mChannelRepository);
  entry->channel->SetSettingRepository(mSettingRepository);
  entry->channel->SetBitsConfigRepository(mBitsConfigRepository);

  mChannels[channelId] = entry;

  return entry;
}

TTV_ErrorCode ttv::chat::ChatChannelSet::SendChatMessage(ChannelId channelId, const std::string& message) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_CHANNEL_ID);
  TTV_RETURN_ON_EMPTY_STDSTRING(message, TTV_EC_INVALID_ARG);

  std::shared_ptr<ChatChannel> channel;

  auto iter = mChannels.find(channelId);
  if (iter == mChannels.end()) {
    return TTV_EC_CHAT_NOT_IN_CHANNEL;
  }

  channel = iter->second->channel;

  // Send the message to the default channel for now
  return channel->SendChatMessage(message);
}

TTV_ErrorCode ttv::chat::ChatChannelSet::FetchUserList(ChannelId channelId, const FetchUserListCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_CHANNEL_ID);

  std::shared_ptr<ChatChannel> channel;

  auto iter = mChannels.find(channelId);
  if (iter == mChannels.end()) {
    return TTV_EC_CHAT_NOT_IN_CHANNEL;
  }

  channel = iter->second->channel;

  return channel->FetchUserList(callback);
}

uint64_t ttv::chat::ChatChannelSet::GetRemainingSlowModeTime(ChannelId channelId) {
  TTV_RETURN_ON_SAME(channelId, 0, TTV_EC_INVALID_CHANNEL_ID);

  std::shared_ptr<ChatChannel> channel;

  auto iter = mChannels.find(channelId);
  if (iter == mChannels.end()) {
    return TTV_EC_CHAT_NOT_IN_CHANNEL;
  }

  channel = iter->second->channel;

  return channel->GetRemainingSlowModeTime();
}

TTV_ErrorCode ttv::chat::ChatChannelSet::SetConnectTrackingStartTime(ChannelId channelId, uint64_t startMilliseconds) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  if (channelId == 0) {
    return TTV_EC_INVALID_CHANNEL_ID;
  }

  // find the channel
  auto iter = mChannels.find(channelId);
  std::shared_ptr<ChannelEntry> entry;

  if (iter == mChannels.end()) {
    return TTV_EC_INVALID_CHANNEL_ID;
  } else {
    entry = iter->second;
    entry->channel->SetConnectTrackingStartTime(startMilliseconds);
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatChannelSet::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  FlushChannelEvents();

  UserComponent::Update();
}

TTV_ErrorCode ttv::chat::ChatChannelSet::FlushChannelEvents() {
  // Flush events for all channels
  TTV_ErrorCode errorFound = TTV_EC_SUCCESS;

  // Copy the channel list in case a callback modifies it
  auto channels = mChannels;

  for (auto iter = channels.begin(); iter != channels.end(); ++iter) {
    auto channel = iter->second->channel;

    TTV_ErrorCode ret = channel->FlushClientEvents();
    if (TTV_FAILED(ret) && TTV_SUCCEEDED(errorFound)) {
      errorFound = ret;
    }
  }

  return errorFound;
}

bool ttv::chat::ChatChannelSet::CheckShutdown() {
  if (!UserComponent::CheckShutdown()) {
    return false;
  }

  // All channels must be shutdown
  for (auto iter = mChannels.begin(); iter != mChannels.end();) {
    auto channel = iter->second->channel;

    if (channel->IsShutdown()) {
      iter = mChannels.erase(iter);
    } else {
      return false;
    }
  }

  return true;
}

void ttv::chat::ChatChannelSet::CompleteShutdown() {
  mChannels.clear();

  UserComponent::CompleteShutdown();
}

void ttv::chat::ChatChannelSet::ChatChannelStateChanged(
  UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec) {
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  auto listener = iter->second->channelListener;

  switch (state) {
    case ChatChannelState::Disconnected:
      mChannels.erase(iter);
      break;
    default:
      break;
  }

  if (listener != nullptr) {
    listener->ChatChannelStateChanged(userId, channelId, state, ec);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelInfoChanged(
  UserId userId, ChannelId channelId, const ChatChannelInfo& info) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelInfoChanged(userId, channelId, info);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelRestrictionsChanged(
  UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelRestrictionsChanged(userId, channelId, restrictions);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelLocalUserChanged(
  UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelLocalUserChanged(userId, channelId, userInfo);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelMessagesReceived(
  UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) {
  // pass the callback to the client
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelMessagesReceived(userId, channelId, messageList);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelSubscriptionNoticeReceived(
  UserId userId, ChannelId channelId, const ttv::chat::SubscriptionNotice& notice) {
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelSubscriptionNoticeReceived(userId, channelId, notice);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelFirstTimeChatterNoticeReceived(
  UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) {
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelFirstTimeChatterNoticeReceived(userId, channelId, notice);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelRaidNoticeReceived(
  UserId userId, ChannelId channelId, const ttv::chat::RaidNotice& notice) {
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelRaidNoticeReceived(userId, channelId, notice);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelUnraidNoticeReceived(
  UserId userId, ChannelId channelId, const ttv::chat::UnraidNotice& notice) {
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelUnraidNoticeReceived(userId, channelId, notice);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelGenericNoticeReceived(
  UserId userId, ChannelId channelId, const ttv::chat::GenericMessageNotice& notice) {
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelGenericNoticeReceived(userId, channelId, notice);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelMessagesCleared(UserId userId, ChannelId channelId) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelMessagesCleared(userId, channelId);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelUserMessagesCleared(UserId userId, ChannelId channelId, UserId clearUserId) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelUserMessagesCleared(userId, channelId, clearUserId);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeUserTimedOut(UserId userId, ChannelId channelId,
  ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeUserTimedOut(
      userId, channelId, std::move(modActionInfo), timeoutDurationSeconds, std::move(reason));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeUserBanned(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeUserBanned(
      userId, channelId, std::move(modActionInfo), std::move(reason));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeUserUntimedOut(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeUserUntimedOut(userId, channelId, std::move(modActionInfo));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeUserUnbanned(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeUserUnbanned(userId, channelId, std::move(modActionInfo));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeMessageDeleted(UserId userId, ChannelId channelId,
  ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeMessageDeleted(
      userId, channelId, std::move(modActionInfo), std::move(messageId), std::move(message));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeClearChat(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeClearChat(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeEmoteOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeEmoteOnly(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeEmoteOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeEmoteOnlyOff(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeFollowersOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t minimumFollowingDurationMinutes) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeFollowersOnly(
      userId, channelId, modId, std::move(modName), minimumFollowingDurationMinutes);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeFollowersOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeFollowersOnlyOff(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeR9K(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeR9K(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeR9KOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeR9KOff(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeSlow(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t slowModeDurationSeconds) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeSlow(
      userId, channelId, modId, std::move(modName), slowModeDurationSeconds);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeSlowOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeSlowOff(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeSubsOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeSubsOnly(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelModNoticeSubsOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelModNoticeSubsOnlyOff(userId, channelId, modId, std::move(modName));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelMessageDeleted(UserId userId, ChannelId channelId, std::string&& messageId,
  std::string&& senderLoginName, std::string&& messageContent) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelMessageDeleted(
      userId, channelId, std::move(messageId), std::move(senderLoginName), std::move(messageContent));
  }
}

void ttv::chat::ChatChannelSet::ChatChannelHostTargetChanged(
  UserId userId, ChannelId channelId, const std::string& targetChannelName, uint32_t numViewers) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelHostTargetChanged(userId, channelId, targetChannelName, numViewers);
  }
}

void ttv::chat::ChatChannelSet::ChatChannelNoticeReceived(
  UserId userId, ChannelId channelId, const std::string& id, const std::map<std::string, std::string>& params) {
  // pass the callback to the client
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->ChatChannelNoticeReceived(userId, channelId, id, params);
  }
}

void ttv::chat::ChatChannelSet::AutoModCaughtSentMessage(UserId userId, ChannelId channelId) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModCaughtSentMessage(userId, channelId);
  }
}

void ttv::chat::ChatChannelSet::AutoModDeniedSentMessage(UserId userId, ChannelId channelId) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModDeniedSentMessage(userId, channelId);
  }
}

void ttv::chat::ChatChannelSet::AutoModApprovedSentMessage(UserId userId, ChannelId channelId) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModApprovedSentMessage(userId, channelId);
  }
}

void ttv::chat::ChatChannelSet::AutoModCaughtMessageForMods(UserId userId, ChannelId channelId, std::string&& messageId,
  std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModCaughtMessageForMods(
      userId, channelId, std::move(messageId), std::move(message), senderId, std::move(senderName), std::move(reason));
  }
}

void ttv::chat::ChatChannelSet::AutoModMessageApprovedByMod(
  UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModMessageApprovedByMod(
      userId, channelId, std::move(messageId), moderatorId, std::move(moderatorName));
  }
}

void ttv::chat::ChatChannelSet::AutoModMessageDeniedByMod(
  UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModMessageDeniedByMod(
      userId, channelId, std::move(messageId), moderatorId, std::move(moderatorName));
  }
}

void ttv::chat::ChatChannelSet::AutoModDeniedSentCheer(UserId userId, ChannelId channelId) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModDeniedSentCheer(userId, channelId);
  }
}

void ttv::chat::ChatChannelSet::AutoModTimedOutSentCheer(UserId userId, ChannelId channelId) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModTimedOutSentCheer(userId, channelId);
  }
}

void ttv::chat::ChatChannelSet::AutoModCaughtCheerForMods(UserId userId, ChannelId channelId, std::string&& messageId,
  std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) {
  // find the client listener
  auto iter = mChannels.find(channelId);
  TTV_ASSERT(iter != mChannels.end());
  if (iter == mChannels.end()) {
    return;
  }

  if (iter->second->channelListener != nullptr) {
    iter->second->channelListener->AutoModCaughtCheerForMods(
      userId, channelId, std::move(messageId), std::move(message), senderId, std::move(senderName), std::move(reason));
  }
}

void ttv::chat::ChatChannelSet::SetMessageFlushInterval(uint64_t milliseconds) {
  mUserMessageFlushInterval = milliseconds;

  for (auto iter = mChannels.begin(); iter != mChannels.end(); ++iter) {
    iter->second->channel->SetMessageFlushInterval(milliseconds);
  }
}

std::string ttv::chat::ChatChannelSet::GetLoggerName() const {
  return kLoggerName;
}

ttv::chat::ChatChannelSet::InternalChannelListener::InternalChannelListener(ChatChannelSet* owner) : mOwner(owner) {}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelStateChanged(
  UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec) {
  mOwner->ChatChannelStateChanged(userId, channelId, state, ec);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelInfoChanged(
  UserId userId, ChannelId channelId, const ChatChannelInfo& info) {
  mOwner->ChatChannelInfoChanged(userId, channelId, info);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelRestrictionsChanged(
  UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions) {
  mOwner->ChatChannelRestrictionsChanged(userId, channelId, restrictions);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelLocalUserChanged(
  UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) {
  mOwner->ChatChannelLocalUserChanged(userId, channelId, userInfo);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelMessagesReceived(
  UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) {
  mOwner->ChatChannelMessagesReceived(userId, channelId, messageList);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelSubscriptionNoticeReceived(
  UserId userId, ChannelId channelId, const SubscriptionNotice& notice) {
  mOwner->ChatChannelSubscriptionNoticeReceived(userId, channelId, notice);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelFirstTimeChatterNoticeReceived(
  UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) {
  mOwner->ChatChannelFirstTimeChatterNoticeReceived(userId, channelId, notice);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelRaidNoticeReceived(
  UserId userId, ChannelId channelId, const RaidNotice& notice) {
  mOwner->ChatChannelRaidNoticeReceived(userId, channelId, notice);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelUnraidNoticeReceived(
  UserId userId, ChannelId channelId, const UnraidNotice& notice) {
  mOwner->ChatChannelUnraidNoticeReceived(userId, channelId, notice);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelGenericNoticeReceived(
  UserId userId, ChannelId channelId, const GenericMessageNotice& notice) {
  mOwner->ChatChannelGenericNoticeReceived(userId, channelId, notice);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelMessagesCleared(
  UserId userId, ChannelId channelId) {
  mOwner->ChatChannelMessagesCleared(userId, channelId);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelUserMessagesCleared(
  UserId userId, ChannelId channelId, UserId clearUserId) {
  mOwner->ChatChannelUserMessagesCleared(userId, channelId, clearUserId);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeUserTimedOut(UserId userId,
  ChannelId channelId, ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) {
  mOwner->ChatChannelModNoticeUserTimedOut(
    userId, channelId, std::move(modActionInfo), timeoutDurationSeconds, std::move(reason));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeUserBanned(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason) {
  mOwner->ChatChannelModNoticeUserBanned(userId, channelId, std::move(modActionInfo), std::move(reason));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeUserUntimedOut(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) {
  mOwner->ChatChannelModNoticeUserUntimedOut(userId, channelId, std::move(modActionInfo));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeUserUnbanned(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) {
  mOwner->ChatChannelModNoticeUserUnbanned(userId, channelId, std::move(modActionInfo));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeMessageDeleted(UserId userId,
  ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) {
  mOwner->ChatChannelModNoticeMessageDeleted(
    userId, channelId, std::move(modActionInfo), std::move(messageId), std::move(message));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeClearChat(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeClearChat(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeEmoteOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeEmoteOnly(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeEmoteOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeEmoteOnlyOff(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeFollowersOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t minimumFollowingDurationMinutes) {
  mOwner->ChatChannelModNoticeFollowersOnly(
    userId, channelId, modId, std::move(modName), minimumFollowingDurationMinutes);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeFollowersOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeFollowersOnlyOff(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeR9K(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeR9K(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeR9KOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeR9KOff(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeSlow(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t slowModeDurationSeconds) {
  mOwner->ChatChannelModNoticeSlow(userId, channelId, modId, std::move(modName), slowModeDurationSeconds);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeSlowOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeSlowOff(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeSubsOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeSubsOnly(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelModNoticeSubsOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  mOwner->ChatChannelModNoticeSubsOnlyOff(userId, channelId, modId, std::move(modName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelMessageDeleted(UserId userId, ChannelId channelId,
  std::string&& messageId, std::string&& senderLoginName, std::string&& messageContent) {
  mOwner->ChatChannelMessageDeleted(
    userId, channelId, std::move(messageId), std::move(senderLoginName), std::move(messageContent));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelHostTargetChanged(
  UserId userId, ChannelId channelId, const std::string& targetChannelName, uint32_t numViewers) {
  mOwner->ChatChannelHostTargetChanged(userId, channelId, targetChannelName, numViewers);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::ChatChannelNoticeReceived(
  UserId userId, ChannelId channelId, const std::string& id, const std::map<std::string, std::string>& params) {
  mOwner->ChatChannelNoticeReceived(userId, channelId, id, params);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModCaughtSentMessage(UserId userId, ChannelId channelId) {
  mOwner->AutoModCaughtSentMessage(userId, channelId);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModDeniedSentMessage(UserId userId, ChannelId channelId) {
  mOwner->AutoModDeniedSentMessage(userId, channelId);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModApprovedSentMessage(
  UserId userId, ChannelId channelId) {
  mOwner->AutoModApprovedSentMessage(userId, channelId);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModCaughtMessageForMods(UserId userId, ChannelId channelId,
  std::string&& messageId, std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) {
  mOwner->AutoModCaughtMessageForMods(
    userId, channelId, std::move(messageId), std::move(message), senderId, std::move(senderName), std::move(reason));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModMessageApprovedByMod(
  UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName) {
  mOwner->AutoModMessageApprovedByMod(userId, channelId, std::move(messageId), moderatorId, std::move(moderatorName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModMessageDeniedByMod(
  UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName) {
  mOwner->AutoModMessageDeniedByMod(userId, channelId, std::move(messageId), moderatorId, std::move(moderatorName));
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModDeniedSentCheer(UserId userId, ChannelId channelId) {
  mOwner->AutoModDeniedSentCheer(userId, channelId);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModTimedOutSentCheer(UserId userId, ChannelId channelId) {
  mOwner->AutoModTimedOutSentCheer(userId, channelId);
}

void ttv::chat::ChatChannelSet::InternalChannelListener::AutoModCaughtCheerForMods(UserId userId, ChannelId channelId,
  std::string&& messageId, std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) {
  mOwner->AutoModCaughtCheerForMods(
    userId, channelId, std::move(messageId), std::move(message), senderId, std::move(senderName), std::move(reason));
}

ttv::chat::IChatChannel::~IChatChannel() {}

ttv::chat::ChatChannelWrapper::ChatChannelWrapper(
  const std::shared_ptr<User>& user, ChannelId channelId, const std::shared_ptr<IChatChannelListener>& listener)
    : mChatChannelListener(listener), mChannelId(channelId) {
  mChatChannelSet = std::make_shared<ChatChannelSet>(user);
}

void ttv::chat::ChatChannelWrapper::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }
}

TTV_ErrorCode ttv::chat::ChatChannelWrapper::Connect() {
  return mChatChannelSet->Connect(mChannelId, mChatChannelListener);
}

TTV_ErrorCode ttv::chat::ChatChannelWrapper::Disconnect() {
  return mChatChannelSet->Disconnect(mChannelId);
}

TTV_ErrorCode ttv::chat::ChatChannelWrapper::SendMessage(const std::string& message) {
  return mChatChannelSet->SendChatMessage(mChannelId, message);
}

TTV_ErrorCode ttv::chat::ChatChannelWrapper::FetchUserList(const FetchUserListCallback& callback) {
  return mChatChannelSet->FetchUserList(mChannelId, callback);
}

uint64_t ttv::chat::ChatChannelWrapper::GetRemainingSlowModeTime() {
  return mChatChannelSet->GetRemainingSlowModeTime(mChannelId);
}
