/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatcommentmanager.h"

#include "twitchsdk/chat/internal/bitsconfigrepository.h"
#include "twitchsdk/chat/internal/task/chatdeletecommenttask.h"
#include "twitchsdk/chat/internal/task/chatgetcommentrepliestask.h"
#include "twitchsdk/chat/internal/task/chatgetcommenttask.h"
#include "twitchsdk/chat/internal/task/chatgetvodcommentstask.h"
#include "twitchsdk/chat/internal/task/chatpostcommentreplytask.h"
#include "twitchsdk/chat/internal/task/chatpostcommenttask.h"
#include "twitchsdk/chat/internal/task/chatreportcommenttask.h"
#include "twitchsdk/core/channel/channelrepository.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/getvodtask.h"
#include "twitchsdk/core/user/user.h"

namespace {
const uint32_t kPrefetchThresholdMilliseconds = 5 * 1000;
const uint64_t kErrorFetchRetryMaxBackOff = 1000 * 60 * 2;
const uint64_t kErrorFetchRetryJitterMilliseconds = 1000;

const char* kLoggerName = "ChatCommentManager";
}  // namespace

ttv::chat::IChatCommentManager::~IChatCommentManager() {}

ttv::chat::ChatCommentManager::ChatCommentManager(const std::shared_ptr<User>& user, const std::string& vodId)
    : Component(),
      mUser(user),
      mVodId(vodId),
      mFetchVodRetryTimer(kErrorFetchRetryMaxBackOff, kErrorFetchRetryJitterMilliseconds),
      mFetchBitsConfigRetryTimer(kErrorFetchRetryMaxBackOff, kErrorFetchRetryJitterMilliseconds),
      mPlayheadTimeMilliseconds(0),
      mStartOfCommentBuffer(0),
      mBitsConfigFetchToken(0),
      mClearCount(0),
      mUserId(mUser != nullptr ? mUser->GetUserId() : 0),
      mChannelId(0),
      mPlayingState(IChatCommentManager::PlayingState::Paused),
      mCommentsState(CommentsState::Idle),
      mVodFetchInFlight(false),
      mCommentsFetchInFlight(false),
      mHasFetchedVod(false),
      mHasFetchedBitsConfig(false) {}

std::string ttv::chat::ChatCommentManager::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::ChatCommentManager::FetchVod() {
  if (mVodId.empty()) {
    return TTV_EC_INVALID_ARG;
  }

  if (mVodFetchInFlight) {
    return TTV_EC_REQUEST_PENDING;
  }

  mVodFetchInFlight = true;

  GetVodTask::Callback fetchCallback = [this](GetVodTask* source, TTV_ErrorCode ec,
                                         const std::shared_ptr<GetVodTask::Result>& result) {
    CompleteTask(source);
    mVodFetchInFlight = false;

    if (TTV_SUCCEEDED(ec) && result->status == VodStatus::Recorded) {
      mHasFetchedVod = true;
      mFetchVodRetryTimer.Clear();

      mChannelId = result->channelId;
    } else {
      mFetchVodRetryTimer.ScheduleNextRetry();
    }
  };

  std::shared_ptr<GetVodTask> task = std::make_shared<GetVodTask>(mVodId, fetchCallback);
  TTV_ErrorCode ec = StartTask(task);

  if (TTV_FAILED(ec)) {
    mVodFetchInFlight = false;
    mFetchVodRetryTimer.ScheduleNextRetry();
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatCommentManager::FetchBitsConfig() {
  if (mBitsConfigFetchToken != 0) {
    return TTV_EC_REQUEST_PENDING;
  }

  TTV_ErrorCode ec = mBitsConfigRepository->FetchChannelBitsConfiguration(mUserId, mChannelId,
    [this](TTV_ErrorCode callbackEc, const std::shared_ptr<BitsConfiguration>& config) {
      if (TTV_SUCCEEDED(callbackEc)) {
        mHasFetchedBitsConfig = true;
        mFetchBitsConfigRetryTimer.Clear();

        mBitsConfiguration = config;
      } else {
        mFetchBitsConfigRetryTimer.ScheduleNextRetry();
      }

      mBitsConfigFetchToken = 0;
    },
    mBitsConfigFetchToken);

  if (TTV_FAILED(ec)) {
    mFetchBitsConfigRetryTimer.ScheduleNextRetry();
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatCommentManager::FetchPlayheadComments() {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (mCommentsFetchInFlight) {
    return TTV_EC_REQUEST_PENDING;
  }

  if (mFetchMessagesRetryTimer.IsSet() && !mFetchMessagesRetryTimer.Check(true)) {
    return TTV_EC_REQUEST_PENDING;
  }

  uint32_t clearCount = mClearCount;
  mCommentsFetchInFlight = true;

  ChatGetVodCommentsTask::Callback fetchCallback = [this, clearCount](ChatGetVodCommentsTask* source, TTV_ErrorCode ec,
                                                     ChatGetVodCommentsTask::Result&& result) {
    CompleteTask(source);

    mCommentsFetchInFlight = false;

    // If a clear has occurred since starting the fetch, we don't care about the results
    if (clearCount != mClearCount) {
      return;
    }

    if (TTV_SUCCEEDED(ec)) {
      if ((result.commentsListBatches.empty() || result.nextCursorUrl == "")) {
        // We've fetched all comments until the end of the VOD.
        SetCommentsState(CommentsState::DoneFetching);
        if (mFetchedLists.empty() && result.commentsListBatches.empty()) {
          // There are no more comments to fetch nor to display.
          SetCommentsState(CommentsState::Finished);
          SetPlayingState(PlayingState::Finished);
          return;
        }
      }

      auto startIter = result.commentsListBatches.begin();
      mNextCursorUrl = std::move(result.nextCursorUrl);
      std::move(startIter, result.commentsListBatches.end(), std::back_inserter(mFetchedLists));
    }
    // Failed so try again soon
    else if (ec != TTV_EC_REQUEST_ABORTED) {
      mListener->ChatCommentsErrorReceived(result.errorMsg, ec);
      mFetchMessagesRetryTimer.Set(kErrorFetchRetryJitterMilliseconds);
    }
  };

  std::shared_ptr<ChatGetVodCommentsTask> fetchTask;
  if (mNextCursorUrl.empty()) {
    fetchTask = std::make_shared<ChatGetVodCommentsTask>(
      mVodId, mPlayheadTimeMilliseconds, mTokenizationOptions, mBitsConfiguration, std::move(fetchCallback));

  } else {
    fetchTask = std::make_shared<ChatGetVodCommentsTask>(
      mVodId, mNextCursorUrl, mTokenizationOptions, mBitsConfiguration, std::move(fetchCallback));
  }

  if (mUser != nullptr) {
    fetchTask->SetLocalUserNames(std::vector<std::string>{mUser->GetUserName(), mUser->GetDisplayName()});
  }

  TTV_ErrorCode ec = StartTask(fetchTask);

  mFetchMessagesRetryTimer.Clear();

  if (TTV_FAILED(ec)) {
    mCommentsFetchInFlight = false;
    mFetchMessagesRetryTimer.Set(kErrorFetchRetryJitterMilliseconds);
  }

  return ec;
}

ttv::Result<ttv::ChannelId> ttv::chat::ChatCommentManager::GetChannelId() const {
  if (!mHasFetchedVod) {
    return MakeErrorResult(TTV_EC_NOT_AVAILABLE);
  }
  return MakeSuccessResult(mChannelId);
}

ttv::Result<ttv::chat::ChatCommentManager::PlayingState> ttv::chat::ChatCommentManager::GetPlayingState() const {
  return MakeSuccessResult(mPlayingState);
}

// Component Overrides
TTV_ErrorCode ttv::chat::ChatCommentManager::Initialize() {
  TTV_ASSERT(mChannelRepository != nullptr);

  TTV_ErrorCode ec = Component::Initialize();
  if (TTV_SUCCEEDED(ec)) {
    // Start the retry timers
    mFetchVodRetryTimer.ScheduleNextRetry();
    mFetchBitsConfigRetryTimer.ScheduleNextRetry();
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatCommentManager::Shutdown() {
  TTV_ErrorCode ec = Component::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    mFetchVodRetryTimer.Clear();
    mFetchBitsConfigRetryTimer.Clear();
  }

  if (mBitsConfigFetchToken != 0 && mBitsConfigRepository != nullptr) {
    mBitsConfigRepository->CancelFetch(mBitsConfigFetchToken);
  }

  return ec;
}

bool ttv::chat::ChatCommentManager::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  if (mBitsConfigFetchToken != 0) {
    return false;
  }

  return true;
}

void ttv::chat::ChatCommentManager::Update() {
  Component::Update();

  if (mState == State::Initialized) {
    if (mHasFetchedVod && mHasFetchedBitsConfig && mPlayingState != PlayingState::Finished) {
      auto commentBufferMs = CommentBufferSizeMs();

      // If the buffer is filled then transition to MoreToFetch
      // Note: if there are not enough comments in the VOD to buffer kPrefetchThresholdMilliseconds then we will
      // transition to DoneFetching state directly from FillingBuffer
      if (mCommentsState == CommentsState::FillingBuffer && commentBufferMs >= kPrefetchThresholdMilliseconds) {
        SetCommentsState(CommentsState::MoreToFetch);
      }

      if ((mCommentsState == CommentsState::MoreToFetch || mCommentsState == CommentsState::FillingBuffer) &&
          (commentBufferMs < kPrefetchThresholdMilliseconds)) {
        FetchPlayheadComments();
      }

      if ((commentBufferMs > 0) ||
          (mCommentsState != CommentsState::MoreToFetch && mCommentsState != CommentsState::FillingBuffer)) {
        if (mPlayingState == PlayingState::Buffering) {
          SetPlayingState(PlayingState::Playing);
        }
      } else {
        if (mPlayingState == PlayingState::Playing) {
          SetPlayingState(PlayingState::Buffering);
        }
      }

      if (mPlayingState == PlayingState::Playing) {
        Advance();
      }
    }

    if (!mHasFetchedVod && mFetchVodRetryTimer.CheckNextRetry()) {
      FetchVod();
    }

    // Need to have finished fetching VOD before we can fetch the bits configuration
    if (mHasFetchedVod && mFetchBitsConfigRetryTimer.CheckNextRetry()) {
      FetchBitsConfig();
    }
  }
}

bool ttv::chat::ChatCommentManager::IsPositionInsideCommentBuffer(uint64_t playheadPos) {
  if (mCommentsState == CommentsState::Finished || mCommentsState == CommentsState::DoneFetching) {
    // If we have fetched all the comments for the vod any playhead position after the start of the comment buffer is
    // considered in buffer
    return playheadPos >= mStartOfCommentBuffer;
  }

  if (mFetchedLists.empty()) {
    // If we haven't fetched anything any position is outside the buffer
    return false;
  }

  if (mFetchedLists.back().comments.empty()) {
    // Invalid state, an empty comment batch should never be created and inserted into mFetchedLists
    TTV_ASSERT(false);
    return false;
  }

  auto lastTimestamp = mFetchedLists.back().comments.back().timestampMilliseconds;

  return playheadPos >= mStartOfCommentBuffer && playheadPos <= lastTimestamp;
}

int64_t ttv::chat::ChatCommentManager::CommentBufferSizeMs() {
  if (mFetchedLists.empty()) {
    return 0;
  }
  auto firstTimestamp = mFetchedLists.begin()->comments.begin()->timestampMilliseconds;
  auto lastTimestamp = mFetchedLists.back().comments.back().timestampMilliseconds;
  return lastTimestamp - firstTimestamp;
}

void ttv::chat::ChatCommentManager::Advance() {
  if (!mHasFetchedVod || mCommentsState == CommentsState::Finished || mPlayingState == PlayingState::Finished) {
    return;
  }

  auto playheadPosMs = mPlayheadTimeMilliseconds;
  for (auto iter = mFetchedLists.begin(); iter != mFetchedLists.end();) {
    // If we're past the timestamp for that batch of messages, send the batch to the listener.
    if (iter->baseTimestampMilliseconds <= playheadPosMs) {
      if (mListener != nullptr) {
        mListener->ChatCommentsReceived(mUserId, GetVodId(), std::move(iter->comments));
      }

      iter = mFetchedLists.erase(iter);
    }
    // If the messages' timestamp isn't past the current VOD time, no need to check any of the later batches.
    else {
      break;
    }
  }

  mStartOfCommentBuffer = mPlayheadTimeMilliseconds;

  if (mFetchedLists.empty() && mCommentsState == CommentsState::DoneFetching) {
    SetCommentsState(CommentsState::Finished);
    SetPlayingState(PlayingState::Finished);
  }
}

TTV_ErrorCode ttv::chat::ChatCommentManager::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatCommentManager::Play() {
  if (mPlayingState == PlayingState::Buffering) {
    return TTV_EC_SUCCESS;
  }
  SetPlayingState(PlayingState::Playing);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatCommentManager::Pause() {
  SetPlayingState(PlayingState::Paused);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatCommentManager::UpdatePlayhead(uint64_t timestampMilliseconds) {
  mPlayheadTimeMilliseconds = timestampMilliseconds;

  if (!(mCommentsState == CommentsState::Idle || mCommentsState == CommentsState::MoreToFetch ||
        mCommentsState == CommentsState::DoneFetching || mCommentsState == CommentsState::Finished)) {
    return TTV_EC_SUCCESS;
  }

  bool insideBuffer = IsPositionInsideCommentBuffer(timestampMilliseconds);

  if (!insideBuffer) {
    // If we encounter a timestamp that is outside of the current buffer we need to clear the buffer and re-request the
    // comments
    mClearCount++;
    mFetchedLists.clear();
    mNextCursorUrl = "";
    mStartOfCommentBuffer = timestampMilliseconds;
    mCommentsState = CommentsState::FillingBuffer;

    // If VOD was paused before seek, should remain paused.
    if (mPlayingState != PlayingState::Paused) {
      SetPlayingState(PlayingState::Buffering);
    }

    mFetchMessagesRetryTimer.Clear();
  }

  return TTV_EC_SUCCESS;
}

ttv::Result<uint64_t> ttv::chat::ChatCommentManager::GetPlayheadTime() const {
  return MakeSuccessResult(mPlayheadTimeMilliseconds);
}

void ttv::chat::ChatCommentManager::SetPlayingState(PlayingState state) {
  if (mPlayingState == state) {
    return;
  }
  mPlayingState = state;

  if (mListener != nullptr) {
    mListener->ChatCommentManagerStateChanged(mUserId, GetVodId(), state);
  }
}

void ttv::chat::ChatCommentManager::SetCommentsState(CommentsState state) {
  switch (state) {
    case CommentsState::Finished:
      TTV_ASSERT(mCommentsState == CommentsState::DoneFetching);
      break;
    case CommentsState::DoneFetching:
      TTV_ASSERT(mCommentsState == CommentsState::MoreToFetch || mCommentsState == CommentsState::FillingBuffer);
      break;
    case CommentsState::MoreToFetch:
      TTV_ASSERT(mCommentsState == CommentsState::FillingBuffer);
      break;
    case CommentsState::Idle:
      TTV_ASSERT(false);  // Idle should only be used upon class instantiation
      break;
    case CommentsState::FillingBuffer:
      TTV_ASSERT(mCommentsState == CommentsState::Idle || mCommentsState == CommentsState::MoreToFetch ||
                 mCommentsState == CommentsState::DoneFetching || mCommentsState == CommentsState::Finished);
      break;
  }

  mCommentsState = state;
}

TTV_ErrorCode ttv::chat::ChatCommentManager::FetchComments(
  uint64_t timestampMilliseconds, uint32_t limit, const FetchCommentsCallback& callback) {
  TTV_RETURN_ON_SAME(limit, 0, TTV_EC_INVALID_ARG);

  // Potentially passing in a nullptr mBitsConfiguration will simply not end up tokenizing bits.
  auto task =
    std::make_shared<ChatGetVodCommentsTask>(mVodId, timestampMilliseconds, mTokenizationOptions, mBitsConfiguration,
      [this, callback](ChatGetVodCommentsTask* source, TTV_ErrorCode ec, ChatGetVodCommentsTask::Result&& result) {
        CompleteTask(source);

        if (callback != nullptr) {
          std::vector<ChatComment> comments;
          for (auto& commentsBatch : result.commentsListBatches) {
            std::move(commentsBatch.comments.begin(), commentsBatch.comments.end(), std::back_inserter(comments));
          }
          callback(ec, std::move(comments), std::move(result.nextCursorUrl));
        }
      });

  task->SetFetchLimit(limit);

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatCommentManager::FetchComments(
  const std::string& cursor, uint32_t limit, const FetchCommentsCallback& callback) {
  TTV_RETURN_ON_EMPTY_STDSTRING(cursor, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_SAME(limit, 0, TTV_EC_INVALID_ARG);

  // Potentially passing in a nullptr mBitsConfiguration will simply not end up tokenizing bits.
  auto task = std::make_shared<ChatGetVodCommentsTask>(mVodId, cursor, mTokenizationOptions, mBitsConfiguration,
    [this, callback](ChatGetVodCommentsTask* source, TTV_ErrorCode ec, ChatGetVodCommentsTask::Result&& result) {
      CompleteTask(source);

      if (callback != nullptr) {
        std::vector<ChatComment> comments;
        for (auto& commentsBatch : result.commentsListBatches) {
          std::move(commentsBatch.comments.begin(), commentsBatch.comments.end(), std::back_inserter(comments));
        }
        callback(ec, std::move(comments), std::move(result.nextCursorUrl));
      }
    });

  task->SetFetchLimit(limit);

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatCommentManager::FetchComment(
  const std::string& commentId, const FetchCommentCallback& callback) {
  if (commentId.empty()) {
    return TTV_EC_INVALID_ARG;
  }

  auto task = std::make_shared<ChatGetCommentTask>(commentId, mTokenizationOptions, mBitsConfiguration,
    [this, callback](ChatGetCommentTask* source, TTV_ErrorCode ec, ChatComment&& comment) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec, std::move(comment));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatCommentManager::DeleteComment(
  const std::string& commentId, const DeleteCommentCallback& callback) {
  if (commentId.empty()) {
    return TTV_EC_INVALID_ARG;
  }
  if (mUser == nullptr || mUser->GetOAuthToken() == nullptr) {
    return TTV_EC_AUTHENTICATION;
  }

  auto task = std::make_shared<ChatDeleteCommentTask>(
    commentId, mUser->GetOAuthToken()->GetToken(), [this, callback](ChatDeleteCommentTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec);
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatCommentManager::PostComment(
  const std::string& message, uint64_t timestampMilliseconds, const PostCommentCallback& callback) {
  if (message.empty()) {
    return TTV_EC_INVALID_ARG;
  }
  if (mUser == nullptr || mUser->GetOAuthToken() == nullptr) {
    return TTV_EC_AUTHENTICATION;
  }

  auto task = std::make_shared<ChatPostCommentTask>(mVodId, message, timestampMilliseconds, mTokenizationOptions,
    mBitsConfiguration, mUser->GetOAuthToken()->GetToken(),
    [this, callback](ChatPostCommentTask* source, TTV_ErrorCode ec, ChatComment&& comment, std::string&& errorMessage) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec, std::move(comment), std::move(errorMessage));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatCommentManager::ReportComment(const std::string& parentCommentId,
  const std::string& reason, const std::string& description, const ReportCommentCallback& callback) {
  if (parentCommentId.empty() || reason.empty()) {
    return TTV_EC_INVALID_ARG;
  }
  if (mUser == nullptr || mUser->GetOAuthToken() == nullptr) {
    return TTV_EC_AUTHENTICATION;
  }

  auto task = std::make_shared<ChatReportCommentTask>(parentCommentId, reason, description,
    mUser->GetOAuthToken()->GetToken(), [this, callback](ChatReportCommentTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec);
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatCommentManager::FetchCommentReplies(
  const std::string& commentId, const FetchCommentRepliesCallback& callback) {
  if (commentId.empty()) {
    return TTV_EC_INVALID_ARG;
  }
  auto task = std::make_shared<ChatGetCommentRepliesTask>(commentId, mTokenizationOptions, mBitsConfiguration,
    [this, callback](ChatGetCommentRepliesTask* source, TTV_ErrorCode ec, std::vector<ChatComment>&& replies) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec, std::move(replies));
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatCommentManager::PostCommentReply(
  const std::string& commentId, const std::string& message, const PostCommentReplyCallback& callback) {
  if (commentId.empty() || message.empty()) {
    return TTV_EC_INVALID_ARG;
  }
  if (mUser == nullptr || mUser->GetOAuthToken() == nullptr) {
    return TTV_EC_AUTHENTICATION;
  }

  auto task = std::make_shared<ChatPostCommentReplyTask>(commentId, message, mTokenizationOptions, mBitsConfiguration,
    mUser->GetOAuthToken()->GetToken(),
    [this, callback](ChatPostCommentReplyTask* source, TTV_ErrorCode ec, ChatComment&& reply) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec, std::move(reply));
      }
    });

  return StartTask(task);
}
