/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatconnection.h"

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/internal/chatchannel.h"
#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/internal/chatreader.h"
#include "twitchsdk/chat/internal/chatsession.h"
#include "twitchsdk/chat/internal/chatsockettransport.h"
#include "twitchsdk/chat/internal/chatuserblocklist.h"
#include "twitchsdk/chat/internal/chatwriter.h"
#include "twitchsdk/chat/internal/defaultchatobjectfactory.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/user/oauthtoken.h"

#include <sstream>

#define SIMULATE_SOCKET_CONNECTION_FAILED 0
#define SIMULATE_NO_WELCOME_RESPONSE 0
#define SIMULATE_NO_JOIN_RESPONSE 0

namespace {
const uint64_t kWelcomeTimeoutMilliseconds = 8 * 1000;
const uint64_t kJoinTimeoutMilliseconds = 8 * 1000;
const uint64_t kMaxEventProcessingTimeMilliseconds = 500;
const char* kIrcCapabilities = ":twitch.tv/tags twitch.tv/commands";
}  // namespace

ttv::chat::ChatConnection::ChatConnection(ChannelId channelId, const std::shared_ptr<User>& user)
    : mConnectionState(CONNECTIONSTATE_DISCONNECTED),
      mListener(nullptr),
      mConnectionError(TTV_EC_SUCCESS),
      mUser(user),
      mChannelId(channelId),
      mDisconnectionRequested(false),
      mFireConnectionErrorEvents(true) {
  mAnonymous = (mUser->GetUserId() == 0);
  if (mAnonymous) {
    mUserName = CreateAnonymousUserName();
  } else {
    mUserName = mUser->GetUserName();
  }
}

ttv::chat::ChatConnection::~ChatConnection() {
  Disconnect();
}

TTV_ErrorCode ttv::chat::ChatConnection::CheckFactoryAvailability(const std::string& uri) {
  return ChatSocketTransport::CheckFactoryAvailability(uri);
}

void ttv::chat::ChatConnection::SetChannelName(const std::string& channelName) {
  mIrcChannelName = "#" + channelName;
}

std::string ttv::chat::ChatConnection::CreateAnonymousUserName() {
  // generate a compatible username
  char buffer[64];
  (void)snprintf(buffer, sizeof(buffer), "%llu", static_cast<unsigned long long>(GetSystemClockTime()));
  buffer[sizeof(buffer) - 1] = 0;

  size_t len = strlen(buffer);

  // reverse the string so the lower order digits come first
  for (size_t i = 0; i < len / 2; ++i) {
    char ch = buffer[i];
    buffer[i] = buffer[len - i - 1];
    buffer[len - i - 1] = ch;
  }

  // clamp the value to 9 characters since justinfan can have up to 9 numeric digits after it
  buffer[9] = '\0';

  return std::string("justinfan") + std::string(buffer);
}

TTV_ErrorCode ttv::chat::ChatConnection::Connect(const std::string& uri) {
  Disconnect();

  SetState(CONNECTIONSTATE_CONNECTING);

  ttv::trace::Message(
    "Chat", MessageLevel::Info, "ChatConnection::Connect(): Attempting to connect to: %s", uri.c_str());

  TTV_ErrorCode ec;

#if SIMULATE_SOCKET_CONNECTION_FAILED
  ec = TTV_EC_SOCKET_CONNECT_FAILED;
#else
  ec = mTransport->Connect(uri);
#endif

  if (TTV_SUCCEEDED(ec)) {
    ttv::trace::Message(
      "Chat", MessageLevel::Info, "ChatConnection::Connect(): Connection succeeded to: %s", uri.c_str());

    SetState(CONNECTIONSTATE_WELCOMING);

    std::string password;
    if (mAnonymous) {
      password = "listen";
    } else {
      password = std::string("oauth:") + mUser->GetOAuthToken()->GetToken();
    }

    mSession->Cap("REQ", kIrcCapabilities);
    mSession->Pass(password);
    mSession->Nick(mUserName);
  } else {
    ttv::trace::Message("Chat", MessageLevel::Info, "ChatConnection::Connect(): Connection failed to: %s", uri.c_str());

    // Don't fire connection failed events when we return an error code synchronously
    mFireConnectionErrorEvents = false;
    SetState(CONNECTIONSTATE_CONNECTION_FAILED);
    mFireConnectionErrorEvents = true;
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatConnection::Disconnect() {
  SetState(CONNECTIONSTATE_DISCONNECTED);
  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatConnection::Join() {
  mSession->Join(mIrcChannelName);
  mJoinAttemptTimeout.Reset(kJoinTimeoutMilliseconds);
}

void ttv::chat::ChatConnection::CreateObjects() {
  if (mTransport == nullptr) {
    TTV_ASSERT(mChatObjectFactory != nullptr);

    mTransport = mChatObjectFactory->CreateChatTransport();
    mSession = std::make_shared<ChatSession>();
    mWriter = std::make_shared<ChatWriter>();
    mReader = std::make_shared<ChatReader>();

    mWriter->SetTransport(mTransport);
    mReader->SetNotifySink(this);
    mTransport->SetReader(std::static_pointer_cast<IChatTransportReader>(mReader));
    mSession->SetWriter(std::static_pointer_cast<IChatWriteNetworkEvent>(mWriter));
  }
}

void ttv::chat::ChatConnection::ReleaseObjects() {
  if (mTransport != nullptr) {
    mTransport->Close();

    mTransport->SetReader(nullptr);
    mSession->SetWriter(nullptr);

    mSession.reset();
    mWriter.reset();
    mReader.reset();
    mTransport.reset();
  }
}

void ttv::chat::ChatConnection::SetState(ConnectionState state) {
  if (state == mConnectionState) {
    return;
  }

  ttv::trace::Message(
    "Chat", MessageLevel::Debug, "ChatConnection::SetState(): Changing state: %d -> %d", mConnectionState, state);

  mConnectionState = state;

  switch (state) {
    case CONNECTIONSTATE_DISCONNECTED: {
      mDisconnectionRequested = true;
      ReleaseObjects();
      break;
    }
    case CONNECTIONSTATE_CONNECTING: {
      mDisconnectionRequested = false;
      mConnectionError = TTV_EC_SUCCESS;
      CreateObjects();
      break;
    }
    case CONNECTIONSTATE_WELCOMING: {
      mWelcomeMessageTimeout.Reset(kWelcomeTimeoutMilliseconds);
      break;
    }
    case CONNECTIONSTATE_WELCOMED: {
      // No op
      break;
    }
    case CONNECTIONSTATE_JOINING: {
      Join();
      break;
    }
    case CONNECTIONSTATE_CONNECTED: {
      mJoinAttemptTimeout.Complete();

      // Send the connection event
      if (mListener != nullptr) {
        mListener->OnConnected(this);
      }

      break;
    }
    case CONNECTIONSTATE_CONNECTION_FAILED: {
      ReleaseObjects();

      if (TTV_SUCCEEDED(mConnectionError)) {
        mConnectionError = TTV_EC_CHAT_COULD_NOT_CONNECT;
      }

      // Send the disconnection event
      if (mFireConnectionErrorEvents && mListener != nullptr) {
        mListener->OnConnectionFailed(this, mConnectionError);
      }
      break;
    }
    case CONNECTIONSTATE_CONNECTION_LOST: {
      ReleaseObjects();

      if (TTV_SUCCEEDED(mConnectionError)) {
        mConnectionError = TTV_EC_CHAT_LOST_CONNECTION;
      }

      // Send the disconnection event
      if (mFireConnectionErrorEvents && mListener != nullptr) {
        mListener->OnConnectionLost(this, mConnectionError);
      }
      break;
    }
  }
}

void ttv::chat::ChatConnection::SetChatObjectFactory(std::shared_ptr<IChatObjectFactory> factory) {
  mChatObjectFactory = factory;
}

TTV_ErrorCode ttv::chat::ChatConnection::Update() {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  switch (mConnectionState) {
    case CONNECTIONSTATE_WELCOMING: {
      if (mWelcomeMessageTimeout.IsTimedOut()) {
        ttv::trace::Message(
          "Chat", MessageLevel::Debug, "ChatConnection::Update(): Timed out waiting for IRC welcome message");

        mConnectionError = TTV_EC_CHAT_COULD_NOT_CONNECT;
        SetState(CONNECTIONSTATE_CONNECTION_FAILED);
      }
      break;
    }
    case CONNECTIONSTATE_WELCOMED: {
      if (!mIrcChannelName.empty()) {
        SetState(CONNECTIONSTATE_JOINING);
      }
      break;
    }
    case CONNECTIONSTATE_JOINING: {
      if (mJoinAttemptTimeout.IsTimedOut()) {
        ttv::trace::Message(
          "Chat", MessageLevel::Debug, "ChatConnection::Update(): Timed out waiting for IRC join response");

        mConnectionError = TTV_EC_CHAT_COULD_NOT_CONNECT;
        SetState(CONNECTIONSTATE_CONNECTION_FAILED);
      }
      break;
    }
    default: { break; }
  }

  // keep a reference around in case the transport is implicitly destroyed by flushing events from the server
  std::shared_ptr<IChatTransport> transport = mTransport;

  // not connected
  if (transport == nullptr || !transport->IsOpen()) {
    // If the transport is not open, and we have not explicitly disconnected,
    // we assume that ChatConnection has lost connection.
    if (mConnectionState != CONNECTIONSTATE_DISCONNECTED) {
      mConnectionError = TTV_EC_CHAT_LOST_CONNECTION;
      SetState(CONNECTIONSTATE_CONNECTION_LOST);
    }

    return ec;
  }

  // process incoming messages from the chat server until none available or shutdown
  uint64_t startTime = GetSystemTimeMilliseconds();
  bool handled = true;
  while (handled) {
    ec = transport->ProcessIncomingEvent(handled);

    // Don't process for too long because we'll block the caller
    if ((GetSystemTimeMilliseconds() - startTime) > kMaxEventProcessingTimeMilliseconds) {
      break;
    }
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatConnection::SendChatMessage(
  const std::string& message, const ChatUserInfo& localUserInfo) {
  // not allowed when anonymous
  if (mAnonymous) {
    return TTV_EC_CHAT_ANON_DENIED;
  }

  std::string line = message;
  Trim(line);

  // empty line
  if (line.size() == 0) {
    return TTV_EC_SUCCESS;
  }

  bool sendReal = true;
  bool sendFake = true;
  bool parseEmotes = true;
  std::string prefix = mUserName;
  std::string fakeLine = line;
  int eventId = IRC_CMD_PRIVMSG;

  if (line[0] == '/') {
    std::vector<std::string> tokens;
    Split(line, tokens, ' ', false);

    // convert to lower case
    (void)std::transform(tokens[0].begin(), tokens[0].end(), tokens[0].begin(), ::tolower);

    // ignore of user is handled on the client
    if (tokens[0] == "/ignore") {
      // "/ignore <username>"
      if (tokens.size() < 2) {
        return TTV_EC_SUCCESS;
      }

      parseEmotes = false;
      sendReal = false;

      // Notify the listener
      if (mListener != nullptr) {
        mListener->OnIgnoreChanged(this, tokens[1], true);
      }
    }
    // unignore of user is handled on the client
    else if (tokens[0] == "/unignore") {
      // "/unignore <username>"
      if (tokens.size() < 2) {
        return TTV_EC_SUCCESS;
      }

      parseEmotes = false;
      sendReal = false;

      // Notify the listener
      if (mListener != nullptr) {
        mListener->OnIgnoreChanged(this, tokens[1], false);
      }
    }
    // We don't echo the host/unhost commands, which matches the behavior on web.
    else if (tokens[0] == "/unhost" || tokens[0] == "/host") {
      parseEmotes = false;
      sendFake = false;
    }
    // clear the chatroom
    else if (tokens[0] == "/clear") {
      parseEmotes = false;
      sendFake = false;
    }
    // me
    else if (tokens[0] == "/me") {
      static const size_t trimLength = strlen("/me ");
      eventId = IRC_CTCP_ACTION;

      fakeLine = fakeLine.substr(std::min(trimLength, fakeLine.size()));
    }

    // NOTE: /w is no longer handled at this level, apps should intercept this directly and call SendMessageToUser
    // instead
  }

  // fake an incoming user message event
  if (sendFake) {
    bool firstTag = true;
    std::stringstream messageTags;

    if (parseEmotes) {
      std::string emotesMessageTag;
      std::string badgesMessageTag;

      if (TokenizeLocalMessage(mUser, mChannelId, fakeLine, emotesMessageTag, badgesMessageTag)) {
        // Reconstruct the full message tag
        if (!emotesMessageTag.empty()) {
          messageTags << "emotes=";
          messageTags << emotesMessageTag;

          firstTag = false;
        }

        if (!badgesMessageTag.empty()) {
          if (firstTag) {
            firstTag = false;
          } else {
            messageTags << ";";
          }

          messageTags << "badges=";
          messageTags << badgesMessageTag;
        }

        if (!firstTag) {
          messageTags << ";";
        }

        messageTags << "user-id=";
        messageTags << mUser->GetUserId();

        std::string colorString;
        if (GenerateColorString(localUserInfo.nameColorARGB, colorString)) {
          messageTags << ";";

          messageTags << "color=";
          messageTags << colorString;
        }

        if (!localUserInfo.displayName.empty()) {
          messageTags << ";";

          messageTags << "display-name=";
          messageTags << localUserInfo.displayName;
        }

        if (localUserInfo.userMode.moderator) {
          messageTags << ";";

          messageTags << "mod=";
          messageTags << "1";
        }

        if (localUserInfo.userMode.subscriber) {
          messageTags << ";";

          messageTags << "subscriber=";
          messageTags << "1";
        }
      }
    }

    ChatNetworkEvent evt(eventId, 2, &prefix, &fakeLine);
    evt.SetPrefix(prefix);
    evt.SetMessageTags(messageTags.str());

    ReceiveEvent(evt);
  }

  // send a message to the server
  if (sendReal) {
    TTV_ASSERT(mSession != nullptr);
    mSession->PrivMsg(mIrcChannelName, line);
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatConnection::ReceiveEvent(const ChatNetworkEvent& evt) {
#if SIMULATE_NO_WELCOME_RESPONSE
  if (mConnectionState == CONNECTIONSTATE_WELCOMING) {
    return;
  }
#endif

  // waiting for a response to the join request
  if (mConnectionState == CONNECTIONSTATE_JOINING) {
#if SIMULATE_NO_JOIN_RESPONSE
    return;
#else
    // ensure that the join notification comes through - sometimes messages are not sent from the server
    switch (evt.GetEventID()) {
      case IRC_CMD_PRIVMSG:
      case IRC_CTCP_ACTION:
      case IRC_CMD_JOIN:
      case IRC_CMD_PART:
      case IRC_CMD_MODE:
      case IRC_RPL_NAMREPLY: {
        SetState(CONNECTIONSTATE_CONNECTED);
      } break;
    }
#endif
  }

  // handle the message
  switch (evt.GetEventID()) {
    // received a private message
    case IRC_CMD_PRIVMSG:
    // an action message
    case IRC_CTCP_ACTION: {
      HandleMessageTags(evt);
      HandlePrivateMessage(evt);
      break;
    }
    // userstate
    case IRC_CMD_USERSTATE: {
      HandleMessageTags(evt);
      HandleUserState(evt);
      break;
    }
    // global userstate
    case IRC_CMD_GLOBALUSERSTATE: {
      // NOTE: No longer used
      break;
    }
    // roomstate
    case IRC_CMD_ROOMSTATE: {
      HandleRoomState(evt);
      break;
    }
    // someone joined/left the channel
    case IRC_CMD_JOIN:
    case IRC_CMD_PART: {
      // These messages shouldn't come through in the custom IRC version we're using
      break;
    }
    // received a ping request - need to reply
    case IRC_CMD_PING: {
      TTV_ASSERT(mSession != nullptr);
      mSession->Pong(evt.GetParam(0));
      break;
    }
    // received a mode change
    case IRC_CMD_MODE: {
      // NOTE: These messages come in way too late and may cause a user's mode to become out of date if an old MODE
      // change is applied
      // HandleModeMessage(evt);
      break;
    }
    // login successful
    case IRC_RPL_WELCOME: {
      SetState(CONNECTIONSTATE_WELCOMED);
      break;
    }
    // notice
    case IRC_CMD_NOTICE: {
      HandleNotice(evt);
      break;
    }
    // capabilities
    case IRC_CMD_CAP: {
      HandleCapMessage(evt);
      break;
    }
    // clear chat
    case IRC_CMD_CLEARCHAT: {
      HandleClearChatMessage(evt);
      break;
    }
    // connection closed or request to leave the channel
    case SYS_EVENT_CLOSE: {
      TTV_ASSERT(mTransport);

      ttv::trace::Message("Chat", MessageLevel::Debug, "Connection to chat server was closed");

      if (mDisconnectionRequested) {
        SetState(CONNECTIONSTATE_DISCONNECTED);
      } else {
        if (TTV_SUCCEEDED(mConnectionError) || mConnectionError == TTV_EC_CHAT_LOST_CONNECTION) {
          SetState(CONNECTIONSTATE_CONNECTION_LOST);
        } else {
          SetState(CONNECTIONSTATE_CONNECTION_FAILED);
        }
      }

      break;
    }
    // host target
    case IRC_CMD_HOSTTARGET: {
      HandleHostTargetMessage(evt);
      break;
    }
    // user notice
    case IRC_CMD_USERNOTICE: {
      HandleUserNotice(evt);
      break;
    }
    // delete a single message
    case IRC_CMD_CLEARMSG: {
      HandleDeleteChatMessage(evt);
      break;
    }
    default: { break; }
  }
}

void ttv::chat::ChatConnection::HandleMessageTags(const ChatNetworkEvent& evt) {
  const std::map<std::string, std::string>& tags = evt.GetMessageTags();

  std::string nick;
  switch (evt.GetEventID()) {
    case IRC_CMD_USERSTATE:
    case IRC_CMD_GLOBALUSERSTATE:
      nick = mUserName;
      break;
    default:
      nick = GetPrefixNick(evt.GetPrefix());
      break;
  }

  auto iter = tags.find("badges");
  if (iter != tags.end() && mListener != nullptr) {
    mListener->OnBadgesChanged(this, nick, iter->second);
  }
}

void ttv::chat::ChatConnection::HandleRoomState(const ChatNetworkEvent& evt) {
  const std::map<std::string, std::string>& tags = evt.GetMessageTags();

  bool restrictionsUpdated = false;
  for (auto iter = tags.begin(); iter != tags.end(); ++iter) {
    const std::string& name = iter->first;
    const std::string& value = iter->second;

    // emote-only = 0,1
    if (name == "emote-only") {
      int emoteOnlyValue;
      if (ParseNum(value, emoteOnlyValue)) {
        mChatRestrictions.emoteOnly = (emoteOnlyValue != 0);
        restrictionsUpdated = true;
      }
    }
    // followers-only = -1 if off, or any int >= 0 if on
    else if (name == "followers-only") {
      int followingMinimumDuration;
      if (ParseNum(value, followingMinimumDuration)) {
        mChatRestrictions.followersOnly = (followingMinimumDuration != -1);
        mChatRestrictions.followersDuration =
          mChatRestrictions.followersOnly ? static_cast<uint32_t>(followingMinimumDuration) : 0;
        restrictionsUpdated = true;
      }
    }
    // r9k = 0,1
    else if (name == "r9k") {
      int r9kValue;
      if (ParseNum(value, r9kValue)) {
        mChatRestrictions.r9k = (r9kValue != 0);
        restrictionsUpdated = true;
      }
    }
    // slow = 0 if off, any unsigned int if on
    else if (name == "slow") {
      int slowModeDuration = 0;
      if (ParseNum(value, slowModeDuration)) {
        // Slow mode duration should handle cases that are less that zero
        if (slowModeDuration < 0) {
          slowModeDuration = 0;
        }

        mChatRestrictions.slowMode = (slowModeDuration != 0);
        mChatRestrictions.slowModeDuration = static_cast<uint32_t>(slowModeDuration);
        restrictionsUpdated = true;
      }
    }
    // subs-only = 0,1
    else if (name == "subs-only") {
      int subsOnly;
      if (ParseNum(value, subsOnly)) {
        mChatRestrictions.subscribersOnly = (subsOnly != 0);
        restrictionsUpdated = true;
      }
    }
  }

  if (restrictionsUpdated && mListener != nullptr) {
    mListener->OnChatRestrictionsChanged(this, mChatRestrictions);
  }
}

void ttv::chat::ChatConnection::HandleCapMessage(const ChatNetworkEvent& evt) {
  // unexpected command format
  if (evt.GetParamCount() < 3) {
    return;
  }

  //:tmi.twitch.tv CAP * NAK :twitch.tv/commands
  //:tmi.twitch.tv CAP * ACK :twitch.tv/tags

  if (evt.GetParam(0) == "*") {
    const std::string& result = evt.GetParam(1);
    const std::string& capability = evt.GetParam(2);

    // we don't expect this to ever happen for Twitch capabilities
    if (result == "NAK") {
      ttv::trace::Message(
        "Chat", MessageLevel::Error, "Capability request failed for capability: %s", capability.c_str());
      TTV_ASSERT(false);
    }
  }
}

void ttv::chat::ChatConnection::HandlePrivateMessage(const ChatNetworkEvent& evt) {
  const std::string& from = GetPrefixNick(evt.GetPrefix());

  // Deprecated, drop messages from jtv
  if (from == "jtv") {
    return;
  }

  if (mListener != nullptr) {
    std::string message = evt.GetParam(1);
    bool action = evt.GetEventID() == IRC_CTCP_ACTION;
    mListener->OnPrivateMessageReceived(this, from, message, evt.GetMessageTags(), action);
  }
}

void ttv::chat::ChatConnection::HandleUserState(const ChatNetworkEvent& evt) {
  if (mListener != nullptr) {
    mListener->OnUserStateChanged(this, evt.GetMessageTags());
  }
}

void ttv::chat::ChatConnection::HandleNotice(const ChatNetworkEvent& evt) {
  std::string text;
  if (evt.GetParamCount() >= 2) {
    text = evt.GetParam(1);
  }

  const auto& messageTags = evt.GetMessageTags();
  auto messageIdItr = messageTags.find("msg-id");

  // user banned
  if (messageIdItr != messageTags.end() && messageIdItr->second == "msg_banned") {
    mConnectionError = TTV_EC_BANNED_USER;

    SetState(CONNECTIONSTATE_CONNECTION_FAILED);
  }

  // invalid login
  if (mConnectionState == CONNECTIONSTATE_WELCOMING) {
    // TODO: This needs to be revisited when we update IRC to v3.  We need to handle bad oauth tokens and server errors
    // differently.
    TTV_ASSERT(text == "Login unsuccessful" || text == "Error logging in" || text == "Login authentication failed");

    mConnectionError = TTV_EC_INVALID_LOGIN;

    SetState(CONNECTIONSTATE_CONNECTION_FAILED);
  }
  // @id=whisper_rate_limit :tmi.twitch.tv NOTICE #mattyurka3 :english phrase here
  else {
    std::string id;
    std::map<std::string, std::string> params = evt.GetMessageTags();
    params["_defaultText"] = text;

    // an extended NOTICE that supports app localization
    auto idIter = params.find("msg-id");
    if (idIter != params.end()) {
      id = idIter->second;
      params.erase("msg-id");
    }

    if (mListener != nullptr) {
      mListener->OnNoticeReceived(this, id, params);
    }
  }
}

void ttv::chat::ChatConnection::HandleClearChatMessage(const ChatNetworkEvent& evt) {
  // clear the entire chat room
  if (evt.GetParamCount() == 1) {
    if (mListener != nullptr) {
      mListener->OnCleared(this, "", evt.GetMessageTags());
    }
  }
  // clear the history of a specific user
  else if (evt.GetParamCount() == 2) {
    const std::string& username = evt.GetParam(1);
    if (mListener != nullptr) {
      mListener->OnCleared(this, username, evt.GetMessageTags());
    }
  }
}

void ttv::chat::ChatConnection::HandleHostTargetMessage(const ChatNetworkEvent& evt) {
  // :tmi.twitch.tv HOSTTARGET #channel :{target} {num_viewers}
  // where target can be a "-" to indicate unhosting

  if (evt.GetParamCount() < 2) {
    ttv::trace::Message("Chat", MessageLevel::Debug, "Unhandled HOSTTARGET format");
    return;
  }

  std::vector<std::string> tokens;
  Split(evt.GetParam(1), tokens, ' ', false);

  std::string target;
  uint32_t numViewers = 0;

  // target
  if (tokens.size() > 0) {
    target = tokens[0];
  }

  // num_viewers
  if (tokens.size() > 1) {
    ParseNum(tokens[1], numViewers);
  }

  // notify the client
  if (mListener != nullptr) {
    mListener->OnHostTargetChanged(this, target, numViewers);
  }
}

void ttv::chat::ChatConnection::HandleUserNotice(const ttv::chat::ChatNetworkEvent& evt) {
  if (mListener != nullptr) {
    const std::string& message = (evt.GetParamCount() >= 2) ? evt.GetParam(1) : "";
    mListener->OnUserNoticeReceived(this, message, evt.GetMessageTags());
  }
}

void ttv::chat::ChatConnection::HandleDeleteChatMessage(const ChatNetworkEvent& evt) {
  if (evt.GetParamCount() < 2) {
    ttv::trace::Message("Chat", MessageLevel::Debug, "Unhandled CLEARMSG format");
    return;
  }

  if (mListener != nullptr) {
    const std::map<std::string, std::string>& tags = evt.GetMessageTags();
    auto idIter = tags.find("target-msg-id");
    auto loginNameIter = tags.find("login");
    if (idIter != tags.end() && loginNameIter != tags.end() && mListener != nullptr) {
      std::string messageId = idIter->second;
      std::string loginName = loginNameIter->second;
      std::string messageContent = evt.GetParam(1);
      mListener->OnMessageDeleted(this, std::move(messageId), std::move(loginName), std::move(messageContent));
    }
  }
}
