/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chathelpers.h"

#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"

namespace {
// The colors used by users which have not explicitly set their color.  Copied from the web codebase.
const uint32_t kDefaultColors[] = {0xffff0000, 0xff0000ff, 0xff008000, 0xffb22222, 0xffff7f50, 0xff9acd32, 0xffff4500,
  0xff2e8b57, 0xffdaa520, 0xffd2691e, 0xff5f9ea0, 0xff1e90ff, 0xffff69b4, 0xff8a2be2, 0xff00ff7f};
const size_t kNumDefaultColors = sizeof(kDefaultColors) / sizeof(kDefaultColors[0]);
}  // namespace

ttv::chat::ChatMessagePacer::ChatMessagePacer() : mNextMessageSubmissionIndex(0) {
  Clear();
}

bool ttv::chat::ChatMessagePacer::TrackMessage() {
  // check to see if messages are being submitted too quickly
  // TODO: verify that this rate is still accurate - I think I've heard of the occasional developer being blacklisted
  // even with this check
  uint64_t oldestTime =
    mMessageSubmissionTimes[mNextMessageSubmissionIndex];  // the next one in the ring buffer is the oldest one
  uint64_t now = ttv::SystemTimeToMs(ttv::GetSystemClockTime());
  if (now - oldestTime < kMessagePacingTimeSpanSeconds * 1000) {
    return false;
  }

  // update the message time tracking
  mMessageSubmissionTimes[mNextMessageSubmissionIndex] = now;
  mNextMessageSubmissionIndex = (mNextMessageSubmissionIndex + 1) % mMessageSubmissionTimes.size();

  return true;
}

void ttv::chat::ChatMessagePacer::Clear() {
  mNextMessageSubmissionIndex = 0;
  mMessageSubmissionTimes.fill(0);
}

ttv::Color ttv::chat::GetRandomUserColor(const std::string& username) {
  static std::hash<std::string> mNameHasher;

  std::string lower = ToLowerCase(username);

  // Create a stable random user color
  size_t index = mNameHasher(lower) % kNumDefaultColors;
  return kDefaultColors[index];
}

ttv::chat::UserMode ttv::chat::ParseUserType(const std::string& str) {
  UserMode mode;
  if (str == "mod") {
    mode.moderator = true;
  } else if (str == "global_mod") {
    mode.globalModerator = true;
  } else if (str == "admin") {
    mode.administrator = true;
  } else if (str == "staff") {
    mode.staff = true;
  }
  return mode;
}

void ttv::chat::UnescapeEmoticonToken(std::string& token) {
  /**
   * Using proper unescaping code doesn't work for a few reasons (I tried)
   *  - The server is escaping the JSON data as HTML for some reason
   *  - If you HTML unescape first you will not catch &lt\;
   *  - If you backspace unescape first you will break B-?\) and produces a broken regex
   * The easiest thing to do for now is what has been working on Android for a while and just replace a select few
   * substrings
   */

  const char* table[] = {
    "&lt\\;",
    "<",
    "&gt\\;",
    ">",
    "&amp\\;",
    "&",
    "&apos\\;",
    "'",
    "&quot\\;",
    "\"",
  };

  for (size_t j = 0; j < sizeof(table) / sizeof(table[0]); j += 2) {
    for (;;) {
      size_t pos = token.find(table[j]);
      if (pos == std::string::npos) {
        break;
      }

      token.replace(pos, strlen(table[j]), table[j + 1]);
    }
  }
}

bool ttv::chat::CompareEmoticonId(const std::string& lhs, const std::string& rhs) {
  auto lhsi = std::strtoul(lhs.c_str(), nullptr, 10);
  auto rhsi = std::strtoul(rhs.c_str(), nullptr, 10);

  if (lhsi == rhsi) {
    // if number value is the same compare by string
    return lhs < rhs;
  }
  // else compare by numeric value
  return lhsi < rhsi;
}

bool ttv::chat::ParseEmoticonId(const ttv::json::Value& data, const std::string& key, std::string& out) {
  if (data.isNull() || !data.isMember(key)) {
    return false;
  }

  if (ttv::ParseString(data, key.c_str(), out)) {
    return true;
  }

  uint64_t emoteId;
  if (ttv::ParseUInt(data, key.c_str(), emoteId)) {
    out = std::to_string(emoteId);
    return true;
  }

  return false;
}
