/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/chatmessagehandler.h"

#include "twitchsdk/chat/internal/ircstring.h"

namespace {
const char* kLogger = "ChatMessageHandler";
}

ttv::chat::ChatMessageHandler::ChatMessageHandler() {}

bool ttv::chat::ChatMessageHandler::HandleMessage(const std::string& message) {
  if (mCallbacks == nullptr) {
    return false;
  }

  uint32_t messageIndexBegin = 0;
  uint32_t messageIndexEnd = 0;

  std::string commandWord = GetWord(message, messageIndexBegin, false, messageIndexEnd);

  if (strcasecmp(commandWord.c_str(), "/block") == 0 || strcasecmp(commandWord.c_str(), "/ignore") == 0) {
    messageIndexBegin = messageIndexEnd;
    std::string blockedUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

    // Extra parameters are ignored
    if (!blockedUserName.empty()) {
      return mCallbacks->BlockUser(blockedUserName);
    } else {
      ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /ignore command");
    }
  } else if (strcasecmp(commandWord.c_str(), "/unignore") == 0) {
    messageIndexBegin = messageIndexEnd;
    std::string unblockedUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

    // Extra parameters are ignored
    if (!unblockedUserName.empty()) {
      return mCallbacks->UnblockUser(unblockedUserName);
    } else {
      ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /unignore command");
    }
  } else if (strcasecmp(commandWord.c_str(), "/w") == 0) {
    messageIndexBegin = messageIndexEnd;
    std::string whisperUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

    messageIndexBegin = messageIndexEnd;
    std::string whisperMessage = GetWord(message, messageIndexBegin, true, messageIndexEnd);

    if (!whisperUserName.empty() && !whisperMessage.empty()) {
      return mCallbacks->WhisperUser(whisperUserName, whisperMessage);
    } else {
      ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /w command");
    }
  } else if (strcasecmp(commandWord.c_str(), "/raid") == 0) {
    messageIndexBegin = messageIndexEnd;
    std::string targetUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

    // Extra parameters are ignored
    if (!targetUserName.empty()) {
      return mCallbacks->CreateRaid(targetUserName);
    } else {
      ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /raid command");
    }
  } else if (strcasecmp(commandWord.c_str(), "/unraid") == 0) {
    return mCallbacks->CancelRaid();
  } else if (strcasecmp(commandWord.c_str(), "/vip") == 0) {
    messageIndexBegin = messageIndexEnd;
    std::string vipUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

    messageIndexBegin = messageIndexEnd;

    if (!vipUserName.empty()) {
      return mCallbacks->GrantVIP(vipUserName);
    } else {
      ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /vip command");
    }
  } else if (strcasecmp(commandWord.c_str(), "/unvip") == 0) {
    messageIndexBegin = messageIndexEnd;
    std::string unvipUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

    messageIndexBegin = messageIndexEnd;

    if (!unvipUserName.empty()) {
      return mCallbacks->RevokeVIP(unvipUserName);
    } else {
      ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /unvip command");
    }
  } else if (strcasecmp(commandWord.c_str(), "/vips") == 0) {
    return mCallbacks->ListVIPs();
  }

  return mCallbacks->PassThrough(message);
}
