/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatnetworkevent.h"

#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/stringutilities.h"

#include <stdlib.h>

/////////////////////////////////////////////////////////////////////////////
// Known Command Tables
/////////////////////////////////////////////////////////////////////////////

static const char* COMMAND_EVENT_STRINGS[] = {
  "NICK",
  "USER",
  "PASS",
  "JOIN",
  "PART",
  "KICK",
  "INVITE",
  "PRIVMSG",
  "NOTICE",
  "MODE",
  "PING",
  "PONG",
  "TOPIC",
  "WHOIS",
  "WHOWAS",
  "NAMES",
  "LIST",
  "USERHOST",
  "QUIT",
  "AWAY",
  "ERROR",
  "TWITCHCLIENT 2",
  "CAP",
  "USERSTATE",
  "GLOBALUSERSTATE",
  "ROOMSTATE",
  "CLEARCHAT",
  "HOSTTARGET",
  "USERNOTICE",
  "CLEARMSG",
  "UNKNOWN",
};

static const char* CTCP_EVENT_STRINGS[] = {
  "ACTION",
  "PING",
  "VERSION",
  "DCC",
  "CLEARCHAT",
  "UNKNOWN",
};

/////////////////////////////////////////////////////////////////////////////
// File Private Utilities
/////////////////////////////////////////////////////////////////////////////

static int GetStringID(const char** ppszList, const char* psz, int iLast) {
  TTV_ASSERT(ppszList != nullptr);

  for (int i = 0; i < iLast; ++i) {
    TTV_ASSERT(ppszList[i] != nullptr);

    if (strcmp(psz, ppszList[i]) == 0) {
      return i;
    }
  }

  return iLast;
}

/////////////////////////////////////////////////////////////////////////////
// Static Utilities
/////////////////////////////////////////////////////////////////////////////

std::string ttv::chat::ChatNetworkEvent::EventIDToString(int idEvent) {
  std::string cmd;

  if (idEvent >= EVENT_OFFSET_COMMAND && idEvent <= IRC_CMD_UNKNOWN) {
    cmd = COMMAND_EVENT_STRINGS[idEvent - EVENT_OFFSET_COMMAND];
  } else if (idEvent >= EVENT_OFFSET_CTCP && idEvent <= IRC_CTCP_UNKNOWN) {
    cmd = CTCP_EVENT_STRINGS[idEvent - EVENT_OFFSET_CTCP];
  } else if (idEvent >= EVENT_OFFSET_CTCP_REPLY && idEvent <= IRC_CTCP_RPL_UNKNOWN) {
    cmd = CTCP_EVENT_STRINGS[idEvent - EVENT_OFFSET_CTCP_REPLY];
  } else if (idEvent > 0) {
    char buf[20];
    (void)snprintf(buf, sizeof(buf), "%03d", idEvent);

    cmd = buf;
  }
  return cmd;
}

int ttv::chat::ChatNetworkEvent::EventStringToID(const std::string& evt) {
  TTV_ASSERT(evt.c_str() != nullptr);

  int idEvent = IRC_CMD_INVALID;
  if (evt.size()) {
    const int idUnknown = sizeof(COMMAND_EVENT_STRINGS) / sizeof(COMMAND_EVENT_STRINGS[0]) - 1;
    int idString = GetStringID(COMMAND_EVENT_STRINGS, evt.c_str(), idUnknown);

    if (idString == idUnknown) {
      char* end = nullptr;
      int idParsed = static_cast<int>(strtol(evt.c_str(), &end, 10));
      if (*end == 0 && idParsed > 0) {
        idEvent = idParsed;
      } else {
        idEvent = IRC_CMD_UNKNOWN;
      }
    } else {
      idEvent = idString + EVENT_OFFSET_COMMAND;
    }
  }
  return idEvent;
}

int ttv::chat::ChatNetworkEvent::CTCPEventStringToID(const std::string& evt, bool bReply) {
  TTV_ASSERT(evt.c_str() != nullptr);

  int idEvent = IRC_CTCP_UNKNOWN;
  if (evt.size()) {
    const int idUnknown = sizeof(CTCP_EVENT_STRINGS) / sizeof(CTCP_EVENT_STRINGS[0]) - 1;
    int idString = GetStringID(CTCP_EVENT_STRINGS, evt.c_str(), idUnknown);

    if (bReply) {
      idEvent = idString + EVENT_OFFSET_CTCP_REPLY;
    } else {
      idEvent = idString + EVENT_OFFSET_CTCP;
    }
  }
  return idEvent;
}

bool ttv::chat::ChatNetworkEvent::IsReplyEvent(int idEvent) {
  return (idEvent > 0) && (idEvent < EVENT_OFFSET_COMMAND);
}

bool ttv::chat::ChatNetworkEvent::IsCTCPEvent(int idEvent) {
  return (idEvent >= EVENT_OFFSET_CTCP) && (idEvent <= IRC_CTCP_UNKNOWN);
}

bool ttv::chat::ChatNetworkEvent::IsCTCPReplyEvent(int idEvent) {
  return (idEvent >= EVENT_OFFSET_CTCP_REPLY) && (idEvent <= IRC_CTCP_RPL_UNKNOWN);
}

bool ttv::chat::ChatNetworkEvent::IsCommandEvent(int idEvent) {
  return (idEvent >= EVENT_OFFSET_COMMAND) && (idEvent <= IRC_CMD_UNKNOWN);
}

bool ttv::chat::ChatNetworkEvent::IsSystemEvent(int idEvent) {
  return (idEvent >= EVENT_OFFSET_SYSTEM) && (idEvent <= SYS_EVENT_UNKNOWN);
}

/////////////////////////////////////////////////////////////////////////////
// Instance Methods
/////////////////////////////////////////////////////////////////////////////

ttv::chat::ChatNetworkEvent::ChatNetworkEvent() {
  mEventId = 0;
  mIncoming = false;
  mAutoPrefix = true;
}

ttv::chat::ChatNetworkEvent::ChatNetworkEvent(int idEvent, int nParams, ...) {
  mEventId = idEvent;
  mIncoming = false;
  mAutoPrefix = true;

  va_list va;
  va_start(va, nParams);

  for (int i = 0; i < nParams; ++i) {
    std::string* psParam = va_arg(va, std::string*);
    TTV_ASSERT(psParam != nullptr);

    if (psParam) {
      AddParam(*psParam);
    }
  }
  va_end(va);
}

ttv::chat::ChatNetworkEvent::~ChatNetworkEvent() {}

void ttv::chat::ChatNetworkEvent::SetEvent(const std::string& evt) {
  mEventName = evt;
}

const std::string& ttv::chat::ChatNetworkEvent::GetEvent() const {
  return mEventName;
}

void ttv::chat::ChatNetworkEvent::SetEventID(int idEvent) {
  mEventId = idEvent;
}

int ttv::chat::ChatNetworkEvent::GetEventID() const {
  return mEventId;
}

void ttv::chat::ChatNetworkEvent::AddParam(const std::string& sParam) {
  mParams.push_back(sParam);
}

const std::string& ttv::chat::ChatNetworkEvent::GetParam(unsigned int iParam) const {
  TTV_ASSERT(iParam < GetParamCount());
  if (iParam < GetParamCount()) {
    return mParams[iParam];
  }

  static const std::string empty;
  return empty;
}

size_t ttv::chat::ChatNetworkEvent::GetParamCount() const {
  return mParams.size();
}

void ttv::chat::ChatNetworkEvent::SetPrefix(const std::string& prefix) {
  mPrefix = prefix;
}

const std::string& ttv::chat::ChatNetworkEvent::GetPrefix() const {
  return mPrefix;
}

void ttv::chat::ChatNetworkEvent::SetMessageTags(const std::string& raw) {
  std::vector<std::string> tags;
  Split(raw, tags, ';', false);

  for (auto& tag : tags) {
    Trim(tag);

    std::vector<std::string> kvp;
    ttv::Split(tag, kvp, '=', false);

    if (kvp.size() == 0) {
      continue;
    }

    std::string key = kvp[0];
    Trim(key);

    std::string value;
    if (kvp.size() > 1) {
      value = kvp[1];
      Trim(value);
      value = UnescapeMessageTag(value);
    }

    mMessageTags[key] = value;
  }
}

const std::map<std::string, std::string>& ttv::chat::ChatNetworkEvent::GetMessageTags() const {
  return mMessageTags;
}

void ttv::chat::ChatNetworkEvent::SetIncoming(bool bIncoming) {
  mIncoming = bIncoming;
}

bool ttv::chat::ChatNetworkEvent::IsIncoming() const {
  return mIncoming;
}

void ttv::chat::ChatNetworkEvent::SetAutoPrefix(bool bAutoPrefix) {
  mAutoPrefix = bAutoPrefix;
}

bool ttv::chat::ChatNetworkEvent::GetAutoPrefix() const {
  return mAutoPrefix;
}
