/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatraid.h"

#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/chat/internal/task/chatraidtask.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/user/user.h"

namespace {
const char* kLoggerName = "ChatRaid";
const char* kTopicPrefix = "raid.";

inline bool ends_with(const std::string& value, const std::string& ending) {
  if (ending.size() > value.size()) {
    return false;
  }
  return std::equal(ending.rbegin(), ending.rend(), value.rbegin());
}
}  // namespace

ttv::chat::ChatRaid::ChatRaid(const std::shared_ptr<User>& user, ChannelId channelId)
    : PubSubComponent(user), mPubSubTopic(kTopicPrefix + std::to_string(channelId)), mSourceChannelId(channelId) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::ChatRaid::GetLoggerName() const {
  return kLoggerName;
}

void ttv::chat::ChatRaid::CompleteShutdown() {
  UserComponent::CompleteShutdown();

  // Cancel any outstanding raids
  if (mListener != nullptr) {
    for (const auto& kvp : mActiveRaids) {
      mListener->RaidCancelled(kvp.second);
    }
  }

  mActiveRaids.clear();

  mListener.reset();
  mPubSub.reset();
  mPubSubTopicListener.reset();
}

TTV_ErrorCode ttv::chat::ChatRaid::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatRaid::Join(const std::string& raidId, const JoinCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  // Need to be logged in to take part in raids
  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRaidTask>(mSourceChannelId, oauthToken->GetToken(),
    [this, user, oauthToken, raidId, callback](ChatRaidTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (TTV_SUCCEEDED(ec)) {
        auto iter = mActiveRaids.find(raidId);
        if (iter != mActiveRaids.end()) {
          iter->second.joined = true;

          if (mListener != nullptr) {
            mListener->RaidUpdated(iter->second);
          }
        }
      } else if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  task->Join(raidId);

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatRaid::Leave(const std::string& raidId, const LeaveCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  // Need to be logged in to take part in raids
  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRaidTask>(mSourceChannelId, oauthToken->GetToken(),
    [this, user, oauthToken, raidId, callback](ChatRaidTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (TTV_SUCCEEDED(ec)) {
        auto iter = mActiveRaids.find(raidId);
        if (iter != mActiveRaids.end()) {
          iter->second.joined = false;

          if (mListener != nullptr) {
            mListener->RaidUpdated(iter->second);
          }
        }
      } else if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  task->Leave(raidId);

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatRaid::Start(UserId targetUserId, const StartCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  // Need to be logged in to manage raids
  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRaidTask>(mSourceChannelId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRaidTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  task->Start(targetUserId);

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatRaid::RaidNow(const StartCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  // Need to be logged in to manage raids
  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRaidTask>(mSourceChannelId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRaidTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  task->RaidNow();

  return StartTask(task);
}

TTV_ErrorCode ttv::chat::ChatRaid::Cancel(const CancelCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  // Need to be logged in to manage raids
  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRaidTask>(mSourceChannelId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRaidTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  task->Cancel(mSourceChannelId);

  return StartTask(task);
}

void ttv::chat::ChatRaid::HandleRaidStatus(RaidStatus& status, const std::string& eventType) {
  auto iter = mActiveRaids.find(status.raidId);

  if (eventType == "raid_cancel_v2") {
    if (iter != mActiveRaids.end()) {
      if (mListener != nullptr) {
        status.joined = iter->second.joined;
        mListener->RaidCancelled(status);
      }

      mActiveRaids.erase(iter);
    }

    mCompletedRaids.insert(status.raidId);
  } else if (eventType == "raid_update_v2") {
    // Prevent race condition where we receive an update after a raid
    // has already been cancelled / fired.
    if (mCompletedRaids.find(status.raidId) == mCompletedRaids.end()) {
      // New raid
      if (iter == mActiveRaids.end()) {
        mActiveRaids[status.raidId] = status;

        if (mListener != nullptr) {
          mListener->RaidStarted(status);
        }
      }
      // Already exists
      else {
        status.joined = iter->second.joined;

        if (status != iter->second) {
          iter->second = status;

          if (mListener != nullptr) {
            mListener->RaidUpdated(status);
          }
        }
      }
    }
  } else if (eventType == "raid_go_v2") {
    if (iter != mActiveRaids.end()) {
      if (mListener != nullptr) {
        status.joined = iter->second.joined;
        mListener->RaidFired(status);
      }

      mActiveRaids.erase(iter);
    }

    mCompletedRaids.insert(status.raidId);
  }
}

void ttv::chat::ChatRaid::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum /*state*/, TTV_ErrorCode /*ec*/) {
  // NOOP
}

void ttv::chat::ChatRaid::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    const auto& jType = msg["type"];
    if (jType.isNull() || !jType.isString()) {
      Log(MessageLevel::Error, "No 'type' field, ignoring");
      return;
    }

    if (ends_with(jType.asString(), "_v2")) {
      const auto& jRaid = msg["raid"];
      if (jRaid.isNull() || !jRaid.isObject()) {
        Log(MessageLevel::Error, "No 'raid' field, ignoring");
        return;
      }

      RaidStatus status;
      if (!ParseRaidStatusJson(jRaid, status)) {
        Log(MessageLevel::Error, "Failed to parse raid status, ignoring");
        return;
      }

      HandleRaidStatus(status, jType.asString());
    }
  }
}
