/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatreader.h"

#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/stringutilities.h"

ttv::chat::ChatReader::ChatReader() : mNotify(nullptr), mLineLength(0) {
  mLineBuffer.resize(IChatTransport::kMaxBufferSize);
}

ttv::chat::ChatReader::~ChatReader() {
  mNotify = nullptr;
}

void ttv::chat::ChatReader::OnConnect(TTV_ErrorCode err, const char* error) {
  ChatNetworkEvent evt;

  if (err != TTV_EC_SUCCESS) {
    evt.SetEventID(SYS_EVENT_CONNECTFAILED);
  } else {
    evt.SetEventID(SYS_EVENT_CONNECTED);
    evt.AddParam(error);
  }

  if (mNotify) {
    mNotify->ReceiveEvent(evt);
  }
}

void ttv::chat::ChatReader::OnError(TTV_ErrorCode /*err*/) {
  assert(mNotify);
  if (!mNotify) {
    return;
  }

  // TODO: get a real error string here
  const char* buf = "";
  ChatNetworkEvent evt(SYS_EVENT_ERROR, 1, buf);
  mNotify->ReceiveEvent(evt);
}

void ttv::chat::ChatReader::OnClose() {
  TTV_ASSERT(mNotify);
  if (!mNotify) {
    return;
  }

  ChatNetworkEvent evt(SYS_EVENT_CLOSE, 0);
  mNotify->ReceiveEvent(evt);
}

void ttv::chat::ChatReader::OnRead(const char* pData, size_t nSize) {
  TTV_ASSERT(pData);
  TTV_ASSERT(mNotify);

  if (!pData) {
    return;
  }

  for (size_t i = 0; i < nSize; ++i) {
    switch (pData[i]) {
      // ignore
      case '\r': {
        break;
      }
      // null shouldn't happen but terminate the line here
      case '\0': {
        OnLineRead(mLineBuffer.data());
        mLineLength = 0;
        break;
      }
      // end of the line
      case '\n': {
        mLineBuffer[mLineLength] = '\0';
        OnLineRead(mLineBuffer.data());
        mLineLength = 0;
        break;
      }
      // copy into the buffer
      default: {
        if (mLineLength < mLineBuffer.size()) {
          mLineBuffer[mLineLength] = pData[i];
          mLineLength++;
        }
        break;
      }
    }
  }
}

void ttv::chat::ChatReader::OnLineRead(const char* line) {
  ttv::trace::Message("ChatTransport", MessageLevel::Debug, "ChatReader::OnLineRead: Received raw line: %s", line);

  ChatNetworkEvent evt;
  evt.SetIncoming(true);

  bool bParseOk = ParseEvent(evt, line);
  TTV_ASSERT(bParseOk);

  if (!bParseOk) {
    ttv::trace::Message("ChatTransport", MessageLevel::Debug, "ChatReader::OnLineRead: Failed to parse line: %s", line);
  } else if (mNotify) {
    mNotify->ReceiveEvent(evt);
  }
}

/////////////////////////////////////////////////////////////////////////////
//  Static Utility Methods
/////////////////////////////////////////////////////////////////////////////

bool ttv::chat::ChatReader::ParseEvent(ChatNetworkEvent& evt, const char* line) {
  TTV_ASSERT(line != nullptr);

  const char* seek = line;
  const char* tmp;

  // Retrieve first word into buffer
  // This is either the message tag, prefix or the command
  if (IrcStringGetWord(mParseBuffer, seek, sizeof(mParseBuffer), &tmp) == 0) {
    return false;
  }

  // Optional message tag
  if (mParseBuffer[0] == char('@')) {
    evt.SetMessageTags(mParseBuffer + 1);

    // Seek ahead to next token
    seek = IrcStringSkipSpaces(tmp);

    // Read the next token
    if (IrcStringGetWord(mParseBuffer, seek, sizeof(mParseBuffer), &tmp) == 0) {
      return false;
    }
  }

  // Optional prefix
  if (mParseBuffer[0] == char(':')) {
    evt.SetPrefix(mParseBuffer + 1);

    // Seek ahead to command
    seek = IrcStringSkipSpaces(tmp);
  }

  // Get next word, this should be the command
  if (IrcStringGetWord(mParseBuffer, seek, sizeof(mParseBuffer), &tmp) == 0) {
    return false;
  }
  seek = IrcStringSkipSpaces(tmp);

  int idEvent = ChatNetworkEvent::EventStringToID(mParseBuffer);
  evt.SetEventID(idEvent);
  evt.SetEvent(mParseBuffer);

  // Get parameters, if any
  while (IrcStringGetWord(mParseBuffer, seek, sizeof(mParseBuffer), &tmp) != 0) {
    if (mParseBuffer[0] == char(':')) {
      // If it's a PRIVMSG, check to see if it's CTCP, and if so
      // change the command id as well as splitting up the CTCP
      // parameters
      //
      // ps. replies to these commands come via NOTICE.  argh.
      if ((idEvent == IRC_CMD_PRIVMSG || idEvent == IRC_CMD_NOTICE) && mParseBuffer[1] == char('\x01')) {
        bool bReply = (idEvent == IRC_CMD_NOTICE);
        // Extract CTCP command
        (void)IrcStringGetWord(mParseBuffer, seek + 2, sizeof(mParseBuffer), &tmp);
        seek = IrcStringSkipSpaces(tmp);

        size_t len = strlen(mParseBuffer);
        TTV_ASSERT(len > 0);
        if (len < 1) {
          return false;
        }

        if (mParseBuffer[len - 1] == '\x01') {
          mParseBuffer[len - 1] = '\0';
          seek--;
        }

        // Copy CTCP command
        idEvent = ChatNetworkEvent::CTCPEventStringToID(mParseBuffer, bReply);
        evt.SetEventID(idEvent);
        evt.SetEvent(mParseBuffer);

        // Copy CTCP parameters into mParseBuffer, this MAY include a terminating \1
        SafeStringCopy(mParseBuffer, seek, sizeof(mParseBuffer));
        len = strlen(mParseBuffer);

        // Replace \1 terminator with nullptr
        if (len > 0 && mParseBuffer[len - 1] == '\x01') {
          mParseBuffer[len - 1] = '\0';
          len--;
        } else {
          //_TRACE("ttv::chat::ChatReader::ParseEvent() CTCP MISSING TERMINATOR");
        }

        if (len > 0) {
          // Copy final CTCP parameters
          evt.AddParam(mParseBuffer);
        }
      } else {
        // Parameter is the entire rest of the line
        // Copy from seek because mParseBuffer only contains a single word
        evt.AddParam(seek + 1);
      }
      break;
    } else {
      // Get next paramter
      evt.AddParam(mParseBuffer);
      seek = IrcStringSkipSpaces(tmp);
    }
  }

  // Special case, NOTICE AUTH at login.
  if (idEvent == IRC_CMD_NOTICE && evt.GetPrefix().size() == 0 && evt.GetParamCount() > 0) {
    evt.SetPrefix(evt.GetParam(0));
  }

  return true;
}
