/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatroom.h"

#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/internal/graphql/sendroommessagequeryinfo.h"
#include "twitchsdk/chat/internal/task/chatroomdeletemessagetask.h"
#include "twitchsdk/chat/internal/task/chatroomdeleteroomtask.h"
#include "twitchsdk/chat/internal/task/chatroomeditmessagetask.h"
#include "twitchsdk/chat/internal/task/chatroomfetchinfotask.h"
#include "twitchsdk/chat/internal/task/chatroomfetchmessagestask.h"
#include "twitchsdk/chat/internal/task/chatroomupdateinfotask.h"
#include "twitchsdk/chat/internal/task/chatroomupdatemodetask.h"
#include "twitchsdk/chat/internal/task/chatroomupdateviewtask.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/graphqltask.h"

namespace {
const char* kLoggerName = "ChatRooms";
const char* kTopicPrefix = "chatrooms-room-v1.";
}  // namespace

ttv::chat::ChatRoom::ChatRoom(const std::shared_ptr<User>& user, const std::string& roomId, ChannelId channelId)
    : PubSubComponent(user),
      mPubSubTopic(kTopicPrefix + roomId),
      mRoomId(roomId),
      mChannelId(channelId),
      mCachedUserColor(0xFFC0C0C0)  //!< Default chat color value for the first local message, #C0C0C0
{
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::ChatRoom::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::ChatRoom::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatRoom::Update() {
  PubSubComponent::Update();

  if (!mMessageNonces.empty()) {
    uint64_t currentTime = GetSystemTimeMilliseconds();

    // Remove any old nonces over a minute if pubsub never fired
    auto iter = mMessageNonces.begin();

    while (iter != mMessageNonces.end()) {
      if (currentTime - iter->second > 60 * 1000) {
        iter = mMessageNonces.erase(iter);
      } else {
        ++iter;
      }
    }
  }
}

TTV_ErrorCode ttv::chat::ChatRoom::DeleteRoom(const DeleteRoomCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomDeleteRoomTask>(mRoomId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomDeleteRoomTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::SendMessage(
  const std::string& messageContent, ChatRoomMessage& placeholderMessage, const SendMessageCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  graphql::SendRoomMessageQueryInfo::InputParams inputs;
  inputs.authToken = oauthToken->GetToken();
  inputs.message = messageContent;
  inputs.nonce = GetGuid();
  inputs.roomId = mRoomId;
  inputs.channelId = std::to_string(mChannelId);

  mMessageNonces[inputs.nonce] = GetSystemTimeMilliseconds();

  auto task = std::make_shared<GraphQLTask<graphql::SendRoomMessageQueryInfo>>(
    std::move(inputs), [this, user, oauthToken, callback](GraphQLTask<graphql::SendRoomMessageQueryInfo>* source,
                         Result<graphql::SendRoomMessageQueryInfo::PayloadType>&& result) {
      CompleteTask(source);

      TTV_ErrorCode ec = TTV_EC_SUCCESS;
      ChatRoomMessage message;
      SendRoomMessageError error;

      if (result.IsError()) {
        ec = result.GetErrorCode();
        if (ec == TTV_EC_AUTHENTICATION) {
          user->ReportOAuthTokenInvalid(oauthToken, ec);
        }
      } else {
        auto& payload = result.GetResult();
        message = std::move(payload.message);
        error = std::move(payload.error);

        mCachedUserBadges = message.messageInfo.badges;
        mCachedUserColor = message.messageInfo.nameColorARGB;

        if (error.code != GraphQLErrorCode::SUCCESS) {
          ec = TTV_EC_GRAPHQL_ERROR;
        } else if (message.roomMessageId.empty()) {
          // Received a payload without a message or an error
          ec = TTV_EC_INVALID_GRAPHQL;
        }
      }

      if (callback != nullptr) {
        callback(ec, std::move(error), std::move(message));
      }
    });

  TTV_ErrorCode ec = StartTask(task);

  if (TTV_SUCCEEDED(ec)) {
    ec = TokenizeLocalMessage(user, messageContent, placeholderMessage);
  } else {
    mMessageNonces.erase(inputs.nonce);
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::EditMessage(const std::string& messageId, const std::string& messageContent,
  ChatRoomMessage& placeholderMessage, const EditMessageCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task =
    std::make_shared<ChatRoomEditMessageTask>(mRoomId, mChannelId, messageId, messageContent, oauthToken->GetToken(),
      [this, user, oauthToken, callback](ChatRoomEditMessageTask* source, TTV_ErrorCode ec, ChatRoomMessage&& message) {
        CompleteTask(source);

        if (ec == TTV_EC_AUTHENTICATION) {
          user->ReportOAuthTokenInvalid(oauthToken, ec);
        }

        if (TTV_SUCCEEDED(ec)) {
          mCachedUserBadges = message.messageInfo.badges;
          mCachedUserColor = message.messageInfo.nameColorARGB;
        }

        if (callback != nullptr) {
          callback(ec, std::move(message));
        }
      });

  TTV_ErrorCode ec = StartTask(task);

  if (TTV_SUCCEEDED(ec)) {
    ec = TokenizeLocalMessage(user, messageContent, placeholderMessage);
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::DeleteMessage(const std::string& messageId, const DeleteMessageCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomDeleteMessageTask>(mRoomId, messageId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomDeleteMessageTask* source, TTV_ErrorCode ec) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec);
      }
    });

  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::FetchMessagesBeforeCursor(
  const std::string& cursor, uint32_t limit, const FetchMessagesCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (limit == 0 || limit > 100) {
    return TTV_EC_INVALID_ARG;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomFetchMessagesTask>(mRoomId, mChannelId, false, limit, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomFetchMessagesTask* source, TTV_ErrorCode ec,
      std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        FlagIgnoredMessages(messages);
        callback(ec, std::move(messages), std::move(nextCursor), moreMessages);
      }
    });

  task->SetCursor(cursor);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::FetchMessagesAfterCursor(
  const std::string& cursor, uint32_t limit, const FetchMessagesCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (limit == 0 || limit > 100) {
    return TTV_EC_INVALID_ARG;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomFetchMessagesTask>(mRoomId, mChannelId, true, limit, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomFetchMessagesTask* source, TTV_ErrorCode ec,
      std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        FlagIgnoredMessages(messages);
        callback(ec, std::move(messages), std::move(nextCursor), moreMessages);
      }
    });

  task->SetCursor(cursor);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::FetchMessagesBeforeTimestamp(
  Timestamp timestamp, uint32_t limit, const FetchMessagesCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (limit == 0 || limit > 100) {
    return TTV_EC_INVALID_ARG;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomFetchMessagesTask>(mRoomId, mChannelId, false, limit, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomFetchMessagesTask* source, TTV_ErrorCode ec,
      std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        FlagIgnoredMessages(messages);
        callback(ec, std::move(messages), std::move(nextCursor), moreMessages);
      }
    });

  task->SetTime(timestamp);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::FetchMessagesAfterTimestamp(
  Timestamp timestamp, uint32_t limit, const FetchMessagesCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (limit == 0 || limit > 100) {
    return TTV_EC_INVALID_ARG;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomFetchMessagesTask>(mRoomId, mChannelId, true, limit, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomFetchMessagesTask* source, TTV_ErrorCode ec,
      std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        FlagIgnoredMessages(messages);
        callback(ec, std::move(messages), std::move(nextCursor), moreMessages);
      }
    });

  task->SetTime(timestamp);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::SetRoomName(const std::string& name, const UpdateRoomInfoCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomUpdateInfoTask>(mRoomId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](
      ChatRoomUpdateInfoTask* source, TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error), std::move(info));
      }
    });

  task->SetName(name);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::SetTopic(const std::string& topic, const UpdateRoomInfoCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomUpdateInfoTask>(mRoomId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](
      ChatRoomUpdateInfoTask* source, TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error), std::move(info));
      }
    });

  task->SetTopic(topic);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::SetRoomRolePermissions(
  RoomRolePermissions permissions, const UpdateRoomInfoCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (permissions.read == RoomRole::Unknown || permissions.send == RoomRole::Unknown) {
    return TTV_EC_INVALID_ARG;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomUpdateInfoTask>(mRoomId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](
      ChatRoomUpdateInfoTask* source, TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error), std::move(info));
      }
    });

  task->SetRoomRolePermissions(permissions);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::EnableSlowMode(uint32_t durationSeconds, const UpdateRoomModesCallback& callback) {
  return SetChatMode(ChatMode::Slow, true, durationSeconds, callback);
}

TTV_ErrorCode ttv::chat::ChatRoom::DisableSlowMode(const UpdateRoomModesCallback& callback) {
  return SetChatMode(ChatMode::Slow, false, 0, callback);
}

TTV_ErrorCode ttv::chat::ChatRoom::EnableR9kMode(const UpdateRoomModesCallback& callback) {
  return SetChatMode(ChatMode::R9k, true, 0, callback);
}

TTV_ErrorCode ttv::chat::ChatRoom::DisableR9kMode(const UpdateRoomModesCallback& callback) {
  return SetChatMode(ChatMode::R9k, false, 0, callback);
}

TTV_ErrorCode ttv::chat::ChatRoom::EnableEmotesOnlyMode(const UpdateRoomModesCallback& callback) {
  return SetChatMode(ChatMode::EmotesOnly, true, 0, callback);
}

TTV_ErrorCode ttv::chat::ChatRoom::DisableEmotesOnlyMode(const UpdateRoomModesCallback& callback) {
  return SetChatMode(ChatMode::EmotesOnly, false, 0, callback);
}

TTV_ErrorCode ttv::chat::ChatRoom::SetLastReadAt(Timestamp lastReadAt, const UpdateRoomViewCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomUpdateViewTask>(mRoomId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomUpdateViewTask* source, TTV_ErrorCode ec, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(info));
      }
    });

  task->SetLastReadAt(lastReadAt);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::SetMuted(bool isMuted, const UpdateRoomViewCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomUpdateViewTask>(mRoomId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomUpdateViewTask* source, TTV_ErrorCode ec, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(info));
      }
    });

  task->SetIsMuted(isMuted);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::SetArchived(bool isArchived, const UpdateRoomViewCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomUpdateViewTask>(mRoomId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomUpdateViewTask* source, TTV_ErrorCode ec, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(info));
      }
    });

  task->SetIsArchived(isArchived);
  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::FetchRoomInfo(const FetchRoomInfoCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomFetchInfoTask>(mRoomId, oauthToken->GetToken(),
    [this, user, oauthToken, callback](ChatRoomFetchInfoTask* source, TTV_ErrorCode ec, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(info));
      }
    });

  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

void ttv::chat::ChatRoom::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "ChatRoomListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::ChatRoom::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) {
  if (jVal.isNull() || !jVal.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    std::string type;

    if (!ParseString(jVal, "type", type)) {
      Log(MessageLevel::Error, "Couldn't find pubsub message type, dropping");
      return;
    }

    const auto& jData = jVal["data"];
    if (jData.isNull() || !jData.isObject()) {
      Log(MessageLevel::Error, "Pub sub message missing data, dropping");
      return;
    }

    if (type == "created_room_message" || type == "edited_room_message" || type == "deleted_room_message") {
      const auto& jMessage = jData["message"];
      if (jMessage.isNull() || !jMessage.isObject()) {
        return;
      }

      ChatRoomMessage message;

      if (!ttv::json::ObjectSchema<json::description::PubSubChatRoomMessage>::Parse(jMessage, message)) {
        Log(MessageLevel::Error, "Unable to serialize pub sub chat room message");
        return;
      }

      if (mListener != nullptr) {
        if (type == "created_room_message") {
          // Drop message if it's one that we sent
          std::string nonce;
          if (ParseString(jMessage, "nonce", nonce)) {
            auto iter = mMessageNonces.find(nonce);
            if (iter != mMessageNonces.end()) {
              mMessageNonces.erase(iter);
              return;
            }
          }

          // Flag message if from ignored user
          auto user = mUser.lock();
          if (user != nullptr) {
            std::shared_ptr<ChatUserBlockList> blockList =
              user->GetComponentContainer()->GetComponent<ChatUserBlockList>();
            if (blockList != nullptr) {
              message.messageInfo.flags.ignored = blockList->IsUserBlocked(message.messageInfo.userId);
            }
          }

          if (mListener != nullptr) {
            mListener->MessageReceived(mRoomId, std::move(message));
          }
        } else if (type == "edited_room_message") {
          if (mListener != nullptr) {
            mListener->MessageEdited(mRoomId, std::move(message));
          }
        } else if (type == "deleted_room_message") {
          if (mListener != nullptr) {
            mListener->MessageDeleted(mRoomId, std::move(message));
          }
        }
      }
    } else if (type == "updated_room") {
      const auto& jRoom = jData["room"];
      if (jRoom.isNull() || !jRoom.isObject()) {
        return;
      }

      ChatRoomInfo roomInfo;

      if (!ttv::json::ObjectSchema<json::description::PubSubChatRoomInfo>::Parse(jRoom, roomInfo)) {
        Log(MessageLevel::Error, "Unable to serialize pub sub chat room info");
        return;
      }

      if (mListener != nullptr) {
        mListener->RoomUpdated(std::move(roomInfo));
      }
    } else {
      Log(MessageLevel::Error, "Unrecognized pub-sub message type (%s), dropping", type.c_str());
      return;
    }
  }
}

TTV_ErrorCode ttv::chat::ChatRoom::SetChatMode(
  ChatMode mode, bool turnOn, uint32_t slowModeDurationSeconds, const UpdateRoomModesCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  auto task = std::make_shared<ChatRoomUpdateModeTask>(mRoomId, mode, turnOn, oauthToken->GetToken(),
    [this, user, oauthToken, callback](
      ChatRoomUpdateModeTask* source, TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
      CompleteTask(source);

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, std::move(error), std::move(info));
      }
    });

  if (mode == ChatMode::Slow && turnOn) {
    if (slowModeDurationSeconds > 0) {
      task->SetSlowModeDurationSeconds(slowModeDurationSeconds);
    } else {
      return TTV_EC_INVALID_ARG;
    }
  }

  TTV_ErrorCode ec = StartTask(task);

  return ec;
}

TTV_ErrorCode ttv::chat::ChatRoom::TokenizeLocalMessage(
  const std::shared_ptr<User>& user, const std::string& message, ChatRoomMessage& chatMessage) {
  // Create a placeholder message to pass through the pipe
  std::string emotesMessageTag = "";
  std::string badgesMessageTag = "";

  ttv::chat::TokenizeLocalMessage(user, mChannelId, message, emotesMessageTag, badgesMessageTag);

  std::string mePrefix = "/me ";

  if (StartsWith(message, mePrefix)) {
    ttv::chat::TokenizeServerMessage(
      message.substr(mePrefix.size()), mTokenizationOptions, emotesMessageTag, nullptr, {}, chatMessage.messageInfo);
    chatMessage.messageInfo.flags.action = true;
  } else {
    ttv::chat::TokenizeServerMessage(
      message, mTokenizationOptions, emotesMessageTag, nullptr, {}, chatMessage.messageInfo);
  }

  chatMessage.roomId = mRoomId;
  chatMessage.roomMessageId = "";
  chatMessage.messageInfo.userId = user->GetUserId();
  chatMessage.messageInfo.userName = user->GetUserName();
  chatMessage.messageInfo.displayName = user->GetDisplayName();
  chatMessage.messageInfo.nameColorARGB = mCachedUserColor;
  chatMessage.messageInfo.timestamp = GetCurrentTimeAsUnixTimestamp();
  chatMessage.messageInfo.numBitsSent = 0;

  if (mCachedUserBadges.empty()) {
    std::vector<std::pair<std::string, std::string>> badgePairs;
    if (ttv::chat::ParseBadgesMessageTag(badgesMessageTag, badgePairs)) {
      std::transform(badgePairs.begin(), badgePairs.end(), std::back_inserter(chatMessage.messageInfo.badges),
        [](const std::pair<std::string, std::string>& pair) {
          MessageBadge badge;
          badge.name = pair.first;
          badge.version = pair.second;
          return badge;
        });
    }
  } else {
    chatMessage.messageInfo.badges = mCachedUserBadges;
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatRoom::FlagIgnoredMessages(std::vector<ChatRoomMessage>& messages) {
  auto user = mUser.lock();
  if (user != nullptr) {
    std::shared_ptr<ChatUserBlockList> blockList = user->GetComponentContainer()->GetComponent<ChatUserBlockList>();
    if (blockList != nullptr) {
      for (auto& message : messages) {
        message.messageInfo.flags.ignored = blockList->IsUserBlocked(message.messageInfo.userId);
      }
    }
  }
}
