/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/chatroommessagehandler.h"

#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kLogger = "ChatRoomMessageHandler";
const uint32_t kDefaultSlowModeDuration = 120;
const uint32_t kMaxSlowModeDuration = 86400;
const uint32_t kDefaultTimeoutLength = 600;
const uint32_t kMaxTimeoutLength = 1209600;
}  // namespace

ttv::chat::ChatRoomMessageHandler::ChatRoomMessageHandler() {}

bool ttv::chat::ChatRoomMessageHandler::HandleMessage(const std::string& message) {
  if (mCallbacks == nullptr) {
    ttv::trace::Message(kLogger, MessageLevel::Info, "No callbacks were passed in");
    return false;
  }

  uint32_t messageIndexBegin = 0;
  uint32_t messageIndexEnd = 0;

  std::string commandWord = GetWord(message, messageIndexBegin, false, messageIndexEnd);

  if (commandWord[0] == '/') {
    if (strcasecmp(commandWord.c_str(), "/ban") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string bannedUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      // Extra parameters are ignored
      if (!bannedUserName.empty()) {
        return mCallbacks->BanUser(bannedUserName);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /ban command");
        return mCallbacks->MalformedCommand(CommandError::Ban, "/ban");
      }
    } else if (strcasecmp(commandWord.c_str(), "/color") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string color = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      // Extra parameters are ignored
      if (!color.empty()) {
        return mCallbacks->SetUserColor(color);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /color command");
        return mCallbacks->MalformedCommand(CommandError::Color, "/color");
      }
    } else if (strcasecmp(commandWord.c_str(), "/help") == 0) {
      return mCallbacks->Help();
    } else if (strcasecmp(commandWord.c_str(), "/ignore") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string ignoredUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      // Extra parameters are ignored
      if (!ignoredUserName.empty()) {
        return mCallbacks->BlockUser(ignoredUserName);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /ignore command");
        return mCallbacks->MalformedCommand(CommandError::Ignore, "/ignore");
      }
    } else if (strcasecmp(commandWord.c_str(), "/listrooms") == 0) {
      return mCallbacks->ListRooms();
    } else if (strcasecmp(commandWord.c_str(), "/mod") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string moddedUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      // Extra parameters are ignored
      if (!moddedUserName.empty()) {
        return mCallbacks->ModUser(moddedUserName);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /mod command");
        return mCallbacks->MalformedCommand(CommandError::Mod, "/mod");
      }
    } else if (strcasecmp(commandWord.c_str(), "/mods") == 0) {
      return mCallbacks->ListModerators();
    } else if (strcasecmp(commandWord.c_str(), "/timeout") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string timedoutUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      messageIndexBegin = messageIndexEnd;
      std::string timeoutLengthString = GetWord(message, messageIndexBegin, true, messageIndexEnd);
      uint32_t timeoutLength = 0;

      if (timeoutLengthString.empty()) {
        // If no duration was passed, the default is 600 seconds.
        timeoutLength = kDefaultTimeoutLength;
      } else {
        if (!ParseNum(timeoutLengthString, timeoutLength)) {
          // Failed to convert to an integer, force failure to call Error function.
          timeoutLength = 0;
        }
      }

      if (!timedoutUserName.empty() && timeoutLength > 0 && timeoutLength <= kMaxTimeoutLength) {
        return mCallbacks->TimeoutUser(timedoutUserName, timeoutLength);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /timeout command");
        return mCallbacks->MalformedCommand(CommandError::Timeout, "/timeout");
      }
    } else if (strcasecmp(commandWord.c_str(), "/topic") == 0) {
      messageIndexBegin = messageIndexEnd;

      while (messageIndexBegin < message.size() && IsWhitespace(message[messageIndexBegin])) {
        messageIndexBegin++;
      }
      std::string topic = message.substr(messageIndexBegin, std::string::npos);

      // Extra parameters are ignored
      if (!topic.empty()) {
        return mCallbacks->SetTopic(topic);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /topic command");
        return mCallbacks->MalformedCommand(CommandError::Topic, "/topic");
      }
    } else if (strcasecmp(commandWord.c_str(), "/unban") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string unbannedUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      // Extra parameters are ignored
      if (!unbannedUserName.empty()) {
        return mCallbacks->UnbanUser(unbannedUserName);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /unban command");
        return mCallbacks->MalformedCommand(CommandError::Unban, "/unban");
      }
    } else if (strcasecmp(commandWord.c_str(), "/unignore") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string unignoredUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      // Extra parameters are ignored
      if (!unignoredUserName.empty()) {
        return mCallbacks->UnblockUser(unignoredUserName);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /unignore command");
        return mCallbacks->MalformedCommand(CommandError::Unignore, "/unignore");
      }
    } else if (strcasecmp(commandWord.c_str(), "/untimeout") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string untimeoutUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      // Extra parameters are ignored
      if (!untimeoutUserName.empty()) {
        return mCallbacks->UntimeoutUser(untimeoutUserName);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /untimeout command");
        return mCallbacks->MalformedCommand(CommandError::Untimeout, "/untimeout");
      }
    } else if (strcasecmp(commandWord.c_str(), "/unmod") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string unmoddedUserName = GetWord(message, messageIndexBegin, false, messageIndexEnd);

      // Extra parameters are ignored
      if (!unmoddedUserName.empty()) {
        return mCallbacks->UnmodUser(unmoddedUserName);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /unmod command");
        return mCallbacks->MalformedCommand(CommandError::Unmod, "/unmod");
      }
    } else if (strcasecmp(commandWord.c_str(), "/slow") == 0) {
      messageIndexBegin = messageIndexEnd;
      std::string durationString = GetWord(message, messageIndexBegin, true, messageIndexEnd);
      uint32_t durationSeconds = 0;

      if (durationString.empty()) {
        durationSeconds = kDefaultSlowModeDuration;
      } else {
        if (!ParseNum(durationString, durationSeconds)) {
          // Failed to convert to an integer, force failure to call Error function.
          durationSeconds = 0;
        }
      }

      if (durationSeconds > 0 && durationSeconds <= kMaxSlowModeDuration) {
        return mCallbacks->SlowMode(true, durationSeconds);
      } else {
        ttv::trace::Message(kLogger, MessageLevel::Info, "Invalid /slow command");
        return mCallbacks->MalformedCommand(CommandError::Slow, "/slow");
      }
    } else if (strcasecmp(commandWord.c_str(), "/slowoff") == 0) {
      return mCallbacks->SlowMode(false, 0);
    } else if (strcasecmp(commandWord.c_str(), "/r9kbeta") == 0) {
      return mCallbacks->R9kMode(true);
    } else if (strcasecmp(commandWord.c_str(), "/r9kbetaoff") == 0) {
      return mCallbacks->R9kMode(false);
    } else if (strcasecmp(commandWord.c_str(), "/emoteonly") == 0) {
      return mCallbacks->EmotesOnlyMode(true);
    } else if (strcasecmp(commandWord.c_str(), "/emoteonlyoff") == 0) {
      return mCallbacks->EmotesOnlyMode(false);
    } else if (strcasecmp(commandWord.c_str(), "/me") == 0) {
      // Check if the /me command is malformed. If it's not, let it fall to PassThrough as a normal message.
      messageIndexBegin = messageIndexEnd;
      std::string remainingMessage = GetWord(message, messageIndexBegin, true, messageIndexEnd);
      if (remainingMessage.empty()) {
        return mCallbacks->MalformedCommand(CommandError::Me, "/me");
      }
    } else {
      return mCallbacks->MalformedCommand(CommandError::Unknown, commandWord);
    }
  }

  return mCallbacks->PassThrough(message);
}
