/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatroomnotifications.h"

#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"

namespace {
const char* kLoggerName = "ChatRoomsNotifications";
const char* kTopicPrefix = "chatrooms-user-v1.";
}  // namespace

ttv::chat::ChatRoomNotifications::ChatRoomNotifications(const std::shared_ptr<User>& user)
    : PubSubComponent(user),
      mPubSubTopic(kTopicPrefix + std::to_string(user->GetUserId())),
      mUserId(user->GetUserId()) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::ChatRoomNotifications::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::ChatRoomNotifications::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatRoomNotifications::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "ChatRoomNotificationsListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::ChatRoomNotifications::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) {
  if (jVal.isNull() || !jVal.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    std::string type;

    if (!ParseString(jVal, "type", type)) {
      Log(MessageLevel::Error, "Couldn't find pubsub message type, dropping");
      return;
    }

    const auto& jData = jVal["data"];
    if (jData.isNull() || !jData.isObject()) {
      Log(MessageLevel::Error, "Pub sub message missing data, dropping");
      return;
    }

    if (type == "updated_room_view") {
      const auto& jRoomView = jData["room_view"];
      if (jRoomView.isNull() || !jRoomView.isObject()) {
        return;
      }

      UserId userId;
      const auto& jUserId = jRoomView["user_id"];
      ParseUserId(jUserId, userId);
      if (userId != mUserId) {
        return;
      }

      const auto& jRoom = jRoomView["room"];
      if (jRoom.isNull() || !jRoom.isObject()) {
        return;
      }

      std::string roomId;
      if (!ParseString(jRoom, "room_id", roomId)) {
        return;
      }

      ChannelId ownerId;
      const auto& jChannelId = jRoom["owner_id"];
      if (!ParseChannelId(jChannelId, ownerId)) {
        return;
      }

      ChatRoomView roomView;
      if (!ttv::json::ObjectSchema<json::description::PubSubChatRoomView>::Parse(jRoomView, roomView)) {
        return;
      }

      if (mListener != nullptr) {
        mListener->RoomViewUpdated(mUserId, ownerId, roomId, std::move(roomView));
      }
    } else if (type == "user_mention") {
      RoomMentionInfo mentionInfo;
      if (!ttv::json::ToObject(jData, mentionInfo)) {
        return;
      }

      if (mListener != nullptr) {
        mListener->RoomMentionReceived(mUserId, std::move(mentionInfo));
      }
    } else if (type == "user_moderation_action") {
      std::string action;
      if (!ParseString(jData, "action", action)) {
        return;
      }

      UserId userId;
      const auto& jUserId = jData["target_id"];
      ParseUserId(jUserId, userId);
      if (userId != mUserId) {
        return;
      }

      ChannelId channelId;
      const auto& jChannelId = jData["channel_id"];
      if (!ParseChannelId(jChannelId, channelId)) {
        return;
      }

      if (mListener != nullptr) {
        if (action == "timeout") {
          Timestamp expiresAt;
          if (!ParseTimestamp(jData, "expires_at", expiresAt)) {
            return;
          }

          mListener->UserTimedOut(userId, channelId, expiresAt);
        } else if (action == "ban") {
          mListener->UserBanned(userId, channelId);
        } else if (action == "unban") {
          mListener->UserUnbanned(userId, channelId);
        }
      }
    } else {
      Log(MessageLevel::Error, "Unrecognized pub-sub message type (%s), dropping", type.c_str());
      return;
    }
  }
}
