/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatsession.h"

#include "twitchsdk/chat/internal/chatchannel.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"

ttv::chat::ChatSession::ChatSession()
    : mNick(""), mWriter(nullptr), mLastActivity(GetSystemClockTime()), mConnected(false) {
  ResetIdleCounter();
  InitDefaultServerCapabilities();
}

ttv::chat::ChatSession::~ChatSession() {
  mWriter = nullptr;
}

/////////////////////////////////////////////////////////////////////////////
// Interface
/////////////////////////////////////////////////////////////////////////////

void ttv::chat::ChatSession::SetWriter(std::shared_ptr<IChatWriteNetworkEvent> writer) {
  mWriter = writer;
}

void ttv::chat::ChatSession::SetNick(const std::string& nick) {
  mNick = nick;
}

void ttv::chat::ChatSession::AddEventHandler(IChatReceiveNetworkEvent* handler) {
  TTV_ASSERT(handler != nullptr);

  std::vector<IChatReceiveNetworkEvent*>::iterator i = std::find(mEventHandlers.begin(), mEventHandlers.end(), handler);
  if (i == mEventHandlers.end()) {
    mEventHandlers.push_back(handler);
  }
}

void ttv::chat::ChatSession::RemoveEventHandler(IChatReceiveNetworkEvent* handler) {
  TTV_ASSERT(handler != nullptr);

  std::vector<IChatReceiveNetworkEvent*>::iterator i = std::find(mEventHandlers.begin(), mEventHandlers.end(), handler);
  if (i != mEventHandlers.end()) {
    (void)mEventHandlers.erase(i);
  }
}

const std::string& ttv::chat::ChatSession::GetNick() const {
  return mNick;
}

void ttv::chat::ChatSession::Cap(const std::string& subCommand, const std::string& capabilities) {
  ChatNetworkEvent evt(IRC_CMD_CAP, 2, &subCommand, &capabilities);
  evt.SetAutoPrefix(false);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Pass(const std::string& pass) {
  ChatNetworkEvent evt(IRC_CMD_PASS, 1, &pass);
  evt.SetAutoPrefix(false);
  DoEvent(evt);
}

void ttv::chat::ChatSession::User(const std::string& nick, const std::string& name) {
  SetNick(nick);
  ChatNetworkEvent evt(IRC_CMD_USER, 2, &nick, &name);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Nick(const std::string& nick) {
  ChatNetworkEvent evt(IRC_CMD_NICK, 1, &nick);
  evt.SetAutoPrefix(false);
  DoEvent(evt);
}

void ttv::chat::ChatSession::TwitchClient() {
  ChatNetworkEvent evt(IRC_CMD_TWITCHCLIENT, 0);
  evt.SetAutoPrefix(false);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Join(const std::string& channel, const std::string& key) {
  ChatNetworkEvent evt(IRC_CMD_JOIN, 1, &channel);
  evt.SetAutoPrefix(false);

  if (key.size()) {
    evt.AddParam(key);
  }

  DoEvent(evt);
}

void ttv::chat::ChatSession::Part(const std::string& channel, const std::string& msg) {
  ChatNetworkEvent evt(IRC_CMD_PART, 1, &channel);
  if (msg.size()) {
    evt.AddParam(msg);
  }
  DoEvent(evt);
}

void ttv::chat::ChatSession::Kick(const std::string& channel, const std::string& user, const std::string& reason) {
  ChatNetworkEvent evt(IRC_CMD_KICK, 2, &channel, &user);

  if (reason.size()) {
    evt.AddParam(reason);
  }

  DoEvent(evt);
}

void ttv::chat::ChatSession::Invite(const std::string& user, const std::string& channel) {
  ChatNetworkEvent evt(IRC_CMD_INVITE, 2, &user, &channel);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Topic(const std::string& channel, const std::string& topic) {
  ChatNetworkEvent evt(IRC_CMD_TOPIC, 1, &channel);

  if (topic.size()) {
    evt.AddParam(topic);
  }

  DoEvent(evt);
}

void ttv::chat::ChatSession::PrivMsg(const std::string& target, const std::string& msg) {
  ChatNetworkEvent evt(IRC_CMD_PRIVMSG, 2, &target, &msg);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Action(const std::string& target, const std::string& msg) {
  CTCP(target, "ACTION", msg);
}

void ttv::chat::ChatSession::Notice(const std::string& target, const std::string& msg) {
  ChatNetworkEvent evt(IRC_CMD_NOTICE, 2, &target, &msg);
  DoEvent(evt);
}

void ttv::chat::ChatSession::CTCP(const std::string& target, const std::string& cmd, const std::string& msg) {
  int idEvent = ChatNetworkEvent::CTCPEventStringToID(cmd);
  ChatNetworkEvent evt(idEvent, 1, &target);
  evt.SetEvent(cmd);

  if (msg.size()) {
    evt.AddParam(msg);
  }

  DoEvent(evt);
}

void ttv::chat::ChatSession::CTCPPing(const std::string& target) {
  ChatNetworkEvent evt(IRC_CTCP_PING, 1, &target);
  evt.SetEvent(ChatNetworkEvent::EventIDToString(IRC_CTCP_PING));

  const int MAGIC_NUMBER = 32;
  char buf[MAGIC_NUMBER];
  uint32_t t = static_cast<uint32_t>(SystemTimeToMs(GetSystemClockTime() / 1000));
  (void)snprintf(buf, sizeof(buf), "%u", t);
  buf[MAGIC_NUMBER - 1] = 0;

  evt.AddParam(buf);

  DoEvent(evt);
}

void ttv::chat::ChatSession::CTCPReply(const std::string& target, const std::string& cmd, const std::string& msg) {
  int idEvent = ChatNetworkEvent::CTCPEventStringToID(cmd, true);
  if (idEvent == IRC_CTCP_RPL_INVALID) {
    //_TRACE("Session(0x%08X)::CTCPReply(\"%s\", \"%s\") INVALID COMMAND", this, cmd.c_str(), msg.c_str());
  } else {
    ChatNetworkEvent evt(idEvent, 1, &target);

    if (msg.size()) {
      evt.AddParam(msg);
    }

    DoEvent(evt);
  }
}

void ttv::chat::ChatSession::Whois(const std::string& target) {
  ChatNetworkEvent evt(IRC_CMD_WHOIS, 1, &target);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Mode(const std::string& target, const std::string& modes) {
  ChatNetworkEvent evt(IRC_CMD_MODE, 1, &target);

  std::vector<std::string> list;
  Split(modes, list, ' ', false);

  for (std::vector<std::string>::iterator i = list.begin(); i != list.end(); ++i) {
    evt.AddParam(*i);
  }

  DoEvent(evt);
}

void ttv::chat::ChatSession::UserHost(const std::string& user) {
  ChatNetworkEvent evt(IRC_CMD_USERHOST, 1, &user);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Quit(const std::string& msg) {
  ChatNetworkEvent evt(IRC_CMD_QUIT, 1, &msg);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Away(const std::string& msg) {
  ChatNetworkEvent evt(IRC_CMD_AWAY, 1, &msg);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Ping(const std::string& target) {
  ChatNetworkEvent evt(IRC_CMD_PING, 1, &target);
  evt.SetAutoPrefix(false);
  DoEvent(evt);
}

void ttv::chat::ChatSession::Pong(const std::string& target) {
  ChatNetworkEvent evt(IRC_CMD_PONG, 1, &target);
  evt.SetAutoPrefix(false);
  DoEvent(evt);
}

void ttv::chat::ChatSession::List() {
  ChatNetworkEvent evt(IRC_CMD_LIST, 0);
  DoEvent(evt);
}

uint64_t ttv::chat::ChatSession::GetIdleTime() {
  return GetSystemClockTime() - mLastActivity;
}

void ttv::chat::ChatSession::ResetIdleCounter() {
  mLastActivity = GetSystemClockTime();
}

void ttv::chat::ChatSession::Raw(const std::string& text) {
  ResetIdleCounter();

  TTV_ASSERT(mWriter);
  mWriter->WriteRaw(text);
}

/////////////////////////////////////////////////////////////////////////////
// Internal Methods
/////////////////////////////////////////////////////////////////////////////

bool ttv::chat::ChatSession::IsMe(const std::string& nick) {
  return Compare(mNick, nick, false);
}

// DispatchEvent - Send an evt to the display handlers, or whatever.
void ttv::chat::ChatSession::DispatchEvent(const ChatNetworkEvent& networkEvent) {
  //_TRACE("Session(0x%08X)::DispatchEvent(\"%s\")", this, networkEvent.GetEvent().c_str());

  for (size_t i = 0; i < mEventHandlers.size(); ++i) {
    IChatReceiveNetworkEvent* pHandler = mEventHandlers[i];
    TTV_ASSERT(pHandler != nullptr);

    pHandler->ReceiveEvent(networkEvent);
  }
}

// DoEvent - Send an outgoing evt
void ttv::chat::ChatSession::DoEvent(const ChatNetworkEvent& networkEvent) {
  //_TRACE("Session(0x%08X)::DoEvent(\"%s\")", this, networkEvent.GetEvent().c_str());

  ResetIdleCounter();

  TTV_ASSERT(mWriter);
  mWriter->WriteEvent(networkEvent);
  DispatchEvent(networkEvent);
}

void ttv::chat::ChatSession::InitDefaultServerCapabilities() {
  m_mapServerCaps.clear();

  m_mapServerCaps["CHANTYPES"] = "@&";
  m_mapServerCaps["PREFIX"] = "(ohv)@%+";
}

bool ttv::chat::ChatSession::HasServerCapability(const std::string& cap) {
  return m_mapServerCaps.find(cap) != m_mapServerCaps.end();
}

std::string ttv::chat::ChatSession::GetServerCapability(const std::string& cap) {
  if (HasServerCapability(cap)) {
    return m_mapServerCaps[cap];
  }
  return "";
}

std::string ttv::chat::ChatSession::GetChannelTypes() {
  std::string s = GetServerCapability("CHANTYPES");
  TTV_ASSERT(s.size());
  return s;
}

void ttv::chat::ChatSession::GetChannelUserModes(std::string* letters, std::string* symbols) {
  std::string s = GetServerCapability("PREFIX");
  TTV_ASSERT(s.size());

  // this code works on UTF8 because we're not looking for a wide character
  std::string::size_type start = s.find_first_of('(');
  TTV_ASSERT(start != std::string::npos);
  if (start != std::string::npos) {
    ++start;

    std::string::size_type end = s.find_first_of(')', start);
    TTV_ASSERT(end != std::string::npos);
    if (end != std::string::npos) {
      if (letters) {
        *letters = s.substr(start, end - start);
      }
      if (symbols) {
        *symbols = s.substr(end + 1, end - start);
      }
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
// IChatReceiveNetworkEvent
/////////////////////////////////////////////////////////////////////////////

void ttv::chat::ChatSession::ReceiveEvent(const ChatNetworkEvent& evt) {
#define HANDLE_IRC_EVENT(e, h) \
  case e:                      \
    h(evt);                    \
    break;

  ResetIdleCounter();

  // Phase 1.  Events that should be handled before being displayed
  switch (evt.GetEventID()) {
    HANDLE_IRC_EVENT(IRC_CMD_NICK, OnNick)
    HANDLE_IRC_EVENT(IRC_CMD_JOIN, OnJoin)
    HANDLE_IRC_EVENT(IRC_CMD_PART, OnPart)
    HANDLE_IRC_EVENT(IRC_CMD_KICK, OnKick)
    HANDLE_IRC_EVENT(IRC_CMD_MODE, OnMode)
    HANDLE_IRC_EVENT(IRC_CMD_QUIT, OnQuit)

    HANDLE_IRC_EVENT(IRC_RPL_WELCOME, OnRplWelcome)
    HANDLE_IRC_EVENT(IRC_RPL_PROTOCTL, OnRplProtoCtl)
    HANDLE_IRC_EVENT(IRC_RPL_NAMREPLY, OnRplNamReply)
    HANDLE_IRC_EVENT(IRC_RPL_LISTSTART, OnRplListStart)
    HANDLE_IRC_EVENT(IRC_RPL_LIST, OnRplList)

    HANDLE_IRC_EVENT(IRC_CTCP_RPL_PING, OnCTCPRplPing)
  }

  // Phase 2. Dispatch Events to external handlers
  DispatchEvent(evt);

  // Phase 3. Events that should be handled after being handled.... uh....
  switch (evt.GetEventID()) {
    HANDLE_IRC_EVENT(IRC_CMD_PING, OnPing)

    HANDLE_IRC_EVENT(IRC_CTCP_PING, OnCTCPPing)
    HANDLE_IRC_EVENT(IRC_CTCP_VERSION, OnCTCPVersion)

    HANDLE_IRC_EVENT(SYS_EVENT_CONNECTED, OnConnect)
    HANDLE_IRC_EVENT(SYS_EVENT_CLOSE, OnDisconnect)
  }

#undef HANDLE_IRC_EVENT
}

/////////////////////////////////////////////////////////////////////////////
// Session state handlers
/////////////////////////////////////////////////////////////////////////////

void ttv::chat::ChatSession::OnConnect(const ChatNetworkEvent& evt) {
  (void)evt;

  InitDefaultServerCapabilities();
  mConnected = true;
}

void ttv::chat::ChatSession::OnDisconnect(const ChatNetworkEvent& evt) {
  (void)evt;

  mConnected = false;
}

void ttv::chat::ChatSession::OnJoin(const ChatNetworkEvent& /*evt*/) {
  // NOTE: Twitch chat doesn't join channels the way that other IRC servers do
}

void ttv::chat::ChatSession::OnPart(const ChatNetworkEvent& /*evt*/) {
  // NOTE: Twitch chat doesn't join channels the way that other IRC servers do
}

void ttv::chat::ChatSession::OnKick(const ChatNetworkEvent& evt) {
  // const std::string& channel = evt.GetParam(0);
  const std::string& user = evt.GetParam(1);

  if (IsMe(user)) {
    // TODO: handle the leave of the channel
  } else {
    // TODO: remove the user from the list
  }
}

void ttv::chat::ChatSession::OnNick(const ChatNetworkEvent& evt) {
  (void)evt;

  // TODO: do we want to allow this?

  // std::string user = GetPrefixNick(evt.GetPrefix());
  // const std::string& nick = evt.GetParam(0);

  // if (IsMe(user))
  //{
  //  SetNick(nick);
  //}

  //// Update channel nick lists
  // if (m_vecChannels.size() > 0)
  //{
  //  //Change user's nick in channel nick lists
  //  for(UINT iChan = 0; iChan < m_vecChannels.size(); ++iChan)
  //  {
  //      Channel* pChan = m_vecChannels[iChan];
  //      TTV_ASSERT(pChan != nullptr);

  //      if(pChan != nullptr)
  //      {
  //          if(pChan->IsOn(user))
  //          {
  //              pChan->ChangeName(user, nick);
  //          }
  //      }
  //  }
  //}
}

void ttv::chat::ChatSession::OnQuit(const ChatNetworkEvent& evt) {
  (void)evt;

  // TODO: do we want to allow quit?

  // std::string user = GetPrefixNick(evt.GetPrefix());

  // if (IsMe(user))
  //{
  //  // Don't think we ever get QUIT for ourself
  //}
  // else
  //{
  //  for (UINT iChan = 0; iChan < m_vecChannels.size(); ++iChan)
  //  {
  //      Channel* pChan = m_vecChannels[iChan];
  //      TTV_ASSERT(pChan != nullptr);

  //      if (pChan != nullptr)
  //      {
  //          if (pChan->IsOn(user))
  //          {
  //              pChan->RemoveName(user);
  //          }
  //      }
  //  }
  //}
}

void ttv::chat::ChatSession::OnPing(const ChatNetworkEvent& evt) {
  ChatNetworkEvent out(IRC_CMD_PONG, 1, &evt.GetParam(0));

  DoEvent(out);
}

void ttv::chat::ChatSession::OnCTCPPing(const ChatNetworkEvent& evt) {
  std::string from = GetPrefixNick(evt.GetPrefix());

  CTCPReply(from, "PING", evt.GetParam(1));
}

void ttv::chat::ChatSession::OnCTCPRplPing(const ChatNetworkEvent& evt) {
  std::string from = GetPrefixNick(evt.GetPrefix());
  std::string sTime = evt.GetParam(1);

  if (sTime.size()) {
    uint32_t now = static_cast<uint32_t>(SystemTimeToMs(GetSystemClockTime())) / 1000;
    uint32_t then = 0;
    ParseNum(sTime, then);  // DWORD then = _tcstoul(sTime.c_str(), nullptr, 10);
    uint32_t seconds = now - then;

    char buf[30];
    (void)snprintf(buf, sizeof(buf), "%u", seconds);

    sTime = buf;

    const_cast<ChatNetworkEvent&>(evt).AddParam(buf);  // zomg bad
  }
}

void ttv::chat::ChatSession::OnCTCPVersion(const ChatNetworkEvent& evt) {
  std::string from = GetPrefixNick(evt.GetPrefix());

  CTCPReply(from, "VERSION", "TwitchIRC");
}

void ttv::chat::ChatSession::OnRplNamReply(const ChatNetworkEvent& evt) {
  (void)evt;
}

void ttv::chat::ChatSession::OnMode(const ChatNetworkEvent& evt) {
  (void)evt;
}

void ttv::chat::ChatSession::OnRplWelcome(const ChatNetworkEvent& evt) {
  const std::string& sMe = evt.GetParam(0);
  TTV_ASSERT(sMe.size());
  if (sMe.size()) {
    SetNick(sMe);
  }
}

void ttv::chat::ChatSession::OnRplProtoCtl(const ChatNetworkEvent& evt) {
  for (unsigned int i = 1; i < evt.GetParamCount() - 1; ++i) {
    const std::string& sParam = evt.GetParam(i);

    std::vector<std::string> var;
    Split(sParam, var, '=', false);

    TTV_ASSERT(var.size() > 0 && var.size() < 3);

    if (var.size() > 0) {
      if (var.size() < 2) {
        m_mapServerCaps[var[0]] = "";
        //_TRACE("Server Capability: %s", var[0].c_str());
      } else {
        m_mapServerCaps[var[0]] = var[1];
        //_TRACE("Server Capability: %s: %s", var[0].c_str(), m_mapServerCaps[var[0]].c_str());
      }
    }
  }
}

void ttv::chat::ChatSession::OnRplListStart(const ChatNetworkEvent& evt) {
  (void)evt;

  // ClearChannelList();

  // TODO: do we want to support the channel list?
}

void ttv::chat::ChatSession::OnRplList(const ChatNetworkEvent& evt) {
  (void)evt;

  // TODO: do we want to support the channel list?

  // std::string channel = evt.GetParam(1);
  // std::string users = evt.GetParam(2);
  // std::string topic = evt.GetParam(3);

  // if(topic[0] == '[' && topic[topic.size() - 1] == ']')
  //{
  //  topic = "";
  //}
  // topic = StringFormat::StripFormatting(topic);

  // if(channel != "*")
  //{
  //  ChannelListEntry* chan = new ChannelListEntry();
  //  chan->name = channel;
  //  chan->users = users;
  //  chan->topic = topic;

  //  m_vecChannelList.push_back(chan);
  //}
}
