/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatsockettransport.h"

#include "twitchsdk/core/httprequestutils.h"

#include <stdio.h>

ttv::chat::IChatTransportReader::~IChatTransportReader() {}

ttv::chat::IChatTransport::~IChatTransport() {}

TTV_ErrorCode ttv::chat::ChatSocketTransport::CheckFactoryAvailability(const std::string& uri) {
  Uri url(uri);
  std::string protocol = url.GetProtocol();

  if (protocol == "ws" || protocol == "wss") {
    return ttv::IsWebSocketProtocolSupported(protocol);
  } else {
    return ttv::IsSocketProtocolSupported(protocol);
  }
}

ttv::chat::ChatSocketTransport::ChatSocketTransport() : mReader(nullptr) {
  mBuffer.resize(kMaxBufferSize);
}

ttv::chat::ChatSocketTransport::~ChatSocketTransport() {}

TTV_ErrorCode ttv::chat::ChatSocketTransport::Connect(const std::string& uri) {
  Uri url(uri);

  ttv::trace::Message("Chat", MessageLevel::Debug, "ChatSocketTransport::Connect %s", uri.c_str());

  TTV_ErrorCode ec = Close();
  TTV_ASSERT(TTV_SUCCEEDED(ec));

  // Create a web socket
  if (url.GetProtocol() == "ws" || url.GetProtocol() == "wss") {
    std::shared_ptr<IWebSocket> socket;
    ec = ttv::CreateWebSocket(uri, socket);

    if (TTV_SUCCEEDED(ec)) {
      ec = socket->Connect();
    }

    if (TTV_SUCCEEDED(ec)) {
      mWebSocket = std::move(socket);
    }
  }
  // Create a raw socket
  else {
    std::shared_ptr<ISocket> socket;
    ec = ttv::CreateSocket(uri, socket);

    if (TTV_SUCCEEDED(ec)) {
      ec = socket->Connect();
    }

    if (TTV_SUCCEEDED(ec)) {
      mSocket = std::move(socket);
    }
  }

  if (TTV_SUCCEEDED(ec)) {
    ttv::trace::Message("Chat", MessageLevel::Debug, "ChatSocketTransport::Connect successful");
  } else {
    ttv::trace::Message("Chat", MessageLevel::Error, "ChatSocketTransport::Connect failed: ec = %s", ErrorToString(ec));
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatSocketTransport::ProcessIncomingEvent(bool& handled) {
  handled = false;

  if (!IsOpen()) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  // Grab some data from the socket
  size_t received = 0;

  if (mSocket != nullptr) {
    ec = mSocket->Recv(reinterpret_cast<uint8_t*>(mBuffer.data()), mBuffer.size(), received);
  } else {
    TTV_ASSERT(mWebSocket != nullptr);

    IWebSocket::MessageType type = IWebSocket::MessageType::None;
    size_t messageSize = 0;
    ec = mWebSocket->Peek(type, messageSize);

    if (SOCKET_SUCCEEDED(ec) && type == IWebSocket::MessageType::None) {
      ec = TTV_EC_SOCKET_EWOULDBLOCK;
    }

    if (TTV_SUCCEEDED(ec)) {
      TTV_ASSERT(type == IWebSocket::MessageType::Text);

      if (mBuffer.size() < messageSize) {
        mBuffer.resize(messageSize);
      }

      ec = mWebSocket->Recv(type, reinterpret_cast<uint8_t*>(&mBuffer[0]), messageSize, received);
      TTV_ASSERT(TTV_FAILED(ec) || (TTV_SUCCEEDED(ec) && received == messageSize));
    }
  }

  // No data available
  if (ec == TTV_EC_SOCKET_EWOULDBLOCK) {
    return TTV_EC_SUCCESS;
  }
  // Something went wrong
  else if (TTV_FAILED(ec)) {
    Close();
    return ec;
  }

  handled = true;

  TTV_ASSERT(mReader);
  if (mReader) {
    // Allow the reader to parse and generate the event structure
    auto reader = mReader;
    reader->OnRead(mBuffer.data(), received);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatSocketTransport::Write(const char* pData, size_t nSize) {
  TTV_ASSERT(pData);

  TTV_ErrorCode ec;

  if (mSocket != nullptr) {
    ec = mSocket->Send(reinterpret_cast<const uint8_t*>(pData), nSize);
  } else if (mWebSocket != nullptr) {
    ec = mWebSocket->Send(IWebSocket::MessageType::Text, reinterpret_cast<const uint8_t*>(pData), nSize);
  } else {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatSocketTransport::Close() {
  if (mSocket == nullptr && mWebSocket == nullptr) {
    return TTV_EC_SUCCESS;
  }

  if (mSocket != nullptr) {
    mSocket->Disconnect();
    mSocket.reset();
  }

  if (mWebSocket != nullptr) {
    mWebSocket->Disconnect();
    mWebSocket.reset();
  }

  if (mReader != nullptr) {
    // Keep the reader alive until the scope exits
    auto reader = mReader;
    reader->OnClose();
  }

  return TTV_EC_SUCCESS;
}

bool ttv::chat::ChatSocketTransport::IsOpen() {
  if (mSocket != nullptr && mSocket->Connected()) {
    return true;
  }

  if (mWebSocket != nullptr && mWebSocket->Connected()) {
    return true;
  }

  return false;
}

void ttv::chat::ChatSocketTransport::SetReader(std::shared_ptr<IChatTransportReader> reader) {
  mReader = reader;
}
